from pathlib import Path
from typing import Literal, Optional, List, Tuple
import logging

from selenium import webdriver
from selenium.webdriver.chrome.service import Service
from selenium.webdriver.chrome.options import Options
from webdriver_manager.chrome import ChromeDriverManager

from selenium.webdriver.remote.webdriver import WebDriver
from selenium.webdriver.remote.webelement import WebElement
from selenium.webdriver.support.ui import WebDriverWait
from selenium.webdriver.support import expected_conditions as EC
from selenium.webdriver.common.by import By
from selenium.webdriver.common.action_chains import ActionChains
from selenium.webdriver.common.keys import Keys
from selenium.common.exceptions import (
    TimeoutException,
    NoSuchElementException,
    MoveTargetOutOfBoundsException,
    InvalidSelectorException,
    SessionNotCreatedException,
    WebDriverException
)


class WebDriverManipulator:
    def __init__(self, driver: WebDriver, logger: logging,default_timeout: int = 60):
        if not isinstance(driver, WebDriver):
            raise TypeError("A instância do driver deve ser um objeto WebDriver.")
        self.driver = driver
        self.default_timeout = default_timeout
        self.action_chains = ActionChains(self.driver)
        self._logger = logger

    def _get_by_strategy(self, selector_type: str) -> By:
        """
        Mapeia strings de tipo de seletor para constantes By do Selenium.

        Args:
            selector_type (str): O tipo de seletor (ex: 'id', 'name', 'xpath', 'css_selector').

        Returns:
            By: A estratégia de localização By correspondente.

        Raises:
            ValueError: Se o tipo de seletor não for suportado.
        """
        strategy_map = {
            'id': By.ID,
            'name': By.NAME,
            'class': By.CLASS_NAME,
            'xpath': By.XPATH,
            'tag_name': By.TAG_NAME,
            'css_selector': By.CSS_SELECTOR,
            'link_text': By.LINK_TEXT,
            'partial_link_text': By.PARTIAL_LINK_TEXT,
        }
        normalized_type = selector_type.lower()
        if normalized_type not in strategy_map:
            self._logger.error(f"Tipo de seletor '{selector_type}' não suportado.")
            raise ValueError(f"Seletor '{selector_type}' não suportado. Escolha entre {list(strategy_map.keys())}.")
        return strategy_map[normalized_type]

    def find_element(
        self,
        selector_value: str,
        selector_type: str = 'xpath',
        timeout: Optional[int] = None,
        raise_exception: bool = True
    ) -> Optional[WebElement]:
        """
        Encontra um único elemento na página visível (fora de frames).

        Args:
            selector_value (str): O valor do seletor (ex: ID, XPath).
            selector_type (str): O tipo de seletor (ex: 'id', 'xpath').
            timeout (Optional[int]): Tempo limite para a espera do elemento. Se None, usa o default_timeout.
            raise_exception (bool): Se deve levantar uma exceção (NoSuchElementException/TimeoutException)
                                    se o elemento não for encontrado.

        Returns:
            Optional[WebElement]: O WebElement encontrado ou None se não encontrado e raise_exception=False.

        Raises:
            NoSuchElementException: Se o elemento não for encontrado e raise_exception=True.
            TimeoutException: Se o elemento não for encontrado dentro do timeout e raise_exception=True.
        """
        current_timeout = timeout if timeout is not None else self.default_timeout
        by_strategy = self._get_by_strategy(selector_type)
        
        try:
            wait = WebDriverWait(self.driver, current_timeout)
            element = wait.until(EC.presence_of_element_located((by_strategy, selector_value)))
            self._logger.debug(f"Elemento '{selector_value}' encontrado com sucesso por {selector_type}.")
            return element
        except TimeoutException as e:
            if raise_exception:
                self._logger.error(f"Timeout: Elemento '{selector_value}' não encontrado por {selector_type} em {current_timeout}s. Detalhes: {e}")
                raise
            self._logger.warning(f"Elemento '{selector_value}' não encontrado em {current_timeout}s (silenciado).")
            return None
        except NoSuchElementException as e:
            if raise_exception:
                self._logger.error(f"NoSuchElement: Elemento '{selector_value}' não encontrado por {selector_type}. Detalhes: {e}")
                raise
            self._logger.warning(f"Elemento '{selector_value}' não encontrado (silenciado).")
            return None
        except InvalidSelectorException as e:
            self._logger.error(f"Seletor inválido: '{selector_value}' ({selector_type}). Detalhes: {e}")
            if raise_exception:
                raise
            return None
        except Exception as e:
            self._logger.error(f"Erro inesperado ao buscar elemento '{selector_value}' por {selector_type}. Detalhes: {e}")
            if raise_exception:
                raise
            return None

    def find_elements(
        self,
        selector_value: str,
        selector_type: str = 'xpath',
        timeout: Optional[int] = None,
        min_elements: int = 0
    ) -> List[WebElement]:
        """
        Encontra múltiplos elementos na página visível (fora de frames).

        Args:
            selector_value (str): O valor do seletor.
            selector_type (str): O tipo de seletor.
            timeout (Optional[int]): Tempo limite para a espera. Se None, usa o default_timeout.
            min_elements (int): Número mínimo de elementos esperados.

        Returns:
            List[WebElement]: Uma lista de WebElements encontrados.

        Raises:
            TimeoutException: Se o número mínimo de elementos não for encontrado dentro do timeout.
        """
        current_timeout = timeout if timeout is not None else self.default_timeout
        by_strategy = self._get_by_strategy(selector_type)

        try:
            wait = WebDriverWait(self.driver, current_timeout)
            if min_elements > 0:
                elements = wait.until(EC.number_of_elements_more_than((by_strategy, selector_value), min_elements -1))
            else:
                elements = wait.until(EC.presence_of_all_elements_located((by_strategy, selector_value)))
            
            if not elements:
                self._logger.warning(f"Nenhum elemento '{selector_value}' encontrado por {selector_type}.")
            else:
                self._logger.debug(f"Encontrados {len(elements)} elementos para '{selector_value}' por {selector_type}.")
            return elements
        except TimeoutException as e:
            self._logger.error(f"Timeout: Não foi possível encontrar {min_elements} ou mais elementos para '{selector_value}' por {selector_type} em {current_timeout}s. Detalhes: {e}")
            raise
        except InvalidSelectorException as e:
            self._logger.error(f"Seletor inválido ao buscar múltiplos elementos: '{selector_value}' ({selector_type}). Detalhes: {e}")
            raise
        except Exception as e:
            self._logger.error(f"Erro inesperado ao buscar múltiplos elementos '{selector_value}' por {selector_type}. Detalhes: {e}")
            raise

    def _switch_to_frame(self, frame_element: Optional[WebElement]):
        """Alterna o driver para o frame especificado ou para o conteúdo padrão."""
        try:
            if frame_element is None:
                self.driver.switch_to.default_content()
                self._logger.debug("Alternado para o conteúdo padrão.")
            else:
                self.driver.switch_to.frame(frame_element)
                self._logger.debug(f"Alternado para o frame: {frame_element.tag_name} [id: {frame_element.get_attribute('id')}]")
        except Exception as e:
            self._logger.error(f"Falha ao alternar para o frame: {e}")
            raise

    def find_element_in_frames(
        self,
        selector_value: str,
        selector_type: str = 'xpath',
        timeout: Optional[int] = None,
        raise_exception: bool = True
    ) -> Optional[WebElement]:
        """
        Busca um único elemento recursivamente em todos os frames da página.

        Args:
            selector_value (str): O valor do seletor do elemento.
            selector_type (str): O tipo de seletor.
            timeout (Optional[int]): Tempo limite total para a busca.
            raise_exception (bool): Se deve levantar uma exceção se o elemento não for encontrado.

        Returns:
            Optional[WebElement]: O WebElement encontrado ou None.

        Raises:
            NoSuchElementException: Se o elemento não for encontrado e raise_exception=True.
            TimeoutException: Se o elemento não for encontrado dentro do timeout e raise_exception=True.
        """
        current_timeout = timeout if timeout is not None else self.default_timeout
        start_time = self.driver.execute_script("return performance.now()") # Tempo de início da busca total
        
        # Pilha para DFS (Depth-First Search) para explorar frames.
        # Armazena tuplas: (frame_element, current_depth)
        frames_stack: List[Tuple[Optional[WebElement], int]] = [(None, 0)] 
        
        found_element: Optional[WebElement] = None
        
        # Usado para evitar loops infinitos em caso de frames cíclicos, embora raro.
        processed_frame_ids = set() 

        self._logger.info(f"Iniciando busca recursiva por '{selector_value}' em frames. Timeout: {current_timeout}s.")

        while frames_stack:
            if (self.driver.execute_script("return performance.now()") - start_time) / 1000 > current_timeout:
                self._logger.warning("Tempo limite de busca em frames atingido.")
                break

            current_frame_element, depth = frames_stack.pop() # Pop do último elemento (DFS)
            
            # Evita processar o mesmo frame várias vezes
            if current_frame_element:
                frame_id = id(current_frame_element)
                if frame_id in processed_frame_ids:
                    continue
                processed_frame_ids.add(frame_id)

            try:
                self._switch_to_frame(current_frame_element)
                
                # Tenta encontrar o elemento no frame atual (timeout curto para não bloquear)
                element = self.find_element(
                    selector_value, selector_type=selector_type, 
                    timeout=1, # Timeout curto para busca individual em cada frame
                    raise_exception=False 
                )
                if element:
                    found_element = element
                    self._logger.info(f"Elemento '{selector_value}' encontrado no frame em profundidade {depth}.")
                    break # Elemento encontrado, sai do loop

                # Se não encontrou, busca frames aninhados no contexto atual
                nested_frames = self.driver.find_elements(
                    'iframe', selector_type='tag_name', timeout=0.5, min_elements=0
                ) + self.driver.find_elements(
                    'frame', selector_type='tag_name', timeout=0.5, min_elements=0
                )

                # Adiciona frames aninhados à pilha
                for nested_frame in nested_frames:
                    if id(nested_frame) not in processed_frame_ids:
                        frames_stack.append((nested_frame, depth + 1))
                        self._logger.debug(f"Adicionado frame aninhado à pilha (profundidade {depth + 1}).")

            except Exception as e:
                self._logger.debug(f"Erro ao explorar frame em profundidade {depth}. Detalhes: {e}")
            finally:
                # Sempre volta para o frame pai depois de explorar um ramo
                if current_frame_element is not None:
                    self.driver.switch_to.parent_frame()
                    self._logger.debug(f"Voltado para o frame pai de profundidade {depth}.")

        # Garante que o driver esteja no conteúdo padrão ao final da busca
        self.driver.switch_to.default_content()

        if found_element is None and raise_exception:
            self._logger.error(f"Elemento '{selector_value}' não encontrado em nenhum frame após busca recursiva.")
            raise NoSuchElementException(f"Elemento '{selector_value}' não encontrado em nenhum frame.")

        return found_element

    def click_element(self, element: WebElement, safe_click: bool = True):
        """
        Clica em um WebElement.

        Args:
            element (WebElement): O elemento para clicar.
            safe_click (bool): Se deve tentar clicar mesmo se o elemento estiver fora da visibilidade, usando ActionChains como fallback.
        """
        try:
            element.click()
            self._logger.debug("Elemento clicado com sucesso.")
        except MoveTargetOutOfBoundsException as e:
            if safe_click:
                self._logger.warning(f"Elemento fora da visibilidade, tentando clique com ActionChains. Detalhes: {e}")
                self.action_chains.move_to_element(element).click().perform()
            else:
                self._logger.error(f"Elemento fora da visibilidade e safe_click=False. Detalhes: {e}")
                raise
        except Exception as e:
            self._logger.error(f"Erro ao clicar no elemento: {e}")
            raise

    def type_text(self, element: WebElement, text: str, clear_first: bool = False):
        """
        Digita um texto em um campo de entrada.

        Args:
            element (WebElement): O campo de entrada.
            text (str): O texto a ser digitado.
            clear_first (bool): Se deve limpar o campo antes de digitar.
        """
        try:
            if clear_first:
                element.clear()
                self._logger.debug("Campo limpo antes de digitar.")
            element.send_keys(text)
            self._logger.debug(f"Texto '{text}' digitado no elemento.")
        except Exception as e:
            self._logger.error(f"Erro ao digitar texto no elemento: {e}")
            raise

    def get_element_text(self, element: WebElement) -> str:
        """
        Retorna o texto visível de um elemento.

        Args:
            element (WebElement): O elemento.

        Returns:
            str: O texto do elemento.
        """
        try:
            text = element.text
            self._logger.debug(f"Texto do elemento obtido: '{text}'.")
            return text
        except Exception as e:
            self._logger.error(f"Erro ao obter texto do elemento: {e}")
            raise

    def get_element_attribute(self, element: WebElement, attribute_name: str) -> str:
        """
        Retorna o valor de um atributo de um elemento.

        Args:
            element (WebElement): O elemento.
            attribute_name (str): O nome do atributo.

        Returns:
            str: O valor do atributo.
        """
        try:
            attribute_value = element.get_attribute(attribute_name)
            self._logger.debug(f"Atributo '{attribute_name}' do elemento obtido: '{attribute_value}'.")
            return attribute_value
        except Exception as e:
            self._logger.error(f"Erro ao obter atributo '{attribute_name}' do elemento: {e}")
            raise

    def wait_for_element_visibility(
        self,
        selector_value: str,
        selector_type: str = 'xpath',
        timeout: Optional[int] = None
    ) -> WebElement:
        """
        Espera até que um elemento esteja visível na página.

        Args:
            selector_value (str): O valor do seletor.
            selector_type (str): O tipo de seletor.
            timeout (Optional[int]): Tempo limite para a espera.

        Returns:
            WebElement: O elemento visível.

        Raises:
            TimeoutException: Se o elemento não estiver visível dentro do tempo limite.
        """
        current_timeout = timeout if timeout is not None else self.default_timeout
        by_strategy = self._get_by_strategy(selector_type)
        try:
            wait = WebDriverWait(self.driver, current_timeout)
            element = wait.until(EC.visibility_of_element_located((by_strategy, selector_value)))
            self._logger.debug(f"Elemento '{selector_value}' visível após espera.")
            return element
        except TimeoutException as e:
            self._logger.error(f"Timeout: Elemento '{selector_value}' não ficou visível em {current_timeout}s. Detalhes: {e}")
            raise
        except Exception as e:
            self._logger.error(f"Erro ao esperar pela visibilidade do elemento '{selector_value}': {e}")
            raise

    def wait_for_element_clickable(
        self,
        selector_value: str,
        selector_type: str = 'xpath',
        timeout: Optional[int] = None
    ) -> WebElement:
        """
        Espera até que um elemento esteja clicável na página.

        Args:
            selector_value (str): O valor do seletor.
            selector_type (str): O tipo de seletor.
            timeout (Optional[int]): Tempo limite para a espera.

        Returns:
            WebElement: O elemento clicável.

        Raises:
            TimeoutException: Se o elemento não estiver clicável dentro do tempo limite.
        """
        current_timeout = timeout if timeout is not None else self.default_timeout
        by_strategy = self._get_by_strategy(selector_type)
        try:
            wait = WebDriverWait(self.driver, current_timeout)
            element = wait.until(EC.element_to_be_clickable((by_strategy, selector_value)))
            self._logger.debug(f"Elemento '{selector_value}' clicável após espera.")
            return element
        except TimeoutException as e:
            self._logger.error(f"Timeout: Elemento '{selector_value}' não ficou clicável em {current_timeout}s. Detalhes: {e}")
            raise
        except Exception as e:
            self._logger.error(f"Erro ao esperar pelo clique do elemento '{selector_value}': {e}")
            raise

    def execute_script(self, script: str, *args):
        """
        Executa um script JavaScript no contexto do driver.

        Args:
            script (str): O script JavaScript a ser executado.
            *args: Argumentos a serem passados para o script.
        """
        try:
            self._logger.debug(f"Executando script JS: {script[:50]}...")
            return self.driver.execute_script(script, *args)
        except Exception as e:
            self._logger.error(f"Erro ao executar script JavaScript: {e}")
            raise
            
    def switch_to_default_content(self):
        """
        Alterna o foco do driver de volta para o conteúdo principal da página.
        """
        try:
            self.driver.switch_to.default_content()
            self._logger.info("Foco do driver restaurado para o conteúdo padrão.")
        except Exception as e:
            self._logger.error(f"Erro ao alternar para o conteúdo padrão: {e}")
            raise

    def go_to_url(self, url: str):
        """Navega o driver para uma URL específica."""
        try:
            self.driver.get(url)
            self._logger.info(f"Navegou para a URL: {url}")
        except Exception as e:
            self._logger.error(f"Erro ao navegar para a URL '{url}': {e}")
            raise

    def get_current_url(self) -> str:
        """Retorna a URL atual do navegador."""
        try:
            url = self.driver.current_url
            self._logger.debug(f"URL atual: {url}")
            return url
        except Exception as e:
            self._logger.error(f"Erro ao obter a URL atual: {e}")
            raise

    def refresh_page(self):
        """Atualiza a página atual."""
        try:
            self.driver.refresh()
            self._logger.info("Página atualizada.")
        except Exception as e:
            self._logger.error(f"Erro ao atualizar a página: {e}")
            raise

    def close_current_tab(self):
        """Fecha a aba atualmente focada."""
        try:
            self.driver.close()
            self._logger.info("Aba atual fechada.")
        except Exception as e:
            self._logger.error(f"Erro ao fechar a aba atual: {e}")
            raise

    def switch_to_tab(self, tab_index: int):
        """
        Muda para a aba do navegador especificada pelo índice.

        Args:
            tab_index (int): O índice da aba para a qual mudar (0 para a primeira).
        """
        try:
            window_handles = self.driver.window_handles
            if 0 <= tab_index < len(window_handles):
                self.driver.switch_to.window(window_handles[tab_index])
                self._logger.info(f"Mudou para a aba com índice: {tab_index}")
            else:
                self._logger.error(f"Índice de aba inválido: {tab_index}. Total de abas: {len(window_handles)}.")
                raise IndexError(f"Aba com índice {tab_index} não existe.")
        except Exception as e:
            self._logger.error(f"Erro ao mudar para a aba {tab_index}: {e}")
            raise

    def take_screenshot(self, file_path: str):
        """
        Tira uma captura de tela da página atual.

        Args:
            file_path (str): O caminho completo do arquivo para salvar a captura de tela (ex: 'screenshot.png').
        """
        try:
            self.driver.save_screenshot(file_path)
            self._logger.info(f"Captura de tela salva em: {file_path}")
        except Exception as e:
            self._logger.error(f"Erro ao tirar captura de tela em '{file_path}': {e}")
            raise


# Exemplo de uso:
if __name__ == "__main__":
    driver = ...

    # Instanciar a classe manipuladora com o driver
    manipulator = WebDriverManipulator(driver)

    # Exemplo de operações:
    manipulator.go_to_url("https://www.google.com")
    
    search_box = manipulator.find_element(
        selector_value='q', 
        selector_type='name', 
        timeout=10
    )
    if search_box:
        manipulator.type_text(search_box, "Python Automation Best Practices", clear_first=True)
        search_box.send_keys(Keys.ENTER)
        
    manipulator.take_screenshot("Google Search_results.png")
    
    # Exemplo de busca em frames (assumindo que há frames na página)
    # try:
    #     element_in_frame = manipulator.find_element_in_frames(
    #         selector_value='element_id_in_frame', 
    #         selector_type='id', 
    #         timeout=15
    #     )
    #     if element_in_frame:
    #         manipulator.click_element(element_in_frame)
    #         logger.info("Elemento dentro do frame clicado.")
    # except NoSuchElementException:
    #     logger.warning("Elemento no frame não encontrado, pulando interação.")
