# ohlcv.py

from abc import ABCMeta
import datetime as dt
from typing import (
    Iterable, Dict, Optional, ClassVar, List, Tuple
)

from attrs import define

from represent import represent, Modifiers

import pandas as pd

from crypto_screening.market.screeners.base import BaseScreener
from crypto_screening.market.screeners.ohlcv import (
    create_ohlcv_dataframe
)
from crypto_screening.collect.market import (
    MarketBase, assets_market_prices, symbols_market_data,
    is_symbol_in_assets_market_prices, symbols_market_prices,
    is_symbol_in_symbols_market_prices, assets_market_data,
    assets_to_symbol_market_prices, assets_market_state,
    symbol_to_assets_market_prices, symbols_market_state,
    merge_assets_market_states, merge_symbols_market_states,
    OHLCV_COLUMNS
)

__all__ = [
    "symbols_ohlcv_market_state",
    "merge_assets_ohlcv_market_states",
    "merge_symbols_ohlcv_market_states",
    "assets_ohlcv_market_state",
    "AssetsOHLCVMarketState",
    "SymbolsOHLCVMarketState",
    "symbols_to_assets_ohlcv_market_state",
    "assets_to_symbols_ohlcv_market_state",
    "OHLCV_COLUMNS"
]

AssetsPrices = Dict[str, Dict[str, Dict[str, List[Tuple[dt.datetime, float]]]]]
SymbolsPrices = Dict[str, Dict[str, List[Tuple[dt.datetime, float]]]]

@define(repr=False)
@represent
class OHLCVMarketBase(MarketBase, metaclass=ABCMeta):
    """
    A class to represent the current market state.

    This object contains the state of the market, as Close,
    bids and asks prices of specific assets, gathered from the network.

    attributes:

    - screeners:
        The screener objects to collect the prices of the assets.
    """

    __modifiers__: ClassVar[Modifiers] = Modifiers(
        **MarketBase.__modifiers__
    )
    __modifiers__.excluded.extend(["open", "high", "low", "close", "volume"])
# end OrderbookMarketBase

AssetsMarketData = Dict[str, Dict[str, Dict[str, List[Tuple[dt.datetime, Dict[str, float]]]]]]
AssetsMarketDatasets = Dict[str, Dict[str, Dict[str, pd.DataFrame]]]

@define(repr=False)
@represent
class AssetsOHLCVMarketState(OHLCVMarketBase):
    """
    A class to represent the current market state.

    This object contains the state of the market, as Close,
    bids and asks prices of specific assets, gathered from the network.

    attributes:

    - screeners:
        The screener objects to collect the prices of the assets.

    - bids:
        The bids prices of the assets.

    - asks:
        The asks prices of the assets.

    - bids_volume:
        The bids volume prices of the assets.

    - asks_volume:
        The asks volume prices of the assets.

    >>> from crypto_screening.collect.orderbook import AssetsOrderbookMarketState
    >>>
    >>> state = assets_ohlcv_market_state(...)
    """

    opens: AssetsPrices
    highs: AssetsPrices
    lows: AssetsPrices
    closes: AssetsPrices
    volumes: AssetsPrices

    def open(
            self, exchange: str, symbol: str, separator: Optional[str] = None
    ) -> List[Tuple[dt.datetime, float]]:
        """
        Returns the bid price for the symbol.

        :param exchange: The exchange name.
        :param symbol: The symbol to find its bid price.
        :param separator: The separator of the assets.

        :return: The bid price for the symbol.
        """

        return assets_market_prices(
            exchange=exchange, symbol=symbol, prices=self.opens,
            separator=separator, provider=self
        )
    # end bid

    def high(
            self, exchange: str, symbol: str, separator: Optional[str] = None
    ) -> List[Tuple[dt.datetime, float]]:
        """
        Returns the ask price for the symbol.

        :param exchange: The exchange name.
        :param symbol: The symbol to find its ask price.
        :param separator: The separator of the assets.

        :return: The ask price for the symbol.
        """

        return assets_market_prices(
            exchange=exchange, symbol=symbol, prices=self.highs,
            separator=separator, provider=self
        )
    # end ask

    def low(
            self, exchange: str, symbol: str, separator: Optional[str] = None
    ) -> List[Tuple[dt.datetime, float]]:
        """
        Returns the bid price for the symbol.

        :param exchange: The exchange name.
        :param symbol: The symbol to find its bid price.
        :param separator: The separator of the assets.

        :return: The bid price for the symbol.
        """

        return assets_market_prices(
            exchange=exchange, symbol=symbol, prices=self.lows,
            separator=separator, provider=self
        )
    # end bid_volume

    def close(
            self, exchange: str, symbol: str, separator: Optional[str] = None
    ) -> List[Tuple[dt.datetime, float]]:
        """
        Returns the ask price for the symbol.

        :param exchange: The exchange name.
        :param symbol: The symbol to find its ask price.
        :param separator: The separator of the assets.

        :return: The ask price for the symbol.
        """

        return assets_market_prices(
            exchange=exchange, symbol=symbol, prices=self.closes,
            separator=separator, provider=self
        )
    # end ask_volume

    def volume(
            self, exchange: str, symbol: str, separator: Optional[str] = None
    ) -> List[Tuple[dt.datetime, float]]:
        """
        Returns the ask price for the symbol.

        :param exchange: The exchange name.
        :param symbol: The symbol to find its ask price.
        :param separator: The separator of the assets.

        :return: The ask price for the symbol.
        """

        return assets_market_prices(
            exchange=exchange, symbol=symbol, prices=self.closes,
            separator=separator, provider=self
        )
    # end ask_volume

    def in_open_prices(
            self,
            exchange: str,
            symbol: str,
            separator: Optional[str] = None
    ) -> bool:
        """
        Checks if the symbol is in the prices' data.

        :param exchange: The exchange name.
        :param symbol: The symbol to search.
        :param separator: The separator of the assets.

        :return: The validation value.
        """

        return is_symbol_in_assets_market_prices(
            exchange=exchange, symbol=symbol,
            separator=separator, prices=self.opens
        )
    # end in_bids_prices

    def in_high_prices(
            self,
            exchange: str,
            symbol: str,
            separator: Optional[str] = None
    ) -> bool:
        """
        Checks if the symbol is in the prices' data.

        :param exchange: The exchange name.
        :param symbol: The symbol to search.
        :param separator: The separator of the assets.

        :return: The validation value.
        """

        return is_symbol_in_assets_market_prices(
            exchange=exchange, symbol=symbol,
            separator=separator, prices=self.highs
        )
    # end in_asks_prices

    def in_low_prices(
            self,
            exchange: str,
            symbol: str,
            separator: Optional[str] = None
    ) -> bool:
        """
        Checks if the symbol is in the prices' data.

        :param exchange: The exchange name.
        :param symbol: The symbol to search.
        :param separator: The separator of the assets.

        :return: The validation value.
        """

        return is_symbol_in_assets_market_prices(
            exchange=exchange, symbol=symbol,
            separator=separator, prices=self.lows
        )
    # end in_bids_volume_prices

    def in_close_prices(
            self,
            exchange: str,
            symbol: str,
            separator: Optional[str] = None
    ) -> bool:
        """
        Checks if the symbol is in the prices' data.

        :param exchange: The exchange name.
        :param symbol: The symbol to search.
        :param separator: The separator of the assets.

        :return: The validation value.
        """

        return is_symbol_in_assets_market_prices(
            exchange=exchange, symbol=symbol,
            separator=separator, prices=self.closes
        )
    # end in_asks_volume_prices

    def in_volume_prices(
            self,
            exchange: str,
            symbol: str,
            separator: Optional[str] = None
    ) -> bool:
        """
        Checks if the symbol is in the prices' data.

        :param exchange: The exchange name.
        :param symbol: The symbol to search.
        :param separator: The separator of the assets.

        :return: The validation value.
        """

        return is_symbol_in_assets_market_prices(
            exchange=exchange, symbol=symbol,
            separator=separator, prices=self.volumes
        )
    # end in_asks_volume_prices

    def data(self) -> AssetsMarketData:
        """
        Returns the structured data of the state.

        :return: The data of the state.
        """

        return assets_market_data(
            columns=OHLCV_COLUMNS,
            prices={name: getattr(self, name) for name in OHLCV_COLUMNS}
        )
    # end data

    def datasets(self) -> AssetsMarketDatasets:
        """
        Rebuilds the dataset from the market state.

        :return: The dataset of the state data.
        """

        datasets: AssetsMarketDatasets = {}

        for exchange, bases in self.data().items():
            for base, quotes in bases.items():
                for quote, rows in quotes.items():
                    dataset = create_ohlcv_dataframe()

                    for time, row in rows:
                        dataset.loc[time] = row
                    # end for
                # end for
            # end for
        # end for

        return datasets
    # end assets_market_state_to_datasets
# end AssetsMarketStates

def assets_ohlcv_market_state(
        screeners: Optional[Iterable[BaseScreener]] = None,
        separator: Optional[str] = None,
        length: Optional[int] = None,
        adjust: Optional[bool] = True
) -> AssetsOHLCVMarketState:
    """
    Fetches the prices and relations between the assets.

    :param screeners: The price screeners.
    :param separator: The separator of the assets.
    :param length: The length of the prices.
    :param adjust: The value to adjust the length of the sequences.

    :return: The prices of the assets.
    """

    return AssetsOHLCVMarketState(
        screeners=screeners,
        **assets_market_state(
            columns=OHLCV_COLUMNS,
            screeners=screeners, separator=separator,
            length=length, adjust=adjust
        )
    )
# end assets_ohlcv_market_state

SymbolsMarketData = Dict[str, Dict[str, List[Tuple[dt.datetime, Dict[str, float]]]]]
SymbolsMarketDatasets = Dict[str, Dict[str, pd.DataFrame]]

@define(repr=False)
@represent
class SymbolsOHLCVMarketState(OHLCVMarketBase):
    """
    A class to represent the current market state.

    This object contains the state of the market, as Close,
    bids and asks prices of specific assets, gathered from the network.

    attributes:

    - screeners:
        The screener objects to collect the prices of the assets.

    - bids:
        The bids prices of the assets.

    - asks:
        The asks prices of the assets.

    - bids_volume:
        The bids volume prices of the assets.

    - asks_volume:
        The asks volume prices of the assets.

    >>> from crypto_screening.collect.orderbook import AssetsOrderbookMarketState
    >>>
    >>> state = assets_ohlcv_market_state(...)
    """

    opens: SymbolsPrices
    highs: SymbolsPrices
    lows: SymbolsPrices
    closes: SymbolsPrices
    volumes: SymbolsPrices

    def open(self, exchange: str, symbol: str) -> List[Tuple[dt.datetime, float]]:
        """
        Returns the bid price for the symbol.

        :param exchange: The exchange name.
        :param symbol: The symbol to find its bid price.

        :return: The bid price for the symbol.
        """

        return symbols_market_prices(
            exchange=exchange, symbol=symbol,
            prices=self.opens, provider=self
        )
    # end open

    def high(self, exchange: str, symbol: str) -> List[Tuple[dt.datetime, float]]:
        """
        Returns the ask price for the symbol.

        :param exchange: The exchange name.
        :param symbol: The symbol to find its ask price.

        :return: The ask price for the symbol.
        """

        return symbols_market_prices(
            exchange=exchange, symbol=symbol,
            prices=self.highs, provider=self
        )
    # end high

    def low(self, exchange: str, symbol: str) -> List[Tuple[dt.datetime, float]]:
        """
        Returns the bid price for the symbol.

        :param exchange: The exchange name.
        :param symbol: The symbol to find its bid price.

        :return: The bid price for the symbol.
        """

        return symbols_market_prices(
            exchange=exchange, symbol=symbol,
            prices=self.lows, provider=self
        )
    # end low

    def close(self, exchange: str, symbol: str) -> List[Tuple[dt.datetime, float]]:
        """
        Returns the ask price for the symbol.

        :param exchange: The exchange name.
        :param symbol: The symbol to find its ask price.

        :return: The ask price for the symbol.
        """

        return symbols_market_prices(
            exchange=exchange, symbol=symbol,
            prices=self.closes, provider=self
        )
    # end close

    def volume(self, exchange: str, symbol: str) -> List[Tuple[dt.datetime, float]]:
        """
        Returns the ask price for the symbol.

        :param exchange: The exchange name.
        :param symbol: The symbol to find its ask price.

        :return: The ask price for the symbol.
        """

        return symbols_market_prices(
            exchange=exchange, symbol=symbol,
            prices=self.volumes, provider=self
        )
    # end volume

    def in_open_prices(self, exchange: str, symbol: str) -> bool:
        """
        Checks if the symbol is in the prices' data.

        :param exchange: The exchange name.
        :param symbol: The symbol to search.

        :return: The validation value.
        """

        return is_symbol_in_symbols_market_prices(
            exchange=exchange, symbol=symbol, prices=self.opens
        )
    # end in_open_prices

    def in_high_prices(self, exchange: str, symbol: str) -> bool:
        """
        Checks if the symbol is in the prices' data.

        :param exchange: The exchange name.
        :param symbol: The symbol to search.

        :return: The validation value.
        """

        return is_symbol_in_symbols_market_prices(
            exchange=exchange, symbol=symbol, prices=self.highs
        )
    # end in_high_prices

    def in_low_prices(self, exchange: str, symbol: str) -> bool:
        """
        Checks if the symbol is in the prices' data.

        :param exchange: The exchange name.
        :param symbol: The symbol to search.

        :return: The validation value.
        """

        return is_symbol_in_symbols_market_prices(
            exchange=exchange, symbol=symbol, prices=self.lows
        )
    # end in_low_prices

    def in_close_prices(self, exchange: str, symbol: str) -> bool:
        """
        Checks if the symbol is in the prices' data.

        :param exchange: The exchange name.
        :param symbol: The symbol to search.

        :return: The in_asks_volume_prices value.
        """

        return is_symbol_in_symbols_market_prices(
            exchange=exchange, symbol=symbol, prices=self.closes
        )
    # end in_close_prices

    def in_volume_prices(self, exchange: str, symbol: str) -> bool:
        """
        Checks if the symbol is in the prices' data.

        :param exchange: The exchange name.
        :param symbol: The symbol to search.

        :return: The in_asks_volume_prices value.
        """

        return is_symbol_in_symbols_market_prices(
            exchange=exchange, symbol=symbol, prices=self.volumes
        )
    # end in_volume_prices

    def data(self) -> SymbolsMarketData:
        """
        Returns the structured data of the state.

        :return: The data of the state.
        """

        return symbols_market_data(
            columns=OHLCV_COLUMNS,
            prices={name: getattr(self, name) for name in OHLCV_COLUMNS}
        )
    # end data

    def datasets(self) -> SymbolsMarketDatasets:
        """
        Rebuilds the dataset from the market state.

        :return: The dataset of the state data.
        """

        datasets: SymbolsMarketDatasets = {}

        for exchange, symbols in self.data().items():
            for symbol, rows in symbols.items():
                dataset = create_ohlcv_dataframe()

                for time, row in rows:
                    dataset.loc[time] = row
                # end for
            # end for
        # end for

        return datasets
    # end symbols_market_state_to_datasets
# end SymbolsMarketStates

def symbols_ohlcv_market_state(
        screeners: Optional[Iterable[BaseScreener]] = None,
        length: Optional[int] = None,
        adjust: Optional[bool] = True
) -> SymbolsOHLCVMarketState:
    """
    Fetches the prices and relations between the assets.

    :param screeners: The price screeners.
    :param length: The length of the prices.
    :param adjust: The value to adjust the length of the sequences.

    :return: The prices of the assets.
    """

    return SymbolsOHLCVMarketState(
        screeners=screeners,
        **symbols_market_state(
            columns=OHLCV_COLUMNS, screeners=screeners,
            length=length, adjust=adjust
        )
    )
# end symbols_ohlcv_market_state

def merge_symbols_ohlcv_market_states(
        *states: SymbolsOHLCVMarketState, sort: Optional[bool] = True
) -> SymbolsOHLCVMarketState:
    """
    Concatenates the states of the market.

    :param states: The states to concatenate.
    :param sort: The value to sort the prices by the time.

    :return: The states object.
    """

    screeners = []

    for state in states:
        screeners.extend(state.screeners)
    # end for

    return SymbolsOHLCVMarketState(
        screeners=set(screeners),
        **merge_symbols_market_states(
            *states, prices={name: {} for name in OHLCV_COLUMNS}, sort=sort
        )
    )
# end merge_symbols_ohlcv_market_states

def merge_assets_ohlcv_market_states(
        *states: AssetsOHLCVMarketState, sort: Optional[bool] = True
) -> AssetsOHLCVMarketState:
    """
    Concatenates the states of the market.

    :param states: The states to concatenate.
    :param sort: The value to sort the prices by the time.

    :return: The states object.
    """

    screeners = []

    for state in states:
        screeners.extend(state.screeners)
    # end for

    return AssetsOHLCVMarketState(
        screeners=set(screeners),
        **merge_assets_market_states(
            *states, prices={name: {} for name in OHLCV_COLUMNS}, sort=sort
        )
    )
# end merge_assets_ohlcv_market_states

def assets_to_symbols_ohlcv_market_state(
        state: AssetsOHLCVMarketState,
        separator: Optional[str] = None
) -> SymbolsOHLCVMarketState:
    """
    Converts an assets market state into a symbols market state.

    :param state: The source state.
    :param separator: The separator for the symbols.

    :return: The results state.
    """

    return SymbolsOHLCVMarketState(
        **{
            name: assets_to_symbol_market_prices(
                prices=getattr(state, name), separator=separator
            ) for name in OHLCV_COLUMNS
        }
    )
# end assets_to_symbols_ohlcv_market_state

def symbols_to_assets_ohlcv_market_state(
        state: SymbolsOHLCVMarketState,
        separator: Optional[str] = None
) -> AssetsOHLCVMarketState:
    """
    Converts a symbols market state into an assets market state.

    :param state: The source state.
    :param separator: The separator for the symbols.

    :return: The results state.
    """

    return AssetsOHLCVMarketState(
        **{
            name: symbol_to_assets_market_prices(
                prices=getattr(state, name), separator=separator
            ) for name in OHLCV_COLUMNS
        }
    )
# end symbols_to_assets_ohlcv_market_state