from pathlib import Path

import numpy as np
import numpy.testing as npt
import pytest

import topsy
from topsy.drawreason import DrawReason


from topsy.canvas import offscreen
from matplotlib import pyplot as plt

@pytest.fixture
def folder():
    folder = Path(__file__).parent / "output"
    folder.mkdir(exist_ok=True)
    return folder

@pytest.fixture(params=[False, True])
def vis(request):
    vis = topsy.test(1000, render_resolution=200, canvas_class = offscreen.VisualizerCanvas, with_cells=request.param)
    vis.scale = 200.0
    return vis


def test_render(vis, folder):
    result = vis.get_sph_presentation_image()
    assert result.dtype == np.uint8
    
    plt.imsave(folder / "test_density_render.png", result)

    reference_result = [ 48,  20,  55, 255,  56,  17,  69, 255,  73,  21, 100, 255,  84,
        30, 126, 255,  88,  39, 139, 255,  89,  40, 141, 255,  85,  32,
       129, 255,  76,  22, 105, 255,  60,  17,  75, 255,  48,  20,  55,
       255,  58,  17,  72, 255,  81,  26, 117, 255,  92,  53, 154, 255,
        94,  76, 170, 255,  95,  89, 177, 255,  95,  90, 177, 255,  94,
        79, 172, 255,  93,  58, 158, 255,  85,  31, 127, 255,  64,  18,
        83, 255,  78,  24, 111, 255,  93,  58, 158, 255,  95,  95, 179,
       255, 100, 124, 188, 255, 108, 143, 191, 255, 112, 148, 192, 255,
       103, 134, 190, 255,  96, 103, 182, 255,  93,  66, 164, 255,  83,
        30, 124, 255,  89,  41, 142, 255,  95,  88, 176, 255, 102, 131,
       189, 255, 129, 166, 195, 255, 181, 199, 207, 255, 211, 214, 219,
       255, 179, 198, 206, 255, 117, 154, 193, 255,  96, 102, 182, 255,
        92,  52, 153, 255,  93,  58, 158, 255,  96, 109, 184, 255, 118,
       155, 193, 255, 184, 201, 208, 255, 211, 185, 168, 255, 193, 119,
        95, 255, 209, 182, 163, 255, 179, 198, 206, 255, 102, 132, 189,
       255,  94,  71, 167, 255,  93,  65, 163, 255,  98, 117, 186, 255,
       126, 164, 195, 255, 212, 214, 220, 255, 195, 127, 100, 255,  47,
        20,  54, 255, 195, 127, 100, 255, 209, 213, 218, 255, 108, 143,
       191, 255,  94,  79, 172, 255,  93,  60, 160, 255,  97, 111, 185,
       255, 119, 157, 194, 255, 193, 205, 211, 255, 211, 186, 170, 255,
       196, 128, 101, 255, 213, 192, 179, 255, 175, 196, 205, 255, 102,
       132, 189, 255,  94,  75, 169, 255,  90,  46, 147, 255,  95,  93,
       178, 255, 104, 136, 190, 255, 136, 172, 196, 255, 190, 204, 210,
       255, 212, 214, 220, 255, 172, 194, 204, 255, 115, 152, 193, 255,
        96, 106, 183, 255,  93,  59, 159, 255,  81,  26, 117, 255,  93,
        65, 163, 255,  96, 103, 182, 255, 103, 133, 190, 255, 113, 150,
       193, 255, 114, 151, 193, 255, 105, 137, 190, 255,  97, 111, 185,
       255,  94,  76, 170, 255,  87,  36, 135, 255,  59,  17,  74, 255,
        85,  31, 127, 255,  93,  64, 162, 255,  95,  88, 176, 255,  96,
       102, 182, 255,  96, 103, 182, 255,  95,  91, 178, 255,  94,  71,
       167, 255,  89,  41, 142, 255,  71,  20,  95, 255]

    npt.assert_allclose(result[::20, ::20].ravel(), reference_result, atol = 5)

    

def test_hdr_rgb_render(vis, folder):
    vis = topsy.test(1000, render_resolution=200, canvas_class = offscreen.VisualizerCanvas, 
                     render_mode='rgb-hdr')
    vis.scale = 20.0
    vis.colormap.update_parameters({"min_mag": 38.0, "max_mag": 40.0})
    result = vis.get_sph_presentation_image()[..., :3]

    import tifffile
    tifffile.imwrite(folder / 'output.tiff', result, photometric='rgb')

    result_ref = [0.0000e+00, 0.0000e+00, 0.0000e+00, 0.0000e+00, 0.0000e+00,
       0.0000e+00, 0.0000e+00, 0.0000e+00, 0.0000e+00, 0.0000e+00,
       0.0000e+00, 0.0000e+00, 0.0000e+00, 0.0000e+00, 6.4880e-02,
       6.9336e-02, 3.5217e-02, 2.0544e-01, 1.6541e-01, 8.9844e-02,
       3.0615e-01, 1.8469e-01, 6.7261e-02, 2.8003e-01, 9.2163e-02,
       0.0000e+00, 1.0266e-01, 0.0000e+00, 0.0000e+00, 0.0000e+00,
       0.0000e+00, 0.0000e+00, 0.0000e+00, 0.0000e+00, 0.0000e+00,
       0.0000e+00, 0.0000e+00, 0.0000e+00, 0.0000e+00, 5.1727e-02,
       5.1666e-02, 9.2957e-02, 1.4478e-01, 1.8115e-01, 2.8369e-01,
       3.3496e-01, 3.5303e-01, 5.5371e-01, 6.7529e-01, 6.0400e-01,
       9.1797e-01, 7.2461e-01, 5.8545e-01, 9.0771e-01, 4.3213e-01,
       2.9126e-01, 5.0488e-01, 1.6150e-01, 3.9032e-02, 1.0016e-01,
       0.0000e+00, 0.0000e+00, 0.0000e+00, 2.6108e-02, 0.0000e+00,
       0.0000e+00, 1.0046e-01, 1.0583e-01, 6.4575e-02, 1.7163e-01,
       2.4866e-01, 2.3535e-01, 2.8442e-01, 4.0967e-01, 4.6313e-01,
       6.4160e-01, 7.3584e-01, 9.4092e-01, 1.4326e+00, 1.4043e+00,
       1.7422e+00, 1.4844e+00, 1.3799e+00, 1.7246e+00, 7.8271e-01,
       6.6211e-01, 8.9307e-01, 3.2764e-01, 2.3462e-01, 2.7783e-01,
       0.0000e+00, 0.0000e+00, 0.0000e+00, 1.1157e-01, 1.0760e-01,
       1.2436e-02, 2.2888e-01, 3.0688e-01, 2.1631e-01, 3.3374e-01,
       5.0293e-01, 4.2993e-01, 4.1748e-01, 6.5332e-01, 6.2207e-01,
       6.7773e-01, 8.8672e-01, 9.7705e-01, 1.3584e+00, 1.4326e+00,
       1.6670e+00, 1.3984e+00, 1.3887e+00, 1.6270e+00, 7.9834e-01,
       7.6221e-01, 8.9160e-01, 4.0430e-01, 3.6450e-01, 3.3521e-01,
       3.4729e-02, 1.0475e-02, 0.0000e+00, 2.0020e-01, 2.4500e-01,
       1.2408e-01, 4.5337e-01, 6.0010e-01, 5.0146e-01, 7.6807e-01,
       1.0264e+00, 9.5312e-01, 8.5303e-01, 1.2705e+00, 1.1934e+00,
       7.9199e-01, 1.3271e+00, 1.2568e+00, 1.0049e+00, 1.3633e+00,
       1.3350e+00, 9.6289e-01, 1.1465e+00, 1.1504e+00, 6.8457e-01,
       7.6758e-01, 7.3291e-01, 4.5020e-01, 4.5972e-01, 3.6157e-01,
       4.0802e-02, 3.5797e-02, 0.0000e+00, 2.4255e-01, 3.1348e-01,
       1.8469e-01, 6.1133e-01, 7.8613e-01, 6.9678e-01, 1.4355e+00,
       1.7363e+00, 1.6963e+00, 2.2207e+00, 2.7754e+00, 2.7461e+00,
       1.9062e+00, 3.2383e+00, 3.2148e+00, 2.2070e+00, 2.7500e+00,
       2.7227e+00, 1.3857e+00, 1.6689e+00, 1.6328e+00, 7.1582e-01,
       8.3984e-01, 7.5928e-01, 4.7852e-01, 5.0342e-01, 3.8403e-01,
       2.3098e-03, 0.0000e+00, 0.0000e+00, 1.7371e-01, 2.3767e-01,
       1.0162e-01, 4.2896e-01, 5.8740e-01, 4.7583e-01, 7.2461e-01,
       9.8779e-01, 9.0088e-01, 8.0762e-01, 1.2305e+00, 1.1377e+00,
       6.8994e-01, 1.2119e+00, 1.0996e+00, 8.3398e-01, 1.2178e+00,
       1.1250e+00, 7.8223e-01, 1.0010e+00, 9.1943e-01, 5.9424e-01,
       6.9092e-01, 5.8887e-01, 4.3018e-01, 4.3896e-01, 3.1226e-01,
       0.0000e+00, 0.0000e+00, 0.0000e+00, 5.8716e-02, 9.5459e-02,
       0.0000e+00, 1.8213e-01, 2.9395e-01, 1.6760e-01, 2.7783e-01,
       4.6924e-01, 3.4424e-01, 3.1104e-01, 5.6494e-01, 4.3604e-01,
       3.4448e-01, 6.0791e-01, 4.7949e-01, 4.1260e-01, 6.2109e-01,
       5.0244e-01, 4.4849e-01, 5.7178e-01, 4.6191e-01, 4.1357e-01,
       4.5483e-01, 3.4351e-01, 3.3203e-01, 3.0591e-01, 1.8652e-01,
       0.0000e+00, 0.0000e+00, 0.0000e+00, 0.0000e+00, 0.0000e+00,
       0.0000e+00, 3.3081e-02, 9.2102e-02, 0.0000e+00, 9.3384e-02,
       2.1033e-01, 1.1346e-01, 1.3831e-01, 2.9419e-01, 1.9678e-01,
       1.8323e-01, 3.3838e-01, 2.4231e-01, 2.3254e-01, 3.4595e-01,
       2.5317e-01, 2.6636e-01, 3.1665e-01, 2.2717e-01, 2.6367e-01,
       2.5098e-01, 1.6150e-01, 2.1973e-01, 1.5894e-01, 6.6223e-02,
       0.0000e+00, 0.0000e+00, 0.0000e+00, 0.0000e+00, 0.0000e+00,
       0.0000e+00, 0.0000e+00, 0.0000e+00, 0.0000e+00, 0.0000e+00,
       2.2003e-02, 0.0000e+00, 9.6664e-03, 9.0576e-02, 4.5563e-02,
       5.1086e-02, 1.2915e-01, 8.4229e-02, 9.2285e-02, 1.3745e-01,
       9.2041e-02, 1.2189e-01, 1.1792e-01, 7.0374e-02, 1.2500e-01,
       7.3303e-02, 2.1027e-02, 9.0942e-02, 4.7646e-03, 0.0000e+00]
    
    assert result.dtype == np.float16
    npt.assert_allclose(result[::20, ::20].ravel(), result_ref, atol=1e-2)

    
def test_particle_pos_smooth(vis):
    # this is testing the test data
    if hasattr(vis.data_loader, '_cell_layout'):
        return # skip this test if we are using cells
    xyzw = vis.data_loader.get_pos_smooth()
    npt.assert_allclose(xyzw[::100],
       [[ 1.6189760e+01, -4.0728635e-01, -1.8409515e+01,  2.0848181e+01],
       [-3.6236227e-01,  1.9854842e-02, -3.4908600e+00,  1.2997785e+00],
       [ 5.6721487e+00, -8.8317409e-02, -9.4208164e+00,  1.0804868e+01],
       [-3.6954129e+00, -5.1248574e+00,  1.4329858e+01,  1.5497326e+01],
       [-2.5594389e+01, -9.0724382e+00, -3.3397295e+00,  2.3571991e+01],
       [-3.6231318e-01,  1.6435374e-02,  1.8260944e+00,  1.0799329e+00],
       [ 9.7273951e+00,  1.8408798e-01, -6.7287006e+00,  1.3380475e+01],
       [ 1.4229246e+01,  2.2913518e+00, -1.6219862e+01,  1.8701763e+01],
       [ 1.0776958e+01,  1.6861650e+01,  1.8014458e+01,  2.3113770e+01],
       [ 8.6214191e-01, -1.3920928e-02,  1.7059642e+00,  1.0834730e+00]])

def test_sph_weighted_output(vis, folder):
    vis.quantity_name = "test-quantity"
    vis.scale = 20.0
    vis.rotate(0.0,0.4)
    vis.render_sph(DrawReason.EXPORT)
    result = vis.get_sph_image()
    assert result.shape == (200,200)

    np.save(folder / "test_weighted.npy", result)

    test = result[::20,::20].flatten()
    expect =  [ 5.43033502e-06,  5.24370580e-06,  4.10593202e-06,  2.02327237e-06,
       -4.18346104e-07, -2.62014078e-06, -4.05347464e-06, -5.11882399e-06,
       -5.35779054e-06, -4.99097268e-06,  5.45265766e-06,  5.09395159e-06,
        3.48505409e-06,  1.16395825e-06, -8.43042187e-07, -1.45847355e-06,
       -7.42325199e-07, -2.18958917e-06, -4.82037649e-06, -4.59195871e-06,
        5.41132204e-06,  4.35603579e-06,  2.17515367e-06,  4.91636740e-07,
       -5.65283109e-08, -1.40854684e-06, -3.74410342e-06, -5.81178483e-06,
       -5.09375786e-06, -3.57636532e-06,  4.87861234e-06,  2.97656561e-06,
        1.66810560e-06,  2.19740036e-06,  9.82135475e-07, -8.12036262e-07,
       -2.89094419e-06, -7.02606530e-06, -3.94840890e-06, -1.54778229e-06,
        3.99404962e-06,  2.29679290e-06,  1.75102468e-06, -2.59838384e-06,
       -7.99023746e-06,  6.96154939e-06,  1.22214078e-05,  4.38679444e-06,
        5.60235185e-06,  1.50137737e-06,  3.14368890e-06,  1.71596139e-06,
       -1.40251836e-06, -4.41343991e-06, -3.76780258e-06,  2.00669024e-06,
       -2.16206718e-06,  1.01332953e-05,  1.07507904e-05,  3.09911479e-06,
        2.31677564e-06,  8.55237090e-07, -6.15154192e-07,  2.68512372e-06,
        2.10978737e-06, -5.41282191e-07,  1.23767513e-05,  1.22587708e-05,
        8.23176015e-06,  1.06019718e-06,  1.33470564e-06, -3.61378696e-08,
       -1.61409139e-06, -2.52063046e-06, -1.01564396e-07,  3.33199682e-06,
        3.70180169e-06,  2.76784954e-06,  2.63906173e-07, -2.26574616e-06,
       -3.76287801e-09, -6.21360527e-07, -2.10721282e-06, -3.51426252e-06,
       -4.18957279e-06, -4.20794186e-06, -4.03539661e-06, -3.86382771e-06,
       -4.00788394e-06, -4.25639519e-06, -1.58968851e-06, -1.60171351e-06,
       -2.46739432e-06, -3.65177402e-06, -4.68898270e-06, -5.36195603e-06,
       -5.62509740e-06, -5.53361224e-06, -5.27684415e-06, -5.15632610e-06]

    npt.assert_allclose(test, expect, atol=1.5e-7)

def test_sph_output(vis, folder):
    vis.render_sph(DrawReason.EXPORT)
    result = vis.get_sph_image()
    assert result.shape == (200,200)
    np.save(folder / "test.npy", result) # for debugging
    test = result[::20,::20].flatten()

    expect = np.array([1.3719737e-13, 2.2577373e-13, 3.4707128e-13, 4.7913626e-13,
       5.7335636e-13, 5.8166834e-13, 5.0068169e-13, 3.7394534e-13,
       2.5203345e-13, 1.6070023e-13, 2.3526553e-13, 4.3171158e-13,
       7.3597628e-13, 1.1049305e-12, 1.4004169e-12, 1.4294217e-12,
       1.1646425e-12, 7.9952174e-13, 4.8636567e-13, 2.7627079e-13,
       3.9335820e-13, 8.0497160e-13, 1.5378436e-12, 2.6523881e-12,
       3.8650120e-12, 4.2805867e-12, 3.1845073e-12, 1.7736138e-12,
       9.1958971e-13, 4.6606371e-13, 6.0535415e-13, 1.3683649e-12,
       2.9855350e-12, 6.3432588e-12, 1.4027483e-11, 2.2303675e-11,
       1.3719674e-11, 4.8580844e-12, 1.7220782e-12, 7.2667176e-13,
       8.1093319e-13, 1.9735192e-12, 4.8798843e-12, 1.4427965e-11,
       8.4100574e-11, 2.6442615e-10, 8.8517936e-11, 1.3847874e-11,
       3.0435474e-12, 1.0120076e-12, 9.0955282e-13, 2.2799319e-12,
       5.9699832e-12, 2.2838169e-11, 2.3453128e-10, 8.0560795e-08,
       2.3452809e-10, 2.1240560e-11, 3.8278256e-12, 1.1763834e-12,
       8.4215100e-13, 2.0578621e-12, 5.1257293e-12, 1.6460699e-11,
       8.2461420e-11, 2.3309185e-10, 7.4100032e-11, 1.2828785e-11,
       3.0442216e-12, 1.0872116e-12, 6.5038285e-13, 1.4875639e-12,
       3.2932765e-12, 7.1966786e-12, 1.5944173e-11, 2.2467811e-11,
       1.2348589e-11, 4.6734639e-12, 1.8962074e-12, 8.2295775e-13,
       4.2770960e-13, 9.1109151e-13, 1.7781172e-12, 3.0819160e-12,
       4.4282217e-12, 4.6020904e-12, 3.4024370e-12, 2.0485410e-12,
       1.1009980e-12, 5.4363285e-13, 2.4588935e-13, 4.9328543e-13,
       8.7989062e-13, 1.3605021e-12, 1.7387084e-12, 1.7740940e-12,
       1.4578791e-12, 1.0015811e-12, 5.9854254e-13, 3.2601019e-13],
      dtype=np.float32)

    # the following test will seem very weak, but the problem is that different handling of e.g.
    # mipmaps by different renderers means that individual pixels can easily be very different
    # from any reference image. So we just check that the overall image is similar.
    npt.assert_allclose(test, expect, rtol=5e-1)

    # now let's also check that the distribution is sharply peaked around the right value
    assert abs((test/expect).mean()-1.0)<0.0015
    assert (test/expect).std() < 0.015

def test_periodic_sph_output(vis, folder):
    vis2 = topsy.test(1000, render_resolution=200, canvas_class = offscreen.VisualizerCanvas, periodic_tiling=True)
    vis2.scale = 200.0
    vis2.render_sph(DrawReason.EXPORT)
    result = vis2.get_sph_image()

    np.save(folder / "test_periodic.npy", result)

    expect = np.array([8.9322626e-08, 3.1703462e-10, 9.2687735e-10, 1.1192928e-09,
       3.1655625e-10, 8.9333170e-08, 3.2010358e-10, 9.2814262e-10,
       1.1169485e-09, 3.1108691e-10, 3.2055755e-10, 1.6879378e-10,
       2.5913338e-10, 2.5908647e-10, 1.7273429e-10, 3.3135905e-10,
       1.7230860e-10, 2.6059924e-10, 2.5635122e-10, 1.6636349e-10,
       1.3825600e-09, 2.7530075e-10, 6.2645300e-10, 6.9163519e-10,
       2.8291827e-10, 1.3944854e-09, 2.7912050e-10, 6.2802880e-10,
       6.8869560e-10, 2.7608971e-10, 8.7882529e-10, 2.4545646e-10,
       5.1961285e-10, 5.8564026e-10, 2.4835095e-10, 8.9120167e-10,
       2.4955563e-10, 5.2136878e-10, 5.8244543e-10, 2.4107497e-10,
       3.5990838e-10, 1.8307128e-10, 2.9074523e-10, 2.9106720e-10,
       1.8820685e-10, 3.7185485e-10, 1.8720442e-10, 2.9251249e-10,
       2.8785399e-10, 1.8088944e-10, 8.9335380e-08, 3.3031011e-10,
       9.4112340e-10, 1.1338572e-09, 3.3094341e-10, 8.9348021e-08,
       3.3437902e-10, 9.4284891e-10, 1.1307086e-09, 3.2375427e-10,
       3.2530026e-10, 1.7408402e-10, 2.6467059e-10, 2.6495947e-10,
       1.7861751e-10, 3.3720080e-10, 1.7819340e-10, 2.6643898e-10,
       2.6174474e-10, 1.7130720e-10, 1.3846500e-09, 2.7765840e-10,
       6.2892319e-10, 6.9430756e-10, 2.8559652e-10, 1.3971296e-09,
       2.8180488e-10, 6.3067879e-10, 6.9111544e-10, 2.7830191e-10,
       8.7712987e-10, 2.4356558e-10, 5.1764731e-10, 5.8348204e-10,
       2.4619712e-10, 8.8906421e-10, 2.4739205e-10, 5.1924093e-10,
       5.8052546e-10, 2.3932253e-10, 3.5584827e-10, 1.7859708e-10,
       2.8607963e-10, 2.8608999e-10, 1.8324889e-10, 3.6689662e-10,
       1.8223464e-10, 2.8758435e-10, 2.8330593e-10, 1.7673871e-10],
      dtype=np.float32)

    npt.assert_allclose(result[::20,::20].flatten(), expect, rtol=1e-1)

def test_rotated_sph_output(vis):
    vis.draw(reason=DrawReason.EXPORT)
    unrotated_output = vis.get_sph_image()
    vis.rotation_matrix = np.array([[0.0, 1.0, 0.0],
                                    [-1.0, 0.0, 0.0],
                                    [0.0, 0.0, 1.0]], dtype=np.float32)
    try:
        vis.draw(reason=DrawReason.EXPORT)
        rotated_output = vis.get_sph_image()
        npt.assert_allclose(unrotated_output.T[:,::-1], rotated_output, rtol=5e-2)


    finally:
        vis.rotation_matrix = np.eye(3, dtype=np.float32)


def test_rgb_sph_output():
    vis = topsy.test(1000, render_resolution=200, canvas_class = offscreen.VisualizerCanvas, 
                     render_mode='rgb')
    result = vis.get_sph_image()
    assert result.shape == (200,200,3)

def test_depth_output(vis, folder):
    vis = topsy.test(1000, render_resolution=200, canvas_class=offscreen.VisualizerCanvas)
    vis.scale = 20.0
    vis.rotation_matrix = np.array([[1.0, 0.0, 0.0],
                                    [0.0, 0.0, 1.0],
                                    [0.0, -1.0, 0.0]], dtype=np.float32)
    vis.render_sph(DrawReason.EXPORT)

    np.save(folder/"test_depth_context.npy", vis.get_sph_image())

    result = vis._sph.get_depth_image(DrawReason.EXPORT)

    np.save(folder / "test_depth.npy", result)  # for debugging

    expect = np.array([-6.1412215e-01, -1.8200874e-02,  4.4874430e-01,  6.8237543e-01,
        6.7839861e-01,  4.8642159e-01,  1.8122435e-01, -1.5840769e-01,
       -4.8412681e-01, -7.8494072e-01, -7.2757006e-01, -1.7276883e-01,
        2.2809744e-01,  3.7608147e-01,  2.7173996e-01, -4.3261051e-03,
       -3.3126712e-01, -5.7852268e-01, -7.1909428e-01, -8.2638502e-01,
       -9.6994162e-01, -4.7681451e-01, -1.4445305e-01, -9.2931986e-02,
       -3.0393243e-01, -6.7068458e-01, -1.2243545e+00, -1.4892983e+00,
       -1.2386465e+00, -9.9033833e-01, -1.2633693e+00, -8.1685185e-01,
       -5.0349355e-01, -4.2420983e-01, -4.1854501e-01, -7.6590419e-01,
       -2.4770129e+00, -3.5506952e+00, -2.4098432e+00, -1.2935197e+00,
       -1.5107369e+00, -1.0751343e+00, -6.8476319e-01, -3.7224770e-01,
       -1.9266486e-01, -5.1854849e-01, -2.8899932e+00, -5.6981363e+00,
       -4.1787076e+00, -1.8048847e+00, -1.6587865e+00, -1.2280405e+00,
       -7.5689793e-01, -3.1625152e-01, -1.5581965e-01, -4.8974395e-01,
       -3.0715656e+00, -6.4643850e+00, -4.9952149e+00, -2.1794629e+00,
       -1.7302644e+00, -1.3548887e+00, -9.3402386e-01, -5.6250334e-01,
       -3.4593463e-01, -7.3299527e-01, -3.0981314e+00, -5.8333817e+00,
       -4.2573762e+00, -1.8872452e+00, -1.7559803e+00, -1.5045440e+00,
       -1.1963260e+00, -9.8268032e-01, -7.4352264e-01, -1.0188949e+00,
       -2.1862769e+00, -3.7116265e+00, -2.6124454e+00, -1.2816906e+00,
       -1.7749834e+00, -1.6487813e+00, -1.4966643e+00, -1.3906789e+00,
       -1.3710022e+00, -1.5978980e+00, -1.9147623e+00, -1.9956529e+00,
       -1.5321469e+00, -9.3367457e-01, -1.8167615e+00, -1.7895186e+00,
       -1.7594409e+00, -1.7537165e+00, -1.7831147e+00, -1.8056393e+00,
       -1.7693257e+00, -1.6105938e+00, -1.3042843e+00, -9.7951174e-01],
      dtype=np.float32)

    npt.assert_allclose(result[::20,::20].ravel(), expect, atol=1e-1)

def test_bivariate_render(folder):
     vis = topsy.test(1000, render_resolution=200, canvas_class=offscreen.VisualizerCanvas, 
                      render_mode='bivariate')
     vis.quantity_name = "test-quantity" 
     vis.scale = 20.0
     vis.rotate(0.0, 0.5)
     vis.render_sph(DrawReason.EXPORT)

     results = vis.get_sph_image()

     results_mapped = vis.get_sph_presentation_image()

     np.save(folder / "test_bivariate.npy", results)  
     plt.imsave(folder / "test_bivariate.png", results_mapped)

     expect_den = [3.9312412e-10, 4.7599746e-10, 5.6863347e-10, 6.7354927e-10,
       7.9762674e-10, 9.3636621e-10, 1.0364271e-09, 9.9064090e-10,
       8.0666807e-10, 6.2319161e-10, 4.5922757e-10, 5.6623506e-10,
       6.9739570e-10, 8.6821822e-10, 1.1148517e-09, 1.5102289e-09,
       2.2756781e-09, 2.2264808e-09, 1.3428380e-09, 8.5172980e-10,
       5.1488797e-10, 6.5311662e-10, 8.4671409e-10, 1.1253422e-09,
       1.5424028e-09, 2.7147782e-09, 8.0830311e-09, 7.9245330e-09,
       2.4507161e-09, 1.1257710e-09, 5.5836152e-10, 7.4878209e-10,
       1.0871194e-09, 1.6185601e-09, 2.3858224e-09, 4.0894625e-09,
       1.3120607e-08, 1.2001490e-08, 3.1105845e-09, 1.3213722e-09,
       5.8979921e-10, 8.6327701e-10, 1.5466143e-09, 3.4398264e-09,
       8.5859266e-09, 1.2374786e-08, 1.1787147e-08, 6.1500760e-09,
       2.5939035e-09, 1.3924026e-09, 5.9847421e-10, 9.1680519e-10,
       1.9213386e-09, 8.9320347e-09, 4.2709676e-08, 8.8528871e-08,
       4.5847727e-08, 8.5347116e-09, 2.3766069e-09, 1.3836838e-09,
       5.7594068e-10, 8.3680546e-10, 1.4578262e-09, 3.6121675e-09,
       8.2479730e-09, 1.0749143e-08, 6.3318337e-09, 3.2644680e-09,
       1.7947719e-09, 1.2198422e-09, 5.3398275e-10, 7.2408496e-10,
       1.0111308e-09, 1.3948880e-09, 1.8043743e-09, 2.0576789e-09,
       1.9310682e-09, 1.5819723e-09, 1.2835830e-09, 1.0111659e-09,
       4.8206089e-10, 6.3126315e-10, 8.1372481e-10, 1.0106245e-09,
       1.1769051e-09, 1.2567239e-09, 1.2362033e-09, 1.1423658e-09,
       9.9833841e-10, 8.3278107e-10, 4.2052023e-10, 5.4207838e-10,
       6.8129874e-10, 8.1927909e-10, 9.2845009e-10, 9.8159580e-10,
       9.6953101e-10, 9.0201596e-10, 7.9703516e-10, 6.7363504e-10]
     
     expect_qty = [ 5.27777684e-06,  5.02338889e-06,  3.76082312e-06,  1.54505994e-06,
       -9.65626100e-07, -3.16612363e-06, -4.65261473e-06, -5.63181175e-06,
       -5.75905915e-06, -5.40677138e-06,  5.36065045e-06,  4.87131274e-06,
        3.06966012e-06,  6.52256404e-07, -1.33067408e-06, -2.14896340e-06,
       -1.56740180e-06, -2.80610334e-06, -5.17499120e-06, -4.92527170e-06,
        5.33639741e-06,  4.08297319e-06,  1.74656770e-06,  1.50693040e-07,
       -2.88195992e-07, -1.48230595e-06, -3.45540184e-06, -5.15719285e-06,
       -4.88091473e-06, -3.81291193e-06,  4.85461123e-06,  2.79135065e-06,
        1.62166316e-06,  2.47225307e-06,  4.93646098e-07, -5.01904879e-07,
       -2.84332714e-06, -7.46157639e-06, -4.45337946e-06, -1.88908473e-06,
        4.05007131e-06,  2.25219401e-06,  1.76695403e-06, -2.64390337e-06,
       -9.28290956e-06,  6.72942633e-06,  1.20076074e-05,  3.30313901e-06,
        4.81804909e-06,  1.06280152e-06,  3.24520306e-06,  1.70116482e-06,
       -1.78587004e-06, -5.22664504e-06, -2.38219945e-06,  2.79576625e-06,
       -4.28452631e-06,  9.85297265e-06,  1.03941711e-05,  2.65043900e-06,
        2.37316044e-06,  8.83661357e-07, -7.17770490e-07,  3.56188730e-06,
       -4.50909425e-07, -3.41397026e-06,  1.26442701e-05,  1.24703765e-05,
        7.91011462e-06,  5.36693335e-07,  1.32300568e-06, -6.44913669e-08,
       -1.64931748e-06, -2.36467213e-06,  8.35140384e-07,  4.45498017e-06,
        4.03878221e-06,  2.54918427e-06, -2.64560157e-07, -2.84444104e-06,
       -2.51696211e-08, -7.01488489e-07, -2.20365700e-06, -3.59735668e-06,
       -4.07940161e-06, -4.07445214e-06, -4.23153369e-06, -4.30124965e-06,
       -4.56926773e-06, -4.84014799e-06, -1.54250984e-06, -1.58435068e-06,
       -2.48652532e-06, -3.70062548e-06, -4.77558069e-06, -5.52261554e-06,
       -5.91475009e-06, -5.97665576e-06, -5.83010342e-06, -5.73830312e-06]
     
     expect_rgba = [187, 179, 186, 255, 190, 184, 191, 255, 177, 190, 196, 255, 130,
       164, 200, 255, 129, 117, 204, 255, 163,  86, 208, 255, 190, 112,
       210, 255, 189, 112, 209, 255, 182,  99, 204, 255, 174,  83, 198,
       255, 191, 183, 189, 255, 193, 189, 195, 255, 166, 190, 200, 255,
       132, 155, 206, 255, 149, 129, 212, 255, 172, 134, 219, 255, 189,
       172, 229, 255, 197, 155, 228, 255, 199, 132, 216, 255, 184, 102,
       205, 255, 193, 186, 192, 255, 187, 194, 199, 255, 144, 175, 205,
       255, 144, 158, 212, 255, 162, 167, 220, 255, 199, 186, 233, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 219, 174, 231, 255, 181,
       100, 212, 255, 193, 189, 195, 255, 162, 188, 202, 255, 154, 182,
       211, 255, 186, 207, 221, 255, 188, 200, 230, 255, 221, 221, 243,
       255, 255, 255, 255, 255, 255, 255, 255, 255, 226, 188, 236, 255,
       162, 129, 216, 255, 184, 192, 196, 255, 155, 185, 206, 255, 174,
       198, 220, 255, 218, 193, 239, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 251, 252, 253, 255, 231, 230, 232,
       255, 160, 183, 217, 255, 166, 187, 197, 255, 147, 178, 207, 255,
       181, 156, 225, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 230, 180,
       182, 255, 183, 204, 217, 255, 141, 174, 196, 255, 132, 159, 205,
       255, 160, 156, 218, 255, 233, 238, 240, 255, 255, 255, 255, 255,
       255, 255, 255, 255, 253, 250, 252, 255, 237, 190, 215, 255, 223,
       197, 182, 255, 149, 168, 214, 255, 116, 151, 194, 255, 120, 132,
       201, 255, 147, 116, 209, 255, 170, 124, 217, 255, 173, 191, 223,
       255, 223, 224, 226, 255, 218, 222, 225, 255, 186, 207, 220, 255,
       151, 157, 215, 255, 161,  94, 209, 255, 101, 115, 192, 255, 115,
       111, 198, 255, 142,  90, 204, 255, 173,  91, 209, 255, 186, 106,
       213, 255, 188, 111, 214, 255, 190, 113, 214, 255, 188, 108, 212,
       255, 188, 105, 209, 255, 183, 101, 205, 255, 108,  75, 188, 255,
       116,  81, 194, 255, 139,  73, 200, 255, 167,  77, 204, 255, 187,
       108, 207, 255, 188, 111, 209, 255, 188, 110, 208, 255, 186, 106,
       207, 255, 182,  98, 204, 255, 177,  88, 200, 255]
     
     npt.assert_allclose(results[::20,::20,0].ravel(), expect_den, rtol=2e-3)
     npt.assert_allclose(results[::20,::20,1].ravel(), expect_qty, atol=1e-4)
     npt.assert_allclose(results_mapped[::20, ::20].ravel(), expect_rgba, atol=5)
     
def test_surface_render(folder):
    vis = topsy.test(int(1e5), render_resolution=200, canvas_class=offscreen.VisualizerCanvas, 
                     render_mode='surface')
    vis.quantity_name = "test-quantity"
    vis.scale = 12.0
    vis.rotate(0.0, 1.0)
    vis._sph.set_log_density_cut(-8.0)

    vis.render_sph(DrawReason.EXPORT)
    result = vis.get_sph_image()
    presentation_result = vis.get_sph_presentation_image()

    np.save(folder / "test_surface.npy", result)  
    plt.imsave(folder / "test_surface.png", presentation_result)

    assert result.shape == (200, 200, 2)
    assert presentation_result.shape == (200, 200, 4)

    quantity_expectation = [ 0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
        0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
        0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
        0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
        0.0000000e+00,  1.2627806e-06, -1.9208553e-06,  0.0000000e+00,
        0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  8.7623819e-05,
       -4.5173900e-05, -1.5075739e-05, -6.2061074e-05, -2.1750957e-05,
        2.1605796e-05,  3.0697254e-05,  0.0000000e+00,  0.0000000e+00,
        3.9650044e-05, -5.5069297e-05, -4.8436669e-07,  1.2586129e-05,
        1.6290449e-05, -2.1750957e-05, -5.1590555e-06, -2.1432481e-05,
        0.0000000e+00, -2.3879318e-05, -4.1565469e-05,  7.4241570e-06,
       -2.9396995e-05,  3.9767228e-06, -6.2929721e-05, -9.5712985e-06,
       -1.9593331e-06,  7.9276455e-05,  0.0000000e+00, -3.1593663e-05,
        9.9126214e-07,  1.6910117e-05, -4.2193398e-05,  5.3727608e-06,
       -2.3907645e-05, -6.7366629e-05,  9.9066547e-06,  4.1470776e-05,
        0.0000000e+00,  8.6858345e-05,  5.6660399e-05, -4.1616713e-05,
       -2.3270250e-05, -2.6348605e-05, -3.0074301e-05,  6.2949191e-05,
       -5.1351271e-06, -9.8931298e-05,  0.0000000e+00,  0.0000000e+00,
       -4.5556503e-06,  7.9481695e-05, -2.2355393e-05,  4.6248635e-05,
        4.3881377e-05, -7.0402835e-05,  7.1850445e-07,  0.0000000e+00,
        0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  5.6903831e-05,
        5.1868010e-05,  9.3222552e-06, -1.9013412e-05, -1.0383947e-05,
        0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
        0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00,
        0.0000000e+00,  0.0000000e+00,  0.0000000e+00,  0.0000000e+00]
    
    depth_expectation = [0.        , 0.        , 0.        , 0.        , 0.        ,
       0.        , 0.        , 0.        , 0.        , 0.        ,
       0.        , 0.        , 0.        , 0.        , 0.        ,
       0.        , 0.        , 0.94520485, 0.9445283 , 0.        ,
       0.        , 0.        , 0.        , 0.40774894, 0.40879694,
       0.41780904, 0.9615057 , 1.0171885 , 1.0240613 , 0.9471786 ,
       0.        , 0.        , 0.47388926, 0.4770228 , 0.4846701 ,
       0.48609883, 0.99314255, 1.0301464 , 1.0184826 , 0.97983366,
       0.        , 0.51723   , 0.54002076, 0.54525614, 0.5543386 ,
       0.5514293 , 0.958703  , 1.0371937 , 1.0066458 , 0.9491798 ,
       0.        , 0.57878494, 0.60793567, 0.6131719 , 0.6098464 ,
       0.6142544 , 0.61150104, 0.92614883, 0.8997109 , 0.58035785,
       0.        , 0.62010926, 0.6665546 , 0.6780898 , 0.67567575,
       0.677758  , 0.6800938 , 0.6768837 , 0.6670484 , 0.6176137 ,
       0.        , 0.        , 0.7003948 , 0.7318063 , 0.7346242 ,
       0.735659  , 0.72950095, 0.72700244, 0.70105433, 0.        ,
       0.        , 0.        , 0.        , 0.7298169 , 0.7576468 ,
       0.77453727, 0.7651245 , 0.73390764, 0.        , 0.        ,
       0.        , 0.        , 0.        , 0.        , 0.        ,
       0.        , 0.        , 0.        , 0.        , 0.        ]
    
    npt.assert_allclose(result[::20, ::20, 0].ravel(), quantity_expectation, rtol=1e-3)
    npt.assert_allclose(result[::20, ::20, 1].ravel(), depth_expectation, rtol=1e-3)
    
    presentation_expectation = np.array([
          0,   0,   0, 255,   0,   0,   0, 255,   0,   0,
          0, 255,   0,   0,   0, 255,   0,   0,   0, 255,
          0,   0,   0, 255,   0,   0,   0, 255,   0,   0,
          0, 255,   0,   0,   0, 255,   0,   0,   0, 255,
          0,   0,   0, 255,   0,   0,   0, 255,   0,   0,
          0, 255,   0,   0,   0, 255,   0,   0,   0, 255,
          0,   0,   0, 255,   0,   0,   0, 255,   0,   0,
         45, 255,   0,   0,  44, 255,   0,   0,   0, 255,
          0,   0,   0, 255,   0,   0,   0, 255,   0,   0,
          0, 255,   0,   0,  10, 255,  26,  31,  81, 255,
         24,  27,  63, 255,  25,  17,  76, 255,   0,   0,
         40, 255, 117, 101, 121, 255,  57,  43,  59, 255,
          0,   0,   0, 255,   0,   0,   0, 255,  14,   9,
         25, 255,  14,  13,  60, 255,  50,  49,  94, 255,
         44,  41,  79, 255,  77,  70, 103, 255, 138, 170,
        228, 255, 152, 156, 203, 255,  29,  36,  80, 255,
          0,   0,   0, 255,   0,   0,  39, 255,  12,  15,
         61, 255,  33,  31,  75, 255,  18,  24,  68, 255,
         93,  90, 138, 255,  59,  37, 133, 255, 169, 180,
        227, 255, 142, 141, 190, 255,  50,  12,  53, 255,
          0,   0,   0, 255,  25,  33,  80, 255,  56,  54,
        101, 255,  51,  46,  79, 255,  28,  35,  91, 255,
         99,  95, 142, 255,   8,  10,  50, 255,  47,  23,
        105, 255, 111, 105, 147, 255,  37,  22,  36, 255,
          0,   0,   0, 255,  14,   4,  25, 255,  56,  23,
         43, 255,  26,  33,  86, 255,  19,  24,  66, 255,
         46,  60, 109, 255,  36,  47,  97, 255,  60,  21,
         48, 255,  15,  15,  59, 255,  14,   6,  28, 255,
          0,   0,   0, 255,   0,   0,   0, 255, 138, 140,
        188, 255,  26,   6,  35, 255,  30,  37,  81, 255,
         86,  46,  57, 255,  55,  31,  44, 255,  33,  13,
         78, 255, 131, 128, 178, 255,   0,   0,   0, 255,
          0,   0,   0, 255,   0,   0,   0, 255,   0,   0,
          0, 255, 160,  66,  93, 255, 125,  59,  74, 255,
        166, 157, 200, 255, 126, 150, 203, 255, 156, 168,
        216, 255,   0,   0,   0, 255,   0,   0,   0, 255,
          0,   0,   0, 255,   0,   0,   0, 255,   0,   0,
          0, 255,   0,   0,   0, 255,   0,   0,   0, 255,
          0,   0,   0, 255,   0,   0,   0, 255,   0,   0,
          0, 255,   0,   0,   0, 255,   0,   0,   0, 255,
    ], dtype=np.uint8)

    npt.assert_allclose(presentation_result[::20, ::20].ravel(), presentation_expectation, atol=5)
