import re
import typing

from pydantic import BaseModel


class RegexRuleSet(BaseModel):
    rules: dict[str, typing.Any]

    def __init__(self, rules):
        super().__init__(rules=rules)

    def __getitem__(self, key):
        for k, v in self.rules.items():
            if re.match(k, key):
                # Return a deep copy to prevent mutation of built-in rules
                from copy import deepcopy
                return deepcopy(v)


REGEXP_TRACKING_PARAMS = re.compile(
    r"^(__s|_branch_match_id|_bta_\w+|_ga|_hs_\w+|_ke|_kx|\w*clid\w*|campid|counter|\w*cmp\w*|\w*_cid|linkId|isMobile|"
    r"\w*cid|t|mod|[\w\-]*source[\w\-]*|src|at_\w+|device\w*|\w*tag\w*|\wlink\w*|taid|customid|r|mod|"
    r"dm_i|ef_id|epik|gbraid|gclsrc|gdffi|gdfms|sh|gdftrk|hsa_\w+|ic_id|igshid|matomo_\w+|"
    r"mc_\w+|mkcid|mkevt|mkrid|mkt_\w+|mkwid|msclkid|mtm_\w+|ncid|nr_email_refer|ns_\w+|"
    r"ocid|pcrid|piwik_\w+|pk_(campaign|keywords|kwd)|redirect_\w+|ref|s_kwcid|sb_referer_host|"
    r"si|sms_\w+|spm|sr_share|srcid|toolid|trk_\w+|utm_\w+|utmz|vero_\w+|wbraid|z|xto|_\w+|cvid|eir)|"
    r"(?:dc|fbc|gc|twc|yc|ysc)lid|(?:click|gbra|msclk|igsh|partner|wbra)id|(?:ads?|mc|ga|gs|itm|mc|mkt|ml|mtm|oly|pk|utm|vero)_|"
    r"(?:\b|_)(?:aff|affi|affiliate|campaign|cl?id|eid|ga|gl|kwd|keyword|medium|ref|referr?er|session|source|uid|xtor)$",
    re.IGNORECASE,
)

REGEXP_GLOBAL_CANONICAL_PARAMS = re.compile(
    r"^id$|^q$|^v$|^chash$|^action$|^redirect\w*$|url$",
)


REGEXP_TRACKING_VALUES = re.compile(
    r"^.*twitter.*|.*social.*|.*email.*|ad$", re.IGNORECASE
)

match_www = r"(www\.)?"

match_any_subdomains = r"([a-z\.]+)?"

DOMAIN_RULES = RegexRuleSet(
    {
        f"{match_any_subdomains}substack.com": {
            "canonical": {"post_id", "publication_id"}
        },
        f"{match_any_subdomains}facebook.com": {
            "canonical": {
                "story_fbid",
                "fbid",
                "legacy_comment_id",
                "reply_comment_id",
                "group_id",
                "photo_id",
                "video_id",
                "comment_id",
            },
            "force_www": True,
        },
        f"{match_any_subdomains}politico.com": {"non_canonical": {"homescreen"}},
        f"{match_any_subdomains}rumble.com": {"force_www": False},
        f"{match_any_subdomains}threads.net": {
            "force_www": True,
        },
        f"{match_any_subdomains}twitter.com": {
            "canonical": {
                "status",
            },
            "non_canonical": {"t", "s", "lang"},
            "force_www": False,
        },
        f"{match_any_subdomains}x.com": {
            "canonical": {
                "status",
            },
            "non_canonical": {"t", "s", "lang"},
            "force_www": False,
        },
        f"{match_any_subdomains}snapchat.com": {
            "non_canonical": {
                "web_client_id",
            },
            "force_www": True,
        },
        f"{match_any_subdomains}instagram.com": {
            "non_canonical": {
                "igsh",
                "hl",
            }
        },
        f"{match_any_subdomains}linkedin.com": {"force_www": True},
        f"{match_any_subdomains}google.com": {"disable": True},
        f"{match_any_subdomains}washingtontimes(news)?.com": {
            "non_canonical": {
                "bt_ee",
                "bt_ts",
            }
        },
        f"{match_any_subdomains}nytimes.com": {
            "force_www": True,
            "non_canonical": {"smid", "emc", "smtyp", "unlocked_article_code"},
        },
        f"{match_any_subdomains}youtube.com": {
            "force_www": True,
            "non_canonical": {"feature", "t", "view_as"},
        },
        f"{match_any_subdomains}eventbrite.com": {
            "force_www": True,
            "non_canonical": {"aff", "hss_channel"},
        },
        f"{match_any_subdomains}newsbreakapp.com": {
            "non_canonical": {"lang", "pd", "s", "send_time"}
        },
        f"{match_any_subdomains}washingtonpost.com": {
            "non_canonical": {"pwapi_token", "isMobile"}
        },
        f"{match_any_subdomains}theepochtimes.com": {"non_canonical": {"src_src"}},
        # ?apiversion=v2&noservercache=1&domshim=1&renderwebcomponents=1&wcseo=1&batchservertelemetry=1&noservertelemetry=1
        f"{match_any_subdomains}msn.com": {
            "non_canonical": {
                "apiversion",
                "noservercache",
                "domshim",
                "renderwebcomponents",
                "wcseo",
                "batchservertelemetry",
                "noservertelemetry",
            }
        },
        # 'alle.com': {
        #     'to' (extract)
        # }
    }
)


PROTOCOLS = {"http", "https"}

# domain/host names
IP_SET = {
    ".",
    ":",
    "0",
    "1",
    "2",
    "3",
    "4",
    "5",
    "6",
    "7",
    "8",
    "9",
    "a",
    "b",
    "c",
    "d",
    "e",
    "f",
}

# https://github.com/python-validators/validators/blob/master/src/validators/domain.py
VALID_DOMAIN_PORT = re.compile(
    # First character of the domain
    r"^(?:[a-zA-Z0-9]"
    # Sub domain + hostname
    + r"(?:[a-zA-Z0-9-_]{0,61}[A-Za-z0-9])?\.)"
    # First 61 characters of the gTLD
    + r"+[A-Za-z0-9][A-Za-z0-9-_]{0,61}"
    # Last character of the gTLD
    + r"[A-Za-z]"
    # Port number
    + r"(\:(6553[0-5]|655[0-2][0-9]|65[0-4][0-9]{2}|"
    + r"6[0-4][0-9]{3}|[1-5][0-9]{4}|[1-9][0-9]{0,3}))?$",
    re.IGNORECASE,
)

# content filters
SITE_STRUCTURE = re.compile(
    # wordpress
    r"/(?:wp-(?:admin|content|includes|json|themes)|"
    r"paged?|seite|search|suche|gall?er[a-z]{1,2}|labels|"
    r"archives|uploads|modules|attachment|oembed)/|"
    # wordpress + short URL
    r"[/_-](?:tags?|schlagwort|[ck]ategor[a-z]{1,2}|[ck]at|auth?or|user)/[^/]+/?$|"
    # mixed/blogspot
    r"[^0-9]/[0-9]+/[0-9]+/$|[^0-9]/[0-9]{4}/$",
    re.IGNORECASE,
)

FILE_TYPE = re.compile(
    r"\.(atom|json|css|xml|js|jpg|jpeg|png|svg|gif|tiff|pdf|ogg|mp3|m4a|aac|"
    r"avi|mp4|mov|web[mp]|flv|ico|pls|zip|tar|gz|iso|swf|woff|eot|ttf)\b|"
    r"[/-](img|jpg|png)(\b|_)",
    re.IGNORECASE,
)  # (?=[&?])

ADULT_AND_VIDEOS = re.compile(
    r"[/_-](?:bild-?kontakte|fick|gangbang|incest|live-?cams?|live-?chat|"
    r"porno?|sexcam|sexyeroti[ck]|swinger|x{3})\b",
    re.IGNORECASE,
)

# language filter
PATH_LANG_FILTER = re.compile(
    r"(?:https?://[^/]+/)([a-z]{2})([_-][a-z]{2,3})?(?:/|$)", re.IGNORECASE
)
ALL_PATH_LANGS = re.compile(r"(?:/)([a-z]{2})([_-][a-z]{2})?(?:/)", re.IGNORECASE)

ALL_PATH_LANGS_NO_TRAILING = re.compile(
    r"(?:/)([a-z]{2})([_-][a-z]{2})?(?:/|$)", re.IGNORECASE
)

HOST_LANG_FILTER = re.compile(
    r"https?://([a-z]{2})\.(?:[^.]{4,})\.(?:[^.]+)(?:\.[^.]+)?/", re.IGNORECASE
)

# navigation/crawls
NAVIGATION_FILTER = re.compile(
    r"[/_-](archives|auth?or|[ck]at|category|kategorie|paged?|schlagwort|seite|tags?|topics?|user)/|\?p=[0-9]+",
    re.IGNORECASE,
)

NOT_CRAWLABLE = re.compile(
    r"/([ck]onta[ck]t|datenschutzerkl.{1,2}rung|login|impressum|imprint)(\.[a-z]{3,4})?/?$|/login\?|"
    r"/(javascript:|mailto:|tel\.?:|whatsapp:)",
    re.IGNORECASE,
)
# |/(www\.)?(facebook\.com|google\.com|instagram\.com|twitter\.com)/
INDEX_PAGE_FILTER = re.compile(
    r".{0,5}/(default|home|index)(\.[a-z]{3,5})?/?$", re.IGNORECASE
)

# document types
EXTENSION_REGEX = re.compile(r"\.[a-z]{2,5}$")
# https://en.wikipedia.org/wiki/List_of_file_formats#Web_page
WHITELISTED_EXTENSIONS = {
    ".adp",
    ".amp",
    ".asp",
    ".aspx",
    ".cfm",
    ".cgi",
    ".do",
    ".htm",
    ".html",
    ".htx",
    ".jsp",
    ".mht",
    ".mhtml",
    ".php",
    ".php3",
    ".php4",
    ".php5",
    ".phtml",
    ".pl",
    ".shtml",
    ".stm",
    ".txt",
    ".xhtml",
    ".xml",
}


HTML_REDIRECT_DOMAINS = {
    "t.co",
    "apple.news",
}


KNOWN_CDN_DOMAINS = {
    "akamaihd.net",
    "cloudflare.com",
    "cloudfront.net",
    "fastly.net",
    "googleusercontent.com",
    "gstatic.com",
    "tiktokcdn.com",
    "tiktokcdn-us.com",
    "akamai.net",
    "akadns.net",
    "cdn77.org",
}


SHORT_URL_PROVIDERS = {
    "00m.in",
    "0rz.tw",
    "0x.co",
    "1-url.net",
    "10.wsls.com",
    "11w.rs",
    "126.am",
    "1517.org",
    "19thne.ws",
    "1b.yt",
    "1blk.co",
    "1ft.io",
    "1link.in",
    "1tk.us",
    "1tn.co",
    "1un.fr",
    "1url.com",
    "1url.cz",
    "1wb2.net",
    "2.gp",
    "2.ht",
    "207.net",
    "20ex.io",
    "21w.co",
    "23o.net",
    "2ad.in",
    "2big.at",
    "2doc.net",
    "2fear.com",
    "2h.ae",
    "2kgam.es",
    "2o7.net",
    "2pl.us",
    "2tu.us",
    "2ty.in",
    "2u.pw",
    "2u.xf.cz",
    "2wsb.tv",
    "2ya.com",
    "38d.gs",
    "3ly.link",
    "3ra.be",
    "3words.ngl.link",
    "3x.si",
    "4.nbcla.com",
    "4.nbcny.com",
    "45.djt.app",
    "45.wh.gov",
    "49rs.co",
    "4dca.org",
    "4for4.co",
    "4i.ae",
    "4nn.cx",
    "4sq.com",
    "4url.cc",
    "4view.me",
    "511cc.org",
    "5em.cz",
    "5url.net",
    "5z8.info",
    "6fr.ru",
    "6g6.eu",
    "6url.com",
    "7.ly",
    "76.gd",
    "77.ai",
    "7fth.cc",
    "7li.in",
    "7ny.tv",
    "7plus.link",
    "7sport.link",
    "7vd.cn",
    "8.wfla.com",
    "808ne.ws",
    "8bp.co",
    "8t6.me",
    "8ty6.link",
    "8u.cz",
    "944.la",
    "98.to",
    "9m.no",
    "9nl.com",
    "9qr.de",
    "a.atvi.com",
    "a.co",  # amazon
    "a.gg",
    "a.nf",
    "a.pgtb.me",
    "a0.fr",
    "a1chs.org",
    "a2a.me",
    "aaa.tl",
    "ab.co",
    "aba.social",
    "aba.tips",
    "abbr.sk",
    "abbrr.com",
    "abc.com.py",
    "abc.net.au",
    "abc30.tv",
    "abc7.la",
    "abc7ne.ws",
    "abcn.ws",
    "abdn.io",
    "abit.ly",
    "abr.ge",
    "acadmu.se",
    "acdc.lnk.to",
    "ace.mn",
    "acgov.org",
    "aclj.org",
    "acluva.org",
    "acortar.link",
    "acr.tw",
    "act.au.org",
    "act.gp",
    "actl.ink",
    "actnbl.org",
    "acton.is",
    "acumati.ca",
    "ad-med.cz",
    "ad5.eu",
    "ad7.biz",
    "adb.ug",
    "adbe.ly",
    "adbl.co",
    "adelph.it",
    "adf.ly",
    "adfa.st",
    "adfly.fr",
    "adfoc.us",
    "adjix.com",
    "adli.pw",
    "adm0.page.link",
    "admy.link",
    "adobe.ly",
    "adv.li",
    "adweek.it",
    "aflcio.mn",
    "aflcio.org",
    "aflds.org",
    "afpius.com",
    "afsc.me",
    "aggi.es",
    "aj.audio",
    "aje.io",
    "aje.me",
    "ajn.me",
    "aka.gr",
    "aka.ms",
    "akhb.ar",
    "al.ly",
    "album.ink",
    "alil.in",
    "almon.co",
    "alnk.to",
    "alturl.com",
    "alz.org",
    "ambar.org",
    "ambi.org",
    "amc.film",
    "amn.gs",
    "amn.st",
    "amon.to",
    "ampr.gs",
    "amz.run",
    "amzn.eu",
    "amzn.to",
    "an.to",
    "ana.net",
    "ancstry.me",
    "any.gs",
    "anz.co",
    "ap.lc",
    "apne.ws",
    "apple.co",
    "apple.news",
    "aqva.pl",
    "aqx.es",
    "ara.cat",
    "ara.tv",
    "arab.news",
    "arcg.is",
    "archny.org",
    "arcnct.co",
    "arena.run",
    "ares.tl",
    "arkinv.st",
    "asa.lnk.to",
    "ask.fm",
    "asm.social",
    "asso.in",
    "at.apa.org",
    "at.law.com",
    "at.pw.org",
    "at.wftv.com",
    "atani.me",
    "atldsa.org",
    "atlfed.org",
    "atlus.link",
    "atm.tk",
    "atmlb.com",
    "atnfl.co",
    "atom.bio",
    "atres.red",
    "atu.ca",
    "au.ms",
    "auda.cy",
    "audmns.com",
    "aura.com",
    "avoa.co",
    "ayt.fr",
    "azali.fr",
    "azarus.io",
    "azmag.gov",
    "b-ex.it",
    "b-gat.es",
    "b-o-e.uk",
    "b.link",
    "b00.fr",
    "b23.ru",
    "b23.tv",
    "b54.in",
    "bacn.me",
    "baid.us",
    "bam.bz",
    "bamwx.com",
    "bank.sbi",
    "bars.tl",
    "baw.com",
    "bbc.in",
    "bc.vc",
    "bccn.rs",
    "bddy.me",
    "beast.pub",
    "bee4.biz",
    "benchurl.com",
    "bento.me",
    "benvgo.com",
    "bet.us",
    "betfury.tv",
    "beth.games",
    "betmgm.com",
    "bfas.io",
    "bgfn.sh",
    "bgovgo.com",
    "bhpho.to",
    "bi.gt",
    "bim.im",
    "binkd.co",
    "bio.link",
    "bio.site",
    "biodiv.us",
    "bit.do",
    "bit.ly",
    "bitly.com",
    "bitly.net",
    "bitly.ws",
    "bitmark.me",
    "bitw.in",
    "bityl.co",
    "bizj.us",
    "bkite.com",
    "bland.ai",
    "blap.net",
    "blawgo.com",
    "blbrd.cm",
    "ble.pl",
    "blet.link",
    "blip.tv",
    "blizz.ly",
    "bloat.me",
    "bloom.bg",
    "blst.to",
    "blt.mx",
    "bmc.link",
    "bndcmpr.co",
    "bnds.us",
    "bofa.com",
    "bohomen.co",
    "boi.re",
    "boi.st",
    "boomer.link",
    "bote.me",
    "bougn.at",
    "boxd.it",
    "bpl.kr",
    "br4.in",
    "bra.in",
    "brezk.com",
    "brill.ws",
    "brk.to",
    "brkwy.io",
    "brnw.ch",
    "brook.gs",
    "brow.nz",
    "brtn.co",
    "brzu.net",
    "bsta.rs",
    "bt.sg",
    "btaxgo.com",
    "btml.us",
    "bu.lk",
    "bucha.film",
    "bucks.as",
    "bucks.com",
    "budurl.com",
    "buff.ly",
    "buk.me",
    "bul.lu",
    "bung.ie",
    "burnurl.com",
    "bwnews.pr",
    "bxl.me",
    "bzfd.it",
    "bzh.me",
    "c-o.in",
    "c-r.link",
    "c23.biz",
    "cachor.ro",
    "cagop.win",
    "caha.org.au",
    "caid.org.uk",
    "cal.news",
    "calbea.rs",
    "captur.in",
    "carmarket.jp",
    "cartn.co",
    "cash.app",
    "catchylink.com",
    "cavs.com",
    "cbp.gov",
    "cbrogan.me",
    "cbs.so",
    "cbs58.com",
    "cbsloc.al",
    "cbsn.ws",
    "cbug.cc",
    "cc.cc",
    "ccj.im",
    "ccnc.me",
    "ccri.edu",
    "ccvic.org",
    "ce.do",
    "cebv.us",
    "ceip.org",
    "cenccu.co",
    "ceolink.co",
    "cf.ly",
    "cf2.me",
    "cf6.co",
    "cflam.es",
    "cfw.pub",
    "chddr.tv",
    "checkd.info",
    "checkthe.info",
    "chfs.me",
    "chilp.it",
    "chixly.co",
    "chng.it",
    "chopsui.gg",
    "chortl.es",
    "chrg.rs",
    "chrn.cl",
    "chron.it",
    "chroni.cl",
    "chrtbl.org",
    "cipc.me",
    "cisa.gov",
    "cit.ac",
    "cjb.net",
    "cjnrc.org",
    "cl.ly",
    "clar.in",
    "clbn.us",
    "clck.ru",
    "clckt.me",
    "cle.clinic",
    "cli.gs",
    "cli.re",
    "clickmeter.com",
    "clikk.in",
    "cltgov.me",
    "clubz.bg",
    "cm.to",
    "cmhc.info",
    "cmplx.it",
    "cn86.org",
    "cna.asia",
    "cnb.cx",
    "cnet.co",
    "cnn.id",
    "cnn.it",
    "cnnmon.ie",
    "cntrvlr.co",
    "coc.is",
    "coinurl.com",
    "comca.st",
    "con.mk",
    "confess.ngl.link",
    "conta.cc",
    "cornyn.co",
    "cort.as",
    "cosun.co",
    "couic.fr",
    "cr.tl",
    "crg.ng",
    "crier.co",
    "crp.org",
    "crpto.la",
    "cruk.ink",
    "crwdstr.ke",
    "crwl.it",
    "cs.is",
    "cs.pn",
    "cst.fyi",
    "cst.org.uk",
    "csulb.edu",
    "ct.edu",
    "ctia.it",
    "ctt.ac",
    "ctt.ec",
    "ctu.sh",
    "ctx.li",
    "ctx.ly",
    "cubs.com",
    "cudder.it",
    "cup.org",
    "cur.lt",
    "cur.lv",
    "curl.im",
    "cut.do",
    "cut.pe",
    "cut.sk",
    "cutit.org",
    "cutt.eu",
    "cutt.ly",
    "cutt.us",
    "cutu.me",
    "cuturl.co",
    "cuturl.com",
    "cvent.me",
    "cvsoci.al",
    "cybr.fr",
    "cyonix.to",
    "d.pr",
    "d1ba.se",
    "d75.eu",
    "da.gd",
    "daa.pl",
    "dai.ly",
    "dailym.ai",
    "dal.city",
    "dav.la",
    "db.tt",
    "db2024.link",
    "dbd.game",
    "dcfs.la",
    "dd.ma",
    "ddot.dc.gov",
    "ddp.net",
    "deacs.info",
    "dean24.io",
    "decenturl.com",
    "delian.io",
    "deloi.tt",
    "dem.sv",
    "demgovs.co",
    "detne.ws",
    "dfl.org",
    "dfl8.me",
    "dfnd.us",
    "dft.ba",
    "dhunt.in",
    "di.do",
    "di.sn",
    "did.li",
    "digbig.com",
    "digg.com",
    "diplo.de",
    "direc.to",
    "discord.gg",
    "disney.com",
    "disq.us",
    "dj-m.ag",
    "dkel.ec",
    "dkn.fm",
    "dkng.co",
    "dkng.me",
    "dky.bz",
    "dld.bz",
    "dlsite.jp",
    "dlvr.it",
    "docdro.id",
    "docs.chain.link",
    "doi.org",
    "doiop.com",
    "dolp.cc",
    "dopice.sk",
    "dozz.es",
    "dpgr.am",
    "dpo.st",
    "drbob.com",
    "drippi.link",
    "droid.ws",
    "drugch.nl",
    "drw.sh",
    "dtjax.com",
    "dtk.id",
    "ducat.it",
    "duke.is",
    "duran.io",
    "dv.gd",
    "dvx.cm",
    "dwarfurl.com",
    "dxe.io",
    "dy.fi",
    "dyna.me",
    "dyo.gs",
    "e.lga.to",
    "e.lilly",
    "e37.eu",
    "easyuri.com",
    "easyurl.net",
    "econ.st",
    "econgrp.co",
    "ecoti.in",
    "ecra.se",
    "ecs.page.link",
    "ecw.co",
    "edf.org",
    "edin.ac",
    "edl.mn",
    "edmu.in",
    "edtru.st",
    "edut.to",
    "ee.co.uk",
    "eef.li",
    "eepurl.com",
    "ehu.ac.uk",
    "ejus.tc",
    "eldbt.com",
    "eliteza.co",
    "ellemag.co",
    "ely.re",
    "emirat.es",
    "emojilink.me",
    "empr.me",
    "en.mdn.tv",
    "en.sun.mv",
    "endpol.io",
    "engt.co",
    "enter.gy",
    "entm.ag",
    "envict.org",
    "eonli.ne",
    "epa.gov",
    "epi.org",
    "eps.sh",
    "ept.ms",
    "eptms.jp",
    "eqbot.com",
    "er.cz",
    "erax.cz",
    "erie.gov",
    "erj.lnk.to",
    "erw.cz",
    "es-rt.com",
    "es.mdn.tv",
    "es.pn",
    "esbo.nyc",
    "esp.to",
    "esqr.co",
    "esxpol.uk",
    "esyurl.com",
    "et.tv",
    "etf.fund",
    "etoro.tw",
    "etown.edu",
    "etsy.me",
    "etwn.us",
    "eu1.hubs.ly",
    "eurac.tv",
    "europa.eu",
    "eviden.cc",
    "evrgn.co",
    "evrn.sl",
    "ew.edu",
    "ewar.ren",
    "ewerl.com",
    "ewg.life",
    "ex9.co",
    "ey.com",
    "ezurl.cc",
    "f-st.co",
    "f.io",
    "f24.my",
    "fa.b",
    "fa.do",
    "faa.gov",
    "fable.co",
    "faccc.org",
    "fal.cn",
    "falter.at",
    "fam.ag",
    "fandom.link",
    "fanlink.to",
    "fanpa.ge",
    "fans.ly",
    "fappa.org",
    "faz.net",
    "fb.me",
    "fbjh.in",
    "fc.bayern",
    "fdip.fr",
    "fdrl.st",
    "fedica.com",
    "feld.ly",
    "fema.gov",
    "fevo.me",
    "ff.im",
    "fff.re",
    "fff.to",
    "fff.wf",
    "ffm.bio",
    "ffm.to",
    "ffrf.us",
    "fhurl.com",
    "fifa.fans",
    "filz.fr",
    "fire.to",
    "firsturl.de",
    "fiverr.com",
    "fkd.sale",
    "flapol.com",
    "flfree.link",
    "flic.kr",
    "flip.it",
    "flix.fan",
    "flosports.link",
    "flow.page",
    "flpbd.it",
    "fly2.ws",
    "flyt.it",
    "fmeq.co",
    "fnd.us",
    "fnk.es",
    "fnlink.co",
    "foe.hn",
    "folha.com",
    "folu.me",
    "fon.gs",
    "food.bit.ly",
    "foodtv.com",
    "fori.io",
    "found.ee",
    "fox9.com",
    "foxs.pt",
    "freak.ws",
    "freech.pl",
    "freedomfest.link",
    "freze.it",
    "frontl.ink",
    "fse.tw",
    "fsu.org.au",
    "ft.trib.al",
    "ftb.ca",
    "ftnt.net",
    "fur.ly",
    "fwd4.me",
    "fwwat.ch",
    "fxn.ws",
    "fxstreet.page.link",
    "fyndry.co",
    "g.igg.com",
    "g.town",
    "g00.me",
    "g2ul0.app.link",
    "gado.gs",
    "gal.to",
    "gala.fan",
    "gazik.co",
    "gbf.wiki",
    "gddy.co",
    "gdhq.info",
    "gdl.ink",
    "gdmo.ae",
    "geni.us",
    "getbook.at",
    "gfme.co",
    "gfuel.ly",
    "gfw.global",
    "gg.gg",
    "giphy.com",
    "git.io",
    "github.com",
    "giv.gg",
    "gjw.us",
    "gl.am",
    "glcmsg.com",
    "glfdig.st",
    "glmr.co",
    "glo.bo",
    "gloo.to",
    "gma.abc",
    "gn24.ae",
    "go.9nl.com",
    "go.abc.org",
    "go.aps.org",
    "go.aws",
    "go.azsos.gov",
    "go.bengals.com",
    "go.bloodhorse.com",
    "go.cal.com",
    "go.cb-w.com",
    "go.cbn.com",
    "go.cms.gov",
    "go.coe.int",
    "go.crkd.gg",
    "go.crooked.com",
    "go.ctcapitolreport.com",
    "go.cwseed.com",
    "go.dexani.io",
    "go.dhs.gov",
    "go.ea.com",
    "go.emilyslist.org",
    "go.ey.com",
    "go.france24.com",
    "go.getproton.me",
    "go.gov.sg",
    "go.icann.org",
    "go.iiss.org",
    "go.j23app.com",
    "go.letelegramme.fr",
    "go.lickd.co",
    "go.me",
    "go.medicare.gov",
    "go.msu.edu",
    "go.mythrivestate.com",
    "go.nasa.gov",
    "go.nationaljournal.com",
    "go.nature.com",
    "go.nd.edu",
    "go.nih.gov",
    "go.nowth.is",
    "go.nps.gov",
    "go.nybooks.com",
    "go.one.org",
    "go.pbgrp.link",
    "go.rallyup.com",
    "go.redwine.blue",
    "go.rutgers.edu",
    "go.senate.gov",
    "go.sequence.xyz",
    "go.shr.lc",
    "go.signalcleveland.org",
    "go.slpr.link",
    "go.sorare.com",
    "go.ted.com",
    "go.ucsd.edu",
    "go.umd.edu",
    "go.unc.edu",
    "go.undp.org",
    "go.unimelb.edu.au",
    "go.unl.edu",
    "go.unu.edu",
    "go.usa.gov",
    "go.uscurrency.gov",
    "go.uva.nl",
    "go.uvm.edu",
    "go.uwm.edu",
    "go.vic.gov.au",
    "go.weho.org",
    "go.wisc.edu",
    "go2.me",
    "go2cut.com",
    "go2l.ink",
    "goapp.st",
    "gobulls.co",
    "godaddy.co",
    "goduke.us",
    "gofos.co",
    "gofrogs.co",
    "gofund.me",
    "goget.fund",
    "gohe.at",
    "gohski.es",
    "gojacks.co",
    "golink.icu",
    "goo.gl",
    "goo.gle",
    "goo.lu",
    "goo.su",
    "good.ly",
    "gopvote.co",
    "goshrink.com",
    "got.cr",
    "gov.tw",
    "gowat.ch",
    "goza.gs",
    "gp.org",
    "gpb.org",
    "gph.is",
    "gq.mn",
    "grabify.link",
    "greencb.co",
    "greens.eu",
    "grem.io",
    "gri.ms",
    "grin.to",
    "grm.my",
    "growsf.org",
    "gsk.to",
    "gstrm.io",
    "gties.us",
    "gtly.to",
    "gty.art",
    "gu.gg",
    "gu.tt",
    "guiama.is",
    "gurl.es",
    "gxetfs.com",
    "ha.do",
    "haas.org",
    "hadej.co",
    "haikyu.jp",
    "haje.io",
    "hav.to",
    "hbr.tk",
    "hchsa.info",
    "hec.su",
    "hellotxt.com",
    "herit.ag",
    "heroes.win",
    "hersh.co",
    "hevito.com",
    "hex.io",
    "hey.com",
    "hg.tv",
    "hid.link",
    "hide.my",
    "hill.cm",
    "hjkl.fr",
    "hmmr.buzz",
    "ho.do",
    "hon.org",
    "hookem.at",
    "hops.me",
    "hover.com",
    "hoyo.link",
    "hpe.to",
    "hrc.im",
    "hrc.io",
    "hrc.org",
    "href.in",
    "href.li",
    "hrvd.me",
    "hry.yt",
    "hslda.org",
    "hsu.edu",
    "ht.ly",
    "htcvive.co",
    "htl.li",
    "htmedia.page.link",
    "htxt.it",
    "hubs.la",
    "hubs.li",
    "hubs.ly",
    "hud.gov",
    "huffp.st",
    "hugeurl.com",
    "huit.re",
    "humane.com",
    "hurl.it",
    "hurl.me",
    "hurl.ws",
    "hvr.co",
    "hwd.fyi",
    "hwld.co",
    "hyperx.gg",
    "hyte.co",
    "i-2.co",
    "i.bybit.com",
    "i.cl",
    "i.mtr.cool",
    "i.stv.tv",
    "i99.cz",
    "iakn.org",
    "iatse.co",
    "ibb.co",
    "ibn.fm",
    "icanhaz.com",
    "ice.gov",
    "ice.io",
    "icit.fr",
    "ick.li",
    "icks.ro",
    "icma.org",
    "icont.ac",
    "idek.net",
    "idf.il",
    "idolplus.page.link",
    "idot.click",
    "iea.li",
    "ift.tt",
    "ihe.art",
    "iherb.co",
    "ihr.fm",
    "iiiii.in",
    "ike4.co",
    "iky.fr",
    "ilfat.to",
    "ilix.in",
    "imdb.to",
    "imojo.in",
    "in.rog.gg",
    "ind.pn",
    "indexpress.page.link",
    "indy.gov",
    "info.ms",
    "infob.ae",
    "infy.com",
    "inl.gov",
    "inmn.io",
    "ino.to",
    "inreply.to",
    "intdy.in",
    "intel.ly",
    "interc.pt",
    "intro.co",
    "invst.ly",
    "inx.lv",
    "ipbes.net",
    "ipu.org",
    "ireland.ie",
    "is.gd",
    "iscool.net",
    "isra.li",
    "isw.pub",
    "iterasi.net",
    "itm.im",
    "itnues.net",
    "ito.mx",
    "its.dot.gov",
    "itv.com",
    "ity.im",
    "iu.tn",
    "iwantth.is",
    "iwf.org",
    "iwnm.jp",
    "iwr.link",
    "ix.sk",
    "ixdf.io",
    "j-p.al",
    "j.gs",
    "j.mp",
    "ja.ma",
    "jab.la",
    "jazeera.mv",
    "jdem.cz",
    "jdm.org",
    "jhu.edu",
    "ji.news",
    "jieb.be",
    "jiii.io",
    "jijr.com",
    "jimwv.com",
    "jinxxy.com",
    "jkhy.co",
    "jm.lnk.to",
    "jmp2.net",
    "jmsbnd.be",
    "jo.my",
    "joyza.co",
    "jp22.net",
    "jqw.de",
    "jrnl.ie",
    "jrpm.me",
    "js.tc",
    "jtim.es",
    "just.as",
    "justinwelsh.link",
    "jvp.org",
    "jw.org",
    "jwatch.us",
    "k6.re",
    "ka.do",
    "kabsc.org",
    "kas.pr",
    "kask.us",
    "kbit.co",
    "kck.st",
    "kcmo.gov",
    "kd2.org",
    "keep.re",
    "ken.sc",
    "kenney.nl",
    "kfd.pl",
    "kg.games",
    "kgs.link",
    "kings.com",
    "kings.tn",
    "kiripo.st",
    "kiro7.com",
    "kissa.be",
    "kissinhell.page.link",
    "kite.link",
    "kks.me",
    "kl.am",
    "klck.me",
    "kmph.com",
    "kofc.org",
    "korta.nu",
    "kp.org",
    "kptv.tv",
    "kr3w.de",
    "krat.si",
    "kratsi.cz",
    "krod.cz",
    "krunchd.com",
    "kto.com",
    "kuc.cz",
    "kuku.lu",
    "kutt.it",
    "kwhit.com",
    "kxb.me",
    "l-bc.co",
    "l-k.be",
    "l.actu.fr",
    "l.bfmtv.com",
    "l.capital.fr",
    "l.challenges.fr",
    "l.cleveland.com",
    "l.ead.me",
    "l.francebleu.fr",
    "l.franceculture.fr",
    "l.franceinter.fr",
    "l.francemusique.fr",
    "l.francetvinfo.fr",
    "l.geo.fr",
    "l.gg",
    "l.kphx.net",
    "l.leparisien.fr",
    "l.ly",
    "l.mediapart.fr",
    "l.nj.com",
    "l.ntdtv.fr",
    "l.prageru.com",
    "l.rmc.fr",
    "l.rmcsport.fr",
    "l.sudouest.fr",
    "l.syracuse.com",
    "l.tf1info.fr",
    "l.varmatin.com",
    "l1nk.dev",
    "l1nq.com",
    "l8r.it",
    "l9.fr",
    "lacity.gov",
    "laist.com",
    "lap.org",
    "lapl.me",
    "lat.ms",
    "latika.me",
    "latugo.com",
    "laud.org",
    "lc-s.co",
    "lc.cx",
    "lcut.in",
    "learn.az",
    "lemde.fr",
    "letop10.",
    "lex18.com",
    "lgblu.es",
    "lght.ly",
    "li.sten.to",
    "libero.it",
    "libhill.co",
    "libro.fm",
    "librti.com",
    "lick.my",
    "lien.li",
    "lien.pl",
    "ligm.in",
    "lihi.cc",
    "liip.to",
    "liltext.com",
    "lin.cr",
    "lin.ee",
    "lin.io",
    "link.ag.fan",
    "link.do",
    "link.tl",
    "link.yle.fi",
    "link3.to",
    "linkbee.com",
    "linkbun.ch",
    "linkco.re",
    "linke.bid",
    "linkmix.co",
    "linkn.co",
    "linktw.in",
    "lit.link",
    "liurl.cn",
    "llk.dk",
    "llu.ch",
    "lmy.de",
    "ln-s.net",
    "ln-s.ru",
    "ln.run",
    "lnk.co",
    "lnk.direct",
    "lnk.gd",
    "lnk.in",
    "lnk.ly",
    "lnk.sk",
    "lnk.to",
    "lnkd.in",
    "lnked.in",
    "lnkfi.re",
    "lnkr.fm",
    "lnks.fr",
    "lnky.fr",
    "lnnk.in",
    "lnp.sn",
    "lnv.gy",
    "loc.gov",
    "lokeyz.com",
    "lolthis.me",
    "loom.ly",
    "loopt.us",
    "lp.ca",
    "lp25.fr",
    "lpc.ca",
    "lru.jp",
    "lseg.group",
    "lsn.click",
    "lsul.su",
    "lsvp.com",
    "lt.tl",
    "lttr.ai",
    "lu.ma",
    "lurk.ly",
    "lurl.no",
    "lvli.li",
    "lvvk.com",
    "lwvsf.org",
    "lynk.my",
    "m1p.fr",
    "m3mi.com",
    "magma.com",
    "maine.gov",
    "make.my",
    "makular.co",
    "manc.it",
    "masn.me",
    "mass.gov",
    "mavrev.com",
    "mayo.rs",
    "mayocl.in",
    "mbr.ae",
    "mbr.mv",
    "mby.me",
    "mcaf.ee",
    "mcafee.ly",
    "mck.co",
    "mcsc.ca",
    "mdl29.net",
    "mdn.tv",
    "mdr.gg",
    "meaa.io",
    "mecknc.gov",
    "media.tv",
    "melzor.com",
    "metamark.net",
    "meteor.link",
    "metrol.ink",
    "mfa.gov.mn",
    "mgshow.link",
    "mi.gov",
    "mic.fr",
    "microb.io",
    "midd.me",
    "migre.me",
    "mil.wa.gov",
    "mile.io",
    "minilien.com",
    "miniurl.be",
    "miniurl.com",
    "mintiphone.page.link",
    "minu.me",
    "minurl.fr",
    "mirrativ.page.link",
    "mitsmr.com",
    "mk.my",
    "mla.bs",
    "mlgw.com",
    "mlion.info",
    "mm.vg",
    "mm4a.org",
    "mmt.fans",
    "mnhs.info",
    "mnoy.es",
    "mnth.ly",
    "mnvkn.gs",
    "mobo.com",
    "moc.ac",
    "modizy.co",
    "mol.co.jp",
    "molf.ar",
    "moneyyy.me",
    "moourl.com",
    "more.sh",
    "mp.gg",
    "mptf.com",
    "mrkup.org",
    "mru.io",
    "mrw.so",
    "ms.spr.ly",
    "msfa.me",
    "msft.it",
    "msi.gm",
    "mtc.one",
    "mts.lnk.to",
    "mtw.so",
    "mubeat.page.link",
    "muji.lu",
    "multco.us",
    "mut.lu",
    "muz.so",
    "mville.io",
    "mwdh2o.com",
    "my-tf.co",
    "mybook.to",
    "mybs.in",
    "mypgc.us",
    "mypolice.page.link",
    "mysapl.org",
    "mysp.ac",
    "myumi.ch",
    "myurl.in",
    "n.pr",
    "n9.cl",
    "nacon.me",
    "nami.org",
    "nas.cr",
    "nature.ly",
    "naver.me",
    "nbc4dc.com",
    "nbc4i.co",
    "nbcbay.com",
    "nbcchi.com",
    "nbcct.co",
    "nbcnews.to",
    "ncdhhs.gov",
    "ndss.org",
    "ndtc.me",
    "ne.gop",
    "ne1.net",
    "nea.org",
    "nearwez.com",
    "nej.md",
    "nesn.com",
    "net.ms",
    "net46.net",
    "news.ae",
    "news.sd.gov",
    "ngl.link",
    "ngpa.us",
    "nh-24.org",
    "nhexit.us",
    "nhs.uk",
    "nhwin.org",
    "niaa.gov.au",
    "nico.ms",
    "nicou.ch",
    "nig.gr",
    "ninten.do",
    "niny.io",
    "nixle.us",
    "njdems.org",
    "njx.me",
    "nm-4.com",
    "nmfc.me",
    "nn.nf",
    "noaa.gov",
    "noice.com",
    "nokia.ly",
    "nomis.cc",
    "nos.nl",
    "notlong.com",
    "nov.io",
    "npi.li",
    "nps.gov",
    "npwf.info",
    "nq.st",
    "nrt.cc",
    "nsfw.in",
    "nswpr.com",
    "nt-z.ro",
    "ntiny.link",
    "ntlms.org",
    "ntn.so",
    "ntry.at",
    "nurl.ng",
    "nuwav.es",
    "nuzufy.com",
    "nvda.ws",
    "nwsdy.li",
    "nxy.in",
    "nycacc.app",
    "nycacc.org",
    "nyccrr.org",
    "nyer.cm",
    "nyfed.org",
    "nygnt.co",
    "nyj.social",
    "nyrange.rs",
    "nysut.cc",
    "nyti.ms",
    "o-x.fr",
    "obi41.nl",
    "oceana.ly",
    "ocfl.net",
    "ocv.im",
    "odesza.co",
    "oe.cd",
    "oeis.org",
    "of.tv",
    "ofa.bo",
    "ogenus.co",
    "ohio.org",
    "okla.st",
    "okok.fr",
    "okt.to",
    "olj.me",
    "olplus.fr",
    "oly.ch",
    "om.ly",
    "oma.io",
    "on.barrons.com",
    "on.bcg.com",
    "on.cfr.org",
    "on.csis.or.id",
    "on.doi.gov",
    "on.forbes.com",
    "on.ft.com",
    "on.gallup.com",
    "on.hwnair.com",
    "on.inc.com",
    "on.iongroup.com",
    "on.kgun9.com",
    "on.knicks.com",
    "on.louisvuitton.com",
    "on.mktw.net",
    "on.msnbc.com",
    "on.natgeo.com",
    "on.nba.com",
    "on.nbc10.com",
    "on.nbc6.com",
    "on.nbc7.com",
    "on.nbcboston.com",
    "on.nbcdfw.com",
    "on.ncaa.com",
    "on.necn.com",
    "on.ny.gov",
    "on.nyc.gov",
    "on.nypl.org",
    "on.pru",
    "on.rt.com",
    "on.sce.com",
    "on.sfpl.org",
    "on.sny.tv",
    "on.soundcloud.com",
    "on.tmj4.com",
    "on.today.com",
    "on.unesco.org",
    "on.wbaltv.com",
    "on.wcvb.com",
    "on.wews.com",
    "on.wowt6.com",
    "on.wsj.com",
    "once.ly",
    "onelink.me",
    "onelink.to",
    "onx.la",
    "opn.to",
    "opr.as",
    "or-f.org",
    "oran.ge",
    "orgcns.org",
    "orlo.uk",
    "osat.io",
    "osf.io",
    "ou.af",
    "ou.edu",
    "ou.gd",
    "oua.be",
    "ouo.io",
    "ow.ly",
    "owl.li",
    "oxf.am",
    "oxford.ly",
    "oyushold.com",
    "p.ppfa.org",
    "p.pw",
    "p1r.es",
    "p2a.co",
    "p4k.in",
    "pacast.com",
    "padi.co",
    "pal.watch",
    "para.pt",
    "parfgc.to",
    "parkrun.me",
    "parky.tv",
    "past.is",
    "paypal.com",
    "pca.st",
    "pccc.me",
    "pck.tv",
    "pckrs.com",
    "pcnb.ca",
    "pcrm.org",
    "pd.am",
    "pdh.co",
    "peacocktv.smart.link",
    "pens.pe",
    "peoplem.ag",
    "perma.cc",
    "peta.asia",
    "peta.vg",
    "petauk.org",
    "pew.org",
    "pewrsr.ch",
    "pff.com",
    "pgat.us",
    "pge.sx",
    "pghdsa.org",
    "ph.au",
    "ph.dog",
    "ph.ly",
    "pheaa.org",
    "phila.gov",
    "philos.link",
    "phmc.pa.gov",
    "pic.gd",
    "pich.in",
    "pillar.io",
    "pim.co",
    "pin.it",
    "pin.st",
    "ping.fm",
    "pipr.es",
    "pitt.ly",
    "piurl.com",
    "pixb.ly",
    "pj.lnk.to",
    "play.st",
    "plos.io",
    "plots.fr",
    "plu.sh",
    "ply.by",
    "pm.wu.cz",
    "pmc19.com",
    "pnqk.me",
    "pnsne.ws",
    "pnt.me",
    "po.do",
    "po.st",
    "pod.link",
    "politi.co",
    "pomagam.pl",
    "pomoc.pl",
    "poprl.com",
    "post.ly",
    "posted.at",
    "posty.pe",
    "ppfr.it",
    "ppst.me",
    "ppt.cc",
    "ppt.li",
    "prejit.cz",
    "preml.ge",
    "pretty.link",
    "prez.ly",
    "prksn.uk",
    "prn.to",
    "profile.to",
    "propub.li",
    "pros.ee",
    "prosyn.org",
    "prov.ly",
    "prz.io",
    "pse.is",
    "pspca.org",
    "ptab.it",
    "ptm.ro",
    "ptnani.org",
    "pubmed.gov",
    "pw-ne.ws",
    "pw2.ro",
    "pwc.to",
    "pwne.ws",
    "pxlme.me",
    "py6.ru",
    "pzdls.co",
    "q.gs",
    "qbn.ru",
    "qicute.com",
    "qkt.io",
    "qlnk.net",
    "qqc.co",
    "qqcarwa.sh",
    "qr.ae",
    "qr.net",
    "qr.paps.jp",
    "qr2.info",
    "qr2.mobi",
    "qrbridge.co",
    "qrbridge.me",
    "qrcd.org",
    "qrco.de",
    "qrly.me",
    "qrs.ly",
    "qrtag.fr",
    "qrto.co",
    "qrto.info",
    "qrto.mobi",
    "qudsnen.co",
    "quip-art.com",
    "qxp.cz",
    "qxp.sk",
    "rac.org",
    "ragncaj.co",
    "rb.gy",
    "rb6.co",
    "rb6.me",
    "rc.surf",
    "rcc-uk.org",
    "rcht.link",
    "rcknr.io",
    "rdbl.co",
    "rdcrss.org",
    "rdcu.be",
    "rdg.ac",
    "rdo.to",
    "rdz.me",
    "read.bi",
    "read.ht",
    "reason.pub",
    "rebelne.ws",
    "rebrand.ly",
    "rebs.us",
    "red24.us",
    "redd.it",
    "redf.in",
    "redir.ec",
    "redir.fr",
    "redirx.com",
    "reds.com",
    "redu.it",
    "ref.so",
    "refini.tv",
    "reise.lc",
    "relai.me",
    "relink.fr",
    "resist.bot",
    "reurl.cc",
    "reut.rs",
    "rfi.my",
    "rfkhro.org",
    "rfkjr.co",
    "ri.ms",
    "rickroll.it",
    "right.bz",
    "riot.com",
    "rip.city",
    "riz.cz",
    "riz.gd",
    "rmgo.org",
    "rnctext.co",
    "road.cc",
    "roadand.tk",
    "rockcha.lk",
    "rod.gs",
    "roflc.at",
    "rol.st",
    "rpt.rs",
    "rsg.ms",
    "rsm.buzz",
    "rsm.us",
    "rsmonkey.com",
    "rsn.pub",
    "rt.se",
    "rt.tc",
    "rtdden.co",
    "rtrj.info",
    "ru.ly",
    "rubyurl.com",
    "run-go.fr",
    "rvns.co",
    "rwrd.io",
    "s-url.fr",
    "s.frk.com",
    "s.goodsmile.link",
    "s.hbr.org",
    "s.id",
    "s.iq.com",
    "s.ksc.de",
    "s.si.edu",
    "s.weibo.com",
    "s7y.us",
    "saacs.info",
    "safe.mn",
    "sagyap.tk",
    "sail.ink",
    "sampur.se",
    "satpa.pe",
    "sbcfire.co",
    "sbjsbd.biz",
    "sbpr.cc",
    "sc.mp",
    "sce.com",
    "sched.co",
    "schrts.co",
    "scim.ag",
    "scl.io",
    "sco.lt",
    "scq.io",
    "scsu.mn",
    "sdcl.org",
    "sdfc.me",
    "sdge.com",
    "sdgs.un.org",
    "sdu.sk",
    "seashe.ph",
    "see.nf",
    "seeme.at",
    "seemless.link",
    "segue.se",
    "seiu.co",
    "selar.co",
    "selday.org",
    "sen.gov",
    "sen.lu",
    "sentinelshare.page.link",
    "sf.gov",
    "sfmta.com",
    "sfnm.co",
    "sforce.co",
    "sge.st",
    "sgq.io",
    "sh.pturl.at",
    "sh.st",
    "shaam.org",
    "shar.as",
    "shar.es",
    "sharein.com",
    "sharetabs.com",
    "shark.dog",
    "shha.re",
    "shor.by",
    "shorl.com",
    "short.cc",
    "short.cm",
    "short.ie",
    "short.ly",
    "short.nr",
    "short.pk",
    "short.to",
    "shortcm.li",
    "shorte.st",
    "shorter.me",
    "shortlinks.co.uk",
    "shortna.me",
    "shorturl.at",
    "shorturl.com",
    "shoturl.us",
    "shre.ink",
    "shrinkee.com",
    "shrinkify.com",
    "shrinkster.com",
    "shrinkurl.in",
    "shrt.in",
    "shrt.st",
    "shrtco.de",
    "shrten.com",
    "shrtm.nu",
    "shrunkin.com",
    "shw.me",
    "shy.si",
    "si.edu",
    "sicax.net",
    "signal.org",
    "simurl.com",
    "sina.lt",
    "sive.rs",
    "sk.gy",
    "skep.us",
    "skoll.wf",
    "skr.sk",
    "skroc.pl",
    "sku.su",
    "skyctc.cc",
    "slam.ly",
    "smar.lnk.to",
    "smardex.io",
    "smartbb.io",
    "smarturl.it",
    "smi.lnk.to",
    "sml.snl.no",
    "smll.co",
    "smpl.is",
    "smrtr.io",
    "smud.org",
    "smvmt.link",
    "sn.im",
    "sn.vc",
    "sndnc.org",
    "sndrs.com",
    "snhr.org",
    "snip.ly",
    "snipr.com",
    "snipurl.com",
    "snsw.us",
    "snurl.com",
    "soc.cr",
    "soccer.com",
    "socy.es",
    "solo.to",
    "song.link",
    "soo.gd",
    "sovrn.co",
    "sp2.ro",
    "space3.gg",
    "spacex.com",
    "specne.ws",
    "spedr.com",
    "spkl.io",
    "spklr.io",
    "splk.it",
    "splr.io",
    "splt.cc",
    "spn.sr",
    "spot.fund",
    "spoti.fi",
    "spotify.link",
    "spr.ly",
    "sprtsnt.ca",
    "sptfy.com",
    "spti.fi",
    "sq6.ru",
    "sqex.link",
    "sqex.to",
    "sqrl.it",
    "sqrx.io",
    "srtz.co",
    "ss.st",
    "ssl.gs",
    "sspca.org",
    "st.news",
    "starturl.com",
    "str.sg",
    "strib.gift",
    "strw.rs",
    "sturly.com",
    "stv.fm",
    "su.pr",
    "subs.ft.com",
    "suibit.xyz",
    "sun.mv",
    "suns.com",
    "suo.im",
    "surl.li",
    "surl.me",
    "sux.cz",
    "swa.is",
    "swan.ac",
    "swll.to",
    "sxm.app.link",
    "sy.app",
    "sy.pe",
    "t.cn",
    "t.co",
    "t.gsu.edu",
    "t.ly",
    "t.ted.com",
    "t.uga.edu",
    "t13.cl",
    "t1p.de",
    "t2m.io",
    "ta.gd",
    "tabzi.com",
    "tagd.co",
    "tanks.ly",
    "tapthe.link",
    "tau.pe",
    "tcf.org",
    "tcl.com",
    "tcmb.tl",
    "tcrn.ch",
    "tdjt.cz",
    "tdy.lol",
    "telezo.co",
    "ter.li",
    "tfd.bz",
    "tfn.org",
    "tgam.ca",
    "tgfb.com",
    "theatln.tc",
    "theaus.in",
    "theb.fi",
    "thebase.page.link",
    "thefp.pub",
    "thegspl.co",
    "theins.ru",
    "thelc.ms",
    "theog.io",
    "thera.ms",
    "thesa.us",
    "thesikh.co",
    "thesun.uk",
    "thetim.es",
    "thinfi.com",
    "thr.cm",
    "thrdl.es",
    "throne.com",
    "throne.me",
    "ti.me",
    "tibr.in",
    "tickpick.page.link",
    "tidal.link",
    "tidd.ly",
    "tier.bot",
    "tighturl.com",
    "tiktok.com",
    "tim.blog",
    "tin.li",
    "tini.cc",
    "tiny.cc",
    "tiny.ie",
    "tiny.io",
    "tiny.lt",
    "tiny.ms",
    "tiny.pl",
    "tiny123.com",
    "tinyarro.ws",
    "tinyl.io",
    "tinypic.com",
    "tinytw.it",
    "tinyuri.ca",
    "tinyurl.com",
    "tinyurl.hu",
    "tinyvid.io",
    "tix.to",
    "tixsu.com",
    "tl.gd",
    "tldr.sk",
    "tldrify.com",
    "tllg.net",
    "tlmdo.co",
    "tmz.me",
    "tn.ai",
    "tnij.org",
    "tnns.ch",
    "tnusa.com",
    "tnvge.co",
    "tny.cz",
    "tny.im",
    "to.ly",
    "to.pbs.org",
    "to.philips",
    "to.welt.de",
    "to.wttw.com",
    "to8.cc",
    "tobtr.com",
    "togoto.us",
    "tohle.de",
    "toi.in",
    "torto.se",
    "tox.cx",
    "toyhou.se",
    "toyota.us",
    "tpc.io",
    "tpmr.com",
    "tps.to",
    "tpt.org",
    "tpu.me",
    "tquiet.us",
    "tr.ee",
    "tr.im",
    "tr.my",
    "tr5.in",
    "traceurl.com",
    "trakqr.com",
    "trck.me",
    "trele.co",
    "tri.ps",
    "trib.al",
    "trib.it",
    "trick.ly",
    "trkr.ws",
    "trst.in",
    "trt.world",
    "trump.com",
    "trunc.it",
    "truthblood.page.link",
    "ts.la",
    "tst.link",
    "tuppu.net",
    "turo.us",
    "tv.nrk.no",
    "twet.fr",
    "twi.im",
    "twirl.at",
    "twit.ac",
    "twitch.com",
    "twitterpan.com",
    "twitthis.com",
    "twiturl.de",
    "twlr.me",
    "tws.io",
    "twtch.tv",
    "twter.co",
    "twtr.in",
    "twtr.to",
    "twurl.cc",
    "twurl.nl",
    "tx.ag",
    "txfb.us",
    "txst.us",
    "u.afp.com",
    "u.mavrev.com",
    "u.nu",
    "u.to",
    "u6e.de",
    "ub0.cc",
    "ubi.li",
    "uby.es",
    "ucam.me",
    "ucberk.li",
    "ucla.in",
    "ucm.edu",
    "ucsfh.org",
    "ufc.ac",
    "ufile.io",
    "ufw.org",
    "ug.cz",
    "ulmt.in",
    "un.org",
    "unctad.org",
    "unes.co",
    "unf.pa",
    "unh.cr",
    "uni.cf",
    "unicef.uk",
    "unimelb.me",
    "unite.us",
    "unlc.us",
    "uofl.me",
    "uoft.me",
    "uol.page.link",
    "updating.me",
    "upzat.com",
    "uqr.me",
    "uqr.to",
    "uquiz.com",
    "ur1.ca",
    "urb.tf",
    "urbn.is",
    "url.co.uk",
    "url.ie",
    "url.kr",
    "url2.fr",
    "url2it.com",
    "url4.eu",
    "url5.org",
    "urlao.com",
    "urlbrief.com",
    "urlcover.com",
    "urlcut.com",
    "urlenco.de",
    "urlgeni.us",
    "urlhawk.com",
    "urlin.it",
    "urlis.net",
    "urlkiss.com",
    "urlkr.com",
    "urlot.com",
    "urlpire.com",
    "urlr.me",
    "urls.fr",
    "urlshortbot.com",
    "urlx.ie",
    "urlx.org",
    "urly.it",
    "urlz.fr",
    "urlzen.com",
    "urnic.com",
    "urub.us",
    "usaid.gov",
    "usaid.link",
    "usat.ly",
    "uscca.co",
    "ussoc.cr",
    "usw.to",
    "utas.au",
    "utd.mn",
    "utepn.ws",
    "utex.as",
    "utfg.sk",
    "utm.io",
    "uwu.org.au",
    "uxuy.com",
    "v-os.net",
    "v.aa.com.tr",
    "v.gd",
    "v.ht",
    "v.ly",
    "v5.gd",
    "vaaa.fr",
    "vaboee.com",
    "valv.im",
    "vaneck.com",
    "varsitytv.link",
    "vast.link",
    "vaza.me",
    "vbly.us",
    "vcas.us",
    "vcha.lnk.to",
    "vd55.com",
    "ver.20m.es",
    "ver.abc.es",
    "ver.ec",
    "ver.tw",
    "verd.in",
    "vfl.ink",
    "vgen.com",
    "vgk.io",
    "vgn.me",
    "vi.sa",
    "vimeo.com",
    "vintaza.co",
    "virg.in",
    "virl.com",
    "vist.ly",
    "vk.cc",
    "vl.am",
    "vntyfr.com",
    "vo.la",
    "vogue.cm",
    "vosd.org",
    "vote24.io",
    "votebe.at",
    "vov.li",
    "vraise.org",
    "vro.in",
    "vsf.org.au",
    "vsfinch.es",
    "vsll.eu",
    "vt802.us",
    "vtr.vc",
    "vu.fr",
    "vur.me",
    "vv.vg",
    "vytext.com",
    "vz.to",
    "vzt.me",
    "w.lne.es",
    "w.lpnt.fr",
    "w.wiki",
    "w1p.fr",
    "w3t.org",
    "wa.link",
    "waa.ai",
    "walmrt.us",
    "wamu.fm",
    "wandtv.com",
    "wapo.st",
    "wapurl.co.uk",
    "washex.am",
    "wb.md",
    "wb1.eu",
    "wb2.biz",
    "wbur.fm",
    "weall.vote",
    "weareuv.us",
    "weartv.com",
    "web99.eu",
    "webwecan.net",
    "wed.li",
    "wef.ch",
    "wellc.me",
    "wfts.tv",
    "whr.tn",
    "wideo.fr",
    "win.gs",
    "wintx.co",
    "wipi.es",
    "wish.org",
    "wix.to",
    "wjactv.com",
    "wkf.ms",
    "wlu.ca",
    "wmata.com",
    "wmg.lnk.to",
    "wmo.int",
    "wn.nr",
    "wntdco.mx",
    "wo.ws",
    "woobox.com",
    "wosu.pm",
    "wp.me",
    "wpu.ir",
    "wpxi.tv",
    "wq.lt",
    "wr.al",
    "wrfr.me",
    "wsac.city",
    "wsdot.com",
    "wshe.es",
    "wsl.tv",
    "wsrjb.org",
    "wtc.la",
    "wtfthis.me",
    "wu.cz",
    "ww7.fr",
    "wwf.to",
    "wwp.news",
    "wwy.me",
    "wxyz.page",
    "x.co.in",
    "x.co.za",
    "x.co",
    "x.ea.com",
    "x.gd",
    "x.nu",
    "x.se",
    "x10.mx",
    "x2c.eu",
    "x2c.eumx",
    "xaddr.com",
    "xav.cc",
    "xbx.lv",
    "xc.pl",
    "xeeurl.com",
    "xgd.in",
    "xhtxs.cn",
    "xib.me",
    "xl8.eu",
    "xn--ki8h.ws",
    "xn--kn8h.to",
    "xn--vi8hl0c.ws",
    "xoe.cz",
    "xr.com",
    "xrl.in",
    "xrl.us",
    "xsm.link",
    "xt3.me",
    "xua.me",
    "xub.me",
    "xurl.jp",
    "xurls.co",
    "xzb.cc",
    "y2u.be",
    "y8.d4t.cn",
    "yagoa.fr",
    "yagoa.me",
    "yahoo.jp",
    "yau.sh",
    "yaythis.me",
    "yeca.eu",
    "yect.com",
    "yep.it",
    "yerl.org",
    "yfrog.com",
    "yhoo.it",
    "ymlp.com",
    "yogh.me",
    "yon.ir",
    "youfap.me",
    "yourcodeiss.com",
    "ysear.ch",
    "ytbe.co",
    "ytube.io",
    "yweb.com",
    "yyv.co",
    "z.umn.edu",
    "z9.fr",
    "zalyst.com",
    "zapit.nu",
    "zc.vg",
    "zdcs.link",
    "zdsk.co",
    "zeek.ir",
    "zi.ma",
    "zi.pe",
    "zii.bz",
    "zip.net",
    "zip.pe",
    "zipansion.com",
    "zipmyurl.com",
    "zkr.cz",
    "zkrat.me",
    "zkrt.cz",
    "zoekje.me",
    "zoodl.com",
    "zoom.us",
    "zpag.es",
    "zsms.net",
    "zst.io",
    "zti.me",
    "zurl.co",
    "zxq.net",
    "zyva.org",
    "zz.gd",
    "zzb.bz",
}


REGEXP_SHORT_URL_EXCLUSIONS = re.compile(
    r"""
    ^archive\.[a-z]{2,3}|^www\.|
    vimeo\.com|
    msnbc\.com|
    throne\.com|
    rumble\.com|
    linktr\.ee|
    l\.facebook.com|
    venmo\.com|
    xkcd\.com|
    github\.com|
    grist\.org|
    imax\.com|
    (www\.)?x\.com
    $""",
    re.IGNORECASE,
)

#  'aft.org',
#  'amu.tv',
#  'cjr.org',
#  'digg.com',
#  'fcc.gov',
#  'frc.org',
#  'ft.com',
#  'glaad.org',
#  'gov.uk',
#  'imgur.com',
#  'inl.gov',
#  'itep.org',
#  'itv.com',
#  'laist.com',
#  'nea.org',
#  'nps.gov',
#  'openai.com',
#  'osf.io',
#  'slate.com',
#  'wjactv.com',
# 'aclj.org',


BIO_LINK_DOMAINS = {
    "beacons.ai",
    "linktr.ee",
    "about.me",
    "many.bio",
    "msha.ke",
    "lnk.bio",
    "tap.bio",
    "bio.site",
    "linke.ro",
    "bio.link",
    "pca.st",
}
