from dataclasses import dataclass
from typing import Optional

from bfcl_eval.model_handler.api_inference.claude import ClaudeHandler
from bfcl_eval.model_handler.api_inference.cohere import CohereHandler
from bfcl_eval.model_handler.api_inference.deepseek import DeepSeekAPIHandler
from bfcl_eval.model_handler.api_inference.dm_cito import DMCitoHandler
from bfcl_eval.model_handler.api_inference.fireworks import FireworksHandler
from bfcl_eval.model_handler.api_inference.functionary import FunctionaryHandler
from bfcl_eval.model_handler.api_inference.gemini import GeminiHandler
from bfcl_eval.model_handler.api_inference.glm import GLMAPIHandler
from bfcl_eval.model_handler.api_inference.gogoagent import GoGoAgentHandler
from bfcl_eval.model_handler.api_inference.gorilla import GorillaHandler
from bfcl_eval.model_handler.api_inference.grok import GrokHandler
from bfcl_eval.model_handler.api_inference.kimi import KimiHandler
from bfcl_eval.model_handler.api_inference.ling import LingAPIHandler
from bfcl_eval.model_handler.api_inference.mining import MiningHandler
from bfcl_eval.model_handler.api_inference.mistral import MistralHandler
from bfcl_eval.model_handler.api_inference.nemotron import NemotronHandler
from bfcl_eval.model_handler.api_inference.nova import NovaHandler
from bfcl_eval.model_handler.api_inference.novita import NovitaHandler
from bfcl_eval.model_handler.api_inference.nvidia import NvidiaHandler
from bfcl_eval.model_handler.api_inference.openai_completion import (
    OpenAICompletionsHandler,
)
from bfcl_eval.model_handler.api_inference.openai_response import OpenAIResponsesHandler
from bfcl_eval.model_handler.api_inference.qwen import (
    QwenAgentNoThinkHandler,
    QwenAgentThinkHandler,
    QwenAPIHandler,
)
from bfcl_eval.model_handler.api_inference.writer import WriterHandler
from bfcl_eval.model_handler.local_inference.arch import ArchHandler
from bfcl_eval.model_handler.local_inference.bielik import BielikHandler
from bfcl_eval.model_handler.local_inference.bitagent import BitAgentHandler
from bfcl_eval.model_handler.local_inference.deepseek_reasoning import (
    DeepseekReasoningHandler,
)
from bfcl_eval.model_handler.local_inference.falcon_fc import Falcon3FCHandler
from bfcl_eval.model_handler.local_inference.gemma import GemmaHandler
from bfcl_eval.model_handler.local_inference.glm import GLMHandler
from bfcl_eval.model_handler.local_inference.granite import (
    GraniteFunctionCallingHandler,
)
from bfcl_eval.model_handler.local_inference.granite_3 import Granite3FCHandler
from bfcl_eval.model_handler.local_inference.hammer import HammerHandler
from bfcl_eval.model_handler.local_inference.llama import LlamaHandler
from bfcl_eval.model_handler.local_inference.llama_3_1 import LlamaHandler_3_1
from bfcl_eval.model_handler.local_inference.minicpm import MiniCPMHandler
from bfcl_eval.model_handler.local_inference.minicpm_fc import MiniCPMFCHandler
from bfcl_eval.model_handler.local_inference.mistral_fc import MistralFCHandler
from bfcl_eval.model_handler.local_inference.phi import PhiHandler
from bfcl_eval.model_handler.local_inference.phi_fc import PhiFCHandler
from bfcl_eval.model_handler.local_inference.quick_testing_oss import (
    QuickTestingOSSHandler,
)
from bfcl_eval.model_handler.local_inference.qwen import QwenHandler
from bfcl_eval.model_handler.local_inference.qwen_fc import QwenFCHandler
from bfcl_eval.model_handler.local_inference.salesforce_llama import (
    SalesforceLlamaHandler,
)
from bfcl_eval.model_handler.local_inference.salesforce_qwen import (
    SalesforceQwenHandler,
)
from bfcl_eval.model_handler.local_inference.think_agent import ThinkAgentHandler

# -----------------------------------------------------------------------------
# A mapping of model identifiers to their respective model configurations.
# Each key corresponds to the model id passed to the `--model` argument
# in both generation and evaluation commands.
# Make sure to update the `supported_models.py` file as well when updating this map.
# -----------------------------------------------------------------------------


@dataclass
class ModelConfig:
    """
    Model configuration class for storing model metadata and settings. 

    Attributes:
        model_name (str): Name of the model as used in the vendor API or on Hugging Face (may not be unique).
        display_name (str): Model name as it should appear on the leaderboard.
        url (str): Reference URL for the model or hosting service.
        org (str): Organization providing the model.
        license (str): License under which the model is released.
        model_handler (str): Handler name for invoking the model.
        input_price (Optional[float]): USD per million input tokens (None for open source models).
        output_price (Optional[float]): USD per million output tokens (None for open source models).
        is_fc_model (bool): True if this model is used in Function-Calling mode, otherwise False for Prompt-based mode.
        underscore_to_dot (bool): True if model does not support '.' in function names, in which case we will replace '.' with '_'. Currently this only matters for checker.  TODO: We should let the tool compilation step also take this into account.

    """

    model_name: str
    display_name: str
    url: str
    org: str
    license: str

    model_handler: str

    # Prices are in USD per million tokens; open source models have None
    input_price: Optional[float] = None
    output_price: Optional[float] = None

    # True if the model is in function-calling mode, False if in prompt mode
    is_fc_model: bool = True

    # True if this model does not allow '.' in function names
    underscore_to_dot: bool = False


# Inference through API calls
api_inference_model_map = {
    "gorilla-openfunctions-v2": ModelConfig(
        model_name="gorilla-openfunctions-v2",
        display_name="Gorilla-OpenFunctions-v2 (FC)",
        url="https://gorilla.cs.berkeley.edu/blogs/7_open_functions_v2.html",
        org="Gorilla LLM",
        license="Apache 2.0",
        model_handler=GorillaHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=False,
    ),
    "DeepSeek-V3.2-Exp": ModelConfig(
        model_name="deepseek-chat",
        display_name="DeepSeek-V3.2-Exp (Prompt)",
        url="https://api-docs.deepseek.com/news/news250528",
        org="DeepSeek",
        license="MIT",
        model_handler=DeepSeekAPIHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "DeepSeek-V3.2-Exp-FC": ModelConfig(
        model_name="deepseek-chat",
        display_name="DeepSeek-V3.2-Exp (FC)",
        url="https://api-docs.deepseek.com/news/news250528",
        org="DeepSeek",
        license="MIT",
        model_handler=DeepSeekAPIHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "DeepSeek-V3.2-Exp-thinking": ModelConfig(
        model_name="deepseek-reasoner",
        display_name="DeepSeek-V3.2-Exp (Prompt + Thinking)",
        url="https://api-docs.deepseek.com/news/news250528",
        org="DeepSeek",
        license="MIT",
        model_handler=DeepSeekAPIHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "gpt-5-2025-08-07-FC": ModelConfig(
        model_name="gpt-5-2025-08-07",
        display_name="GPT-5-2025-08-07 (FC)",
        url="https://openai.com/index/introducing-gpt-5/",
        org="OpenAI",
        license="Proprietary",
        model_handler=OpenAIResponsesHandler,
        input_price=1.25,
        output_price=10,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "gpt-5-2025-08-07": ModelConfig(
        model_name="gpt-5-2025-08-07",
        display_name="GPT-5-2025-08-07 (Prompt)",
        url="https://openai.com/index/introducing-gpt-5/",
        org="OpenAI",
        license="Proprietary",
        model_handler=OpenAIResponsesHandler,
        input_price=1.25,
        output_price=10,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "gpt-5-mini-2025-08-07-FC": ModelConfig(
        model_name="gpt-5-mini-2025-08-07",
        display_name="GPT-5-mini-2025-08-07 (FC)",
        url="https://openai.com/index/introducing-gpt-5/",
        org="OpenAI",
        license="Proprietary",
        model_handler=OpenAIResponsesHandler,
        input_price=0.25,
        output_price=2,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "gpt-5-mini-2025-08-07": ModelConfig(
        model_name="gpt-5-mini-2025-08-07",
        display_name="GPT-5-mini-2025-08-07 (Prompt)",
        url="https://openai.com/index/introducing-gpt-5/",
        org="OpenAI",
        license="Proprietary",
        model_handler=OpenAIResponsesHandler,
        input_price=0.25,
        output_price=2,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "gpt-5-nano-2025-08-07-FC": ModelConfig(
        model_name="gpt-5-nano-2025-08-07",
        display_name="GPT-5-nano-2025-08-07 (FC)",
        url="https://openai.com/index/introducing-gpt-5/",
        org="OpenAI",
        license="Proprietary",
        model_handler=OpenAIResponsesHandler,
        input_price=0.05,
        output_price=0.4,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "gpt-5-nano-2025-08-07": ModelConfig(
        model_name="gpt-5-nano-2025-08-07",
        display_name="GPT-5-nano-2025-08-07 (Prompt)",
        url="https://openai.com/index/introducing-gpt-5/",
        org="OpenAI",
        license="Proprietary",
        model_handler=OpenAIResponsesHandler,
        input_price=0.05,
        output_price=0.4,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "gpt-4.1-2025-04-14-FC": ModelConfig(
        model_name="gpt-4.1-2025-04-14",
        display_name="GPT-4.1-2025-04-14 (FC)",
        url="https://openai.com/index/gpt-4-1/",
        org="OpenAI",
        license="Proprietary",
        model_handler=OpenAIResponsesHandler,
        input_price=2,
        output_price=8,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "gpt-4.1-2025-04-14": ModelConfig(
        model_name="gpt-4.1-2025-04-14",
        display_name="GPT-4.1-2025-04-14 (Prompt)",
        url="https://openai.com/index/gpt-4-1/",
        org="OpenAI",
        license="Proprietary",
        model_handler=OpenAIResponsesHandler,
        input_price=2,
        output_price=8,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "gpt-4.1-mini-2025-04-14-FC": ModelConfig(
        model_name="gpt-4.1-mini-2025-04-14",
        display_name="GPT-4.1-mini-2025-04-14 (FC)",
        url="https://openai.com/index/gpt-4-1/",
        org="OpenAI",
        license="Proprietary",
        model_handler=OpenAIResponsesHandler,
        input_price=0.4,
        output_price=1.6,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "gpt-4.1-mini-2025-04-14": ModelConfig(
        model_name="gpt-4.1-mini-2025-04-14",
        display_name="GPT-4.1-mini-2025-04-14 (Prompt)",
        url="https://openai.com/index/gpt-4-1/",
        org="OpenAI",
        license="Proprietary",
        model_handler=OpenAIResponsesHandler,
        input_price=0.4,
        output_price=1.6,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "gpt-4.1-nano-2025-04-14-FC": ModelConfig(
        model_name="gpt-4.1-nano-2025-04-14",
        display_name="GPT-4.1-nano-2025-04-14 (FC)",
        url="https://openai.com/index/gpt-4-1/",
        org="OpenAI",
        license="Proprietary",
        model_handler=OpenAIResponsesHandler,
        input_price=0.1,
        output_price=0.4,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "gpt-4.1-nano-2025-04-14": ModelConfig(
        model_name="gpt-4.1-nano-2025-04-14",
        display_name="GPT-4.1-nano-2025-04-14 (Prompt)",
        url="https://openai.com/index/gpt-4-1/",
        org="OpenAI",
        license="Proprietary",
        model_handler=OpenAIResponsesHandler,
        input_price=0.1,
        output_price=0.4,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "gpt-4o-2024-11-20": ModelConfig(
        model_name="gpt-4o-2024-11-20",
        display_name="GPT-4o-2024-11-20 (Prompt)",
        url="https://openai.com/index/hello-gpt-4o/",
        org="OpenAI",
        license="Proprietary",
        model_handler=OpenAIResponsesHandler,
        input_price=2.5,
        output_price=10,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "gpt-4o-2024-11-20-FC": ModelConfig(
        model_name="gpt-4o-2024-11-20",
        display_name="GPT-4o-2024-11-20 (FC)",
        url="https://openai.com/index/hello-gpt-4o/",
        org="OpenAI",
        license="Proprietary",
        model_handler=OpenAIResponsesHandler,
        input_price=2.5,
        output_price=10,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "gpt-4o-mini-2024-07-18": ModelConfig(
        model_name="gpt-4o-mini-2024-07-18",
        display_name="GPT-4o-mini-2024-07-18 (Prompt)",
        url="https://openai.com/index/gpt-4o-mini-advancing-cost-efficient-intelligence/",
        org="OpenAI",
        license="Proprietary",
        model_handler=OpenAIResponsesHandler,
        input_price=0.15,
        output_price=0.6,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "gpt-4o-mini-2024-07-18-FC": ModelConfig(
        model_name="gpt-4o-mini-2024-07-18",
        display_name="GPT-4o-mini-2024-07-18 (FC)",
        url="https://openai.com/index/gpt-4o-mini-advancing-cost-efficient-intelligence/",
        org="OpenAI",
        license="Proprietary",
        model_handler=OpenAIResponsesHandler,
        input_price=0.15,
        output_price=0.6,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "o3-2025-04-16": ModelConfig(
        model_name="o3-2025-04-16",
        display_name="o3-2025-04-16 (Prompt)",
        url="https://openai.com/index/introducing-o3-and-o4-mini/",
        org="OpenAI",
        license="Proprietary",
        model_handler=OpenAIResponsesHandler,
        input_price=2,
        output_price=8,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "o3-2025-04-16-FC": ModelConfig(
        model_name="o3-2025-04-16",
        display_name="o3-2025-04-16 (FC)",
        url="https://openai.com/index/introducing-o3-and-o4-mini/",
        org="OpenAI",
        license="Proprietary",
        model_handler=OpenAIResponsesHandler,
        input_price=2,
        output_price=8,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "o4-mini-2025-04-16": ModelConfig(
        model_name="o4-mini-2025-04-16",
        display_name="o4-mini-2025-04-16 (Prompt)",
        url="https://openai.com/index/introducing-o3-and-o4-mini/",
        org="OpenAI",
        license="Proprietary",
        model_handler=OpenAIResponsesHandler,
        input_price=1.10,
        output_price=4.40,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "o4-mini-2025-04-16-FC": ModelConfig(
        model_name="o4-mini-2025-04-16",
        display_name="o4-mini-2025-04-16 (FC)",
        url="https://openai.com/index/introducing-o3-and-o4-mini/",
        org="OpenAI",
        license="Proprietary",
        model_handler=OpenAIResponsesHandler,
        input_price=1.10,
        output_price=4.40,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "claude-opus-4-1-20250805": ModelConfig(
        model_name="claude-opus-4-1-20250805",
        display_name="Claude-Opus-4-1-20250805 (Prompt)",
        url="https://www.anthropic.com/news/claude-4",
        org="Anthropic",
        license="Proprietary",
        model_handler=ClaudeHandler,
        input_price=15,
        output_price=75,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "claude-opus-4-1-20250805-FC": ModelConfig(
        model_name="claude-opus-4-1-20250805",
        display_name="Claude-Opus-4-1-20250805 (FC)",
        url="https://www.anthropic.com/news/claude-4",
        org="Anthropic",
        license="Proprietary",
        model_handler=ClaudeHandler,
        input_price=15,
        output_price=75,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "claude-sonnet-4-5-20250929": ModelConfig(
        model_name="claude-sonnet-4-5-20250929",
        display_name="Claude-Sonnet-4-5-20250929 (Prompt)",
        url="https://www.anthropic.com/news/claude-sonnet-4-5",
        org="Anthropic",
        license="Proprietary",
        model_handler=ClaudeHandler,
        input_price=3,
        output_price=15,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "claude-sonnet-4-5-20250929-FC": ModelConfig(
        model_name="claude-sonnet-4-5-20250929",
        display_name="Claude-Sonnet-4-5-20250929 (FC)",
        url="https://www.anthropic.com/news/claude-sonnet-4-5",
        org="Anthropic",
        license="Proprietary",
        model_handler=ClaudeHandler,
        input_price=3,
        output_price=15,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "claude-haiku-4-5-20251001": ModelConfig(
        model_name="claude-haiku-4-5-20251001",
        display_name="claude-3.5-haiku-20241022 (Prompt)",
        url="https://www.anthropic.com/news/claude-haiku-4-5",
        org="Anthropic",
        license="Proprietary",
        model_handler=ClaudeHandler,
        input_price=0.8,
        output_price=4,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "claude-haiku-4-5-20251001-FC": ModelConfig(
        model_name="claude-haiku-4-5-20251001",
        display_name="claude-3.5-haiku-20241022 (FC)",
        url="https://www.anthropic.com/news/claude-haiku-4-5",
        org="Anthropic",
        license="Proprietary",
        model_handler=ClaudeHandler,
        input_price=0.8,
        output_price=4,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "nova-pro-v1.0": ModelConfig(
        model_name="us.amazon.nova-pro-v1:0",
        display_name="Amazon-Nova-Pro-v1:0 (FC)",
        url="https://aws.amazon.com/cn/ai/generative-ai/nova/",
        org="Amazon",
        license="Proprietary",
        model_handler=NovaHandler,
        input_price=0.8,
        output_price=3.2,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "nova-lite-v1.0": ModelConfig(
        model_name="us.amazon.nova-lite-v1:0",
        display_name="Amazon-Nova-Lite-v1:0 (FC)",
        url="https://aws.amazon.com/cn/ai/generative-ai/nova/",
        org="Amazon",
        license="Proprietary",
        model_handler=NovaHandler,
        input_price=0.06,
        output_price=0.24,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "nova-micro-v1.0": ModelConfig(
        model_name="us.amazon.nova-micro-v1:0",
        display_name="Amazon-Nova-Micro-v1:0 (FC)",
        url="https://aws.amazon.com/cn/ai/generative-ai/nova/",
        org="Amazon",
        license="Proprietary",
        model_handler=NovaHandler,
        input_price=0.035,
        output_price=0.14,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "open-mistral-nemo-2407": ModelConfig(
        model_name="open-mistral-nemo-2407",
        display_name="Open-Mistral-Nemo-2407 (Prompt)",
        url="https://mistral.ai/news/mistral-nemo/",
        org="Mistral AI",
        license="Proprietary",
        model_handler=MistralHandler,
        input_price=0.15,
        output_price=0.15,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "open-mistral-nemo-2407-FC": ModelConfig(
        model_name="open-mistral-nemo-2407",
        display_name="Open-Mistral-Nemo-2407 (FC)",
        url="https://mistral.ai/news/mistral-nemo/",
        org="Mistral AI",
        license="Proprietary",
        model_handler=MistralHandler,
        input_price=0.15,
        output_price=0.15,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "mistral-large-2411": ModelConfig(
        model_name="mistral-large-2411",
        display_name="mistral-large-2411 (Prompt)",
        url="https://docs.mistral.ai/guides/model-selection/",
        org="Mistral AI",
        license="Proprietary",
        model_handler=MistralHandler,
        input_price=2,
        output_price=6,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "mistral-large-2411-FC": ModelConfig(
        model_name="mistral-large-2411",
        display_name="mistral-large-2411 (FC)",
        url="https://docs.mistral.ai/guides/model-selection/",
        org="Mistral AI",
        license="Proprietary",
        model_handler=MistralHandler,
        input_price=2,
        output_price=6,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "mistral-small-2506": ModelConfig(
        model_name="mistral-small-2506",
        display_name="Mistral-Small-2506 (Prompt)",
        url="https://docs.mistral.ai/guides/model-selection/",
        org="Mistral AI",
        license="Proprietary",
        model_handler=MistralHandler,
        input_price=0.1,
        output_price=0.3,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "mistral-small-2506-FC": ModelConfig(
        model_name="mistral-small-2506",
        display_name="Mistral-small-2506 (FC)",
        url="https://docs.mistral.ai/guides/model-selection/",
        org="Mistral AI",
        license="Proprietary",
        model_handler=MistralHandler,
        input_price=0.1,
        output_price=0.3,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "mistral-medium-2505": ModelConfig(
        model_name="mistral-medium-2505",
        display_name="Mistral-Medium-2505",
        url="https://docs.mistral.ai/guides/model-selection/",
        org="Mistral AI",
        license="Proprietary",
        model_handler=MistralHandler,
        input_price=0.4,
        output_price=2,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "mistral-medium-2505-FC": ModelConfig(
        model_name="mistral-medium-2505",
        display_name="Mistral-Medium-2505 (FC)",
        url="https://docs.mistral.ai/guides/model-selection/",
        org="Mistral AI",
        license="Proprietary",
        model_handler=MistralHandler,
        input_price=0.4,
        output_price=2,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "firefunction-v2-FC": ModelConfig(
        model_name="accounts/fireworks/models/firefunction-v2",
        display_name="FireFunction-v2 (FC)",
        url="https://huggingface.co/fireworks-ai/firefunction-v2",
        org="Fireworks",
        license="Apache 2.0",
        model_handler=FireworksHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=False,
    ),
    "gemini-2.5-flash-lite-preview-06-17-FC": ModelConfig(
        model_name="gemini-2.5-flash-lite-preview-06-17",
        display_name="Gemini-2.5-Flash-Lite-Preview-06-17 (FC)",
        url="https://deepmind.google/technologies/gemini/flash-lite/",
        org="Google",
        license="Proprietary",
        model_handler=GeminiHandler,
        input_price=0.1,
        output_price=0.4,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "gemini-2.5-flash-lite-preview-06-17": ModelConfig(
        model_name="gemini-2.5-flash-lite-preview-06-17",
        display_name="Gemini-2.5-Flash-Lite-Preview-06-17 (Prompt)",
        url="https://deepmind.google/technologies/gemini/flash-lite/",
        org="Google",
        license="Proprietary",
        model_handler=GeminiHandler,
        input_price=0.1,
        output_price=0.4,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "gemini-2.5-flash-FC": ModelConfig(
        model_name="gemini-2.5-flash",
        display_name="Gemini-2.5-Flash (FC)",
        url="https://deepmind.google/technologies/gemini/flash/",
        org="Google",
        license="Proprietary",
        model_handler=GeminiHandler,
        input_price=0.3,
        output_price=2.5,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "gemini-2.5-flash": ModelConfig(
        model_name="gemini-2.5-flash",
        display_name="Gemini-2.5-Flash (Prompt)",
        url="https://deepmind.google/technologies/gemini/flash/",
        org="Google",
        license="Proprietary",
        model_handler=GeminiHandler,
        input_price=0.3,
        output_price=2.5,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "gemini-2.5-pro-FC": ModelConfig(
        model_name="gemini-2.5-pro",
        display_name="Gemini-2.5-Pro (FC)",
        url="https://deepmind.google/technologies/gemini/pro/",
        org="Google",
        license="Proprietary",
        model_handler=GeminiHandler,
        input_price=1.5,
        output_price=10,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "gemini-2.5-pro": ModelConfig(
        model_name="gemini-2.5-pro",
        display_name="Gemini-2.5-Pro (Prompt)",
        url="https://deepmind.google/technologies/gemini/pro/",
        org="Google",
        license="Proprietary",
        model_handler=GeminiHandler,
        input_price=1.5,
        output_price=10,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "meetkai/functionary-small-v3.1-FC": ModelConfig(
        model_name="meetkai/functionary-small-v3.1",
        display_name="Functionary-Small-v3.1 (FC)",
        url="https://huggingface.co/meetkai/functionary-small-v3.1",
        org="MeetKai",
        license="MIT",
        model_handler=FunctionaryHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "meetkai/functionary-medium-v3.1-FC": ModelConfig(
        model_name="meetkai/functionary-medium-v3.1",
        display_name="Functionary-Medium-v3.1 (FC)",
        url="https://huggingface.co/meetkai/functionary-medium-v3.1",
        org="MeetKai",
        license="MIT",
        model_handler=FunctionaryHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "command-r7b-12-2024-FC": ModelConfig(
        model_name="command-r7b-12-2024",
        display_name="Command R7B (FC)",
        url="https://cohere.com/blog/command-r7b",
        org="Cohere",
        license="cc-by-nc-4.0",
        model_handler=CohereHandler,
        input_price=0.0375,
        output_price=0.15,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "command-a-03-2025-FC": ModelConfig(
        model_name="command-a-03-2025",
        display_name="Command A (FC)",
        url="https://cohere.com/blog/command-a",
        org="Cohere",
        license="CC-BY-NC 4.0 License (w/ Acceptable Use Addendum)",
        model_handler=CohereHandler,
        input_price=2.5,
        output_price=10,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "command-a-reasoning-08-2025-FC": ModelConfig(
        model_name="command-a-reasoning-08-2025-FC",
        display_name="Command A Reasoning (FC)",
        url="https://cohere.com/blog/command-a-reasoning",
        org="Cohere",
        license="CC-BY-NC 4.0 License (w/ Acceptable Use Addendum)",
        model_handler=CohereHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "nvidia/llama-3.1-nemotron-ultra-253b-v1": ModelConfig(
        model_name="nvidia/llama-3.1-nemotron-ultra-253b-v1",
        display_name="Llama-3.1-Nemotron-Ultra-253B-v1 (FC)",
        url="https://huggingface.co/nvidia/Llama-3_1-Nemotron-Ultra-253B-v1",
        org="NVIDIA",
        license="nvidia-open-model-license",
        model_handler=NemotronHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=False,
    ),
    "nvidia/nemotron-4-340b-instruct": ModelConfig(
        model_name="nvidia/nemotron-4-340b-instruct",
        display_name="Nemotron-4-340b-instruct (Prompt)",
        url="https://huggingface.co/nvidia/nemotron-4-340b-instruct",
        org="NVIDIA",
        license="nvidia-open-model-license",
        model_handler=NvidiaHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "BitAgent/GoGoAgent": ModelConfig(
        model_name="BitAgent/GoGoAgent",
        display_name="GoGoAgent",
        url="https://gogoagent.ai",
        org="BitAgent",
        license="Proprietary",
        model_handler=GoGoAgentHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "palmyra-x-004": ModelConfig(
        model_name="palmyra-x-004",
        display_name="palmyra-x-004 (FC)",
        url="https://writer.com/engineering/actions-with-palmyra-x-004/",
        org="Writer",
        license="Proprietary",
        model_handler=WriterHandler,
        input_price=5,
        output_price=12,
        is_fc_model=True,
        underscore_to_dot=False,
    ),
    "grok-4-0709-FC": ModelConfig(
        model_name="grok-4-0709",
        display_name="Grok-4-0709 (FC)",
        url="https://docs.x.ai/docs/models",
        org="xAI",
        license="Proprietary",
        model_handler=GrokHandler,
        input_price=3,
        output_price=15,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "grok-4-0709": ModelConfig(
        model_name="grok-4-0709",
        display_name="Grok-4-0709 (Prompt)",
        url="https://docs.x.ai/docs/models",
        org="xAI",
        license="Proprietary",
        model_handler=GrokHandler,
        input_price=3,
        output_price=15,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "qwen3-0.6b-FC": ModelConfig(
        model_name="qwen3-0.6b",
        display_name="Qwen3-0.6B (FC)",
        url="https://huggingface.co/Qwen/Qwen3-0.6B",
        org="Qwen",
        license="apache-2.0",
        model_handler=QwenAPIHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "qwen3-0.6b": ModelConfig(
        model_name="qwen3-0.6b",
        display_name="Qwen3-0.6B (Prompt)",
        url="https://huggingface.co/Qwen/Qwen3-0.6B",
        org="Qwen",
        license="apache-2.0",
        model_handler=QwenAPIHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "qwen3-1.7b-FC": ModelConfig(
        model_name="qwen3-1.7b",
        display_name="Qwen3-1.7B (FC)",
        url="https://huggingface.co/Qwen/Qwen3-1.7B",
        org="Qwen",
        license="apache-2.0",
        model_handler=QwenAPIHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "qwen3-1.7b": ModelConfig(
        model_name="qwen3-1.7b",
        display_name="Qwen3-1.7B (Prompt)",
        url="https://huggingface.co/Qwen/Qwen3-1.7B",
        org="Qwen",
        license="apache-2.0",
        model_handler=QwenAPIHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "qwen3-4b-FC": ModelConfig(
        model_name="qwen3-4b",
        display_name="Qwen3-4B (FC)",
        url="https://huggingface.co/Qwen/Qwen3-4B",
        org="Qwen",
        license="apache-2.0",
        model_handler=QwenAPIHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "qwen3-4b": ModelConfig(
        model_name="qwen3-4b",
        display_name="Qwen3-4B (Prompt)",
        url="https://huggingface.co/Qwen/Qwen3-4B",
        org="Qwen",
        license="apache-2.0",
        model_handler=QwenAPIHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "qwen3-8b-FC": ModelConfig(
        model_name="qwen3-8b",
        display_name="Qwen3-8B (FC)",
        url="https://huggingface.co/Qwen/Qwen3-8B",
        org="Qwen",
        license="apache-2.0",
        model_handler=QwenAPIHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "qwen3-8b": ModelConfig(
        model_name="qwen3-8b",
        display_name="Qwen3-8B (Prompt)",
        url="https://huggingface.co/Qwen/Qwen3-8B",
        org="Qwen",
        license="apache-2.0",
        model_handler=QwenAPIHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "qwen3-14b-FC": ModelConfig(
        model_name="qwen3-14b",
        display_name="Qwen3-14B (FC)",
        url="https://huggingface.co/Qwen/Qwen3-14B",
        org="Qwen",
        license="apache-2.0",
        model_handler=QwenAPIHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "qwen3-14b": ModelConfig(
        model_name="qwen3-14b",
        display_name="Qwen3-14B (Prompt)",
        url="https://huggingface.co/Qwen/Qwen3-14B",
        org="Qwen",
        license="apache-2.0",
        model_handler=QwenAPIHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "qwen3-32b-FC": ModelConfig(
        model_name="qwen3-32b",
        display_name="Qwen3-32B (FC)",
        url="https://huggingface.co/Qwen/Qwen3-32B",
        org="Qwen",
        license="apache-2.0",
        model_handler=QwenAPIHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "qwen3-32b": ModelConfig(
        model_name="qwen3-32b",
        display_name="Qwen3-32B (Prompt)",
        url="https://huggingface.co/Qwen/Qwen3-32B",
        org="Qwen",
        license="apache-2.0",
        model_handler=QwenAPIHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "qwen3-30b-a3b-instruct-2507-FC": ModelConfig(
        model_name="qwen3-30b-a3b-instruct-2507",
        display_name="Qwen3-30B-A3B-Instruct-2507 (FC)",
        url="https://huggingface.co/Qwen/Qwen3-30B-A3B-Instruct-2507",
        org="Qwen",
        license="apache-2.0",
        model_handler=QwenAPIHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "qwen3-30b-a3b-instruct-2507": ModelConfig(
        model_name="qwen3-30b-a3b-instruct-2507",
        display_name="Qwen3-30B-A3B-Instruct-2507 (Prompt)",
        url="https://huggingface.co/Qwen/Qwen3-30B-A3B-Instruct-2507",
        org="Qwen",
        license="apache-2.0",
        model_handler=QwenAPIHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "qwen3-235b-a22b-instruct-2507-FC": ModelConfig(
        model_name="qwen3-235b-a22b-instruct-2507",
        display_name="Qwen3-235B-A22B-Instruct-2507 (FC)",
        url="https://huggingface.co/Qwen/Qwen3-235B-A22B-Instruct-2507",
        org="Qwen",
        license="apache-2.0",
        model_handler=QwenAPIHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=False,
    ),
    "qwen3-235b-a22b-instruct-2507": ModelConfig(
        model_name="qwen3-235b-a22b-instruct-2507",
        display_name="Qwen3-235B-A22B-Instruct-2507 (Prompt)",
        url="https://huggingface.co/Qwen/Qwen3-235B-A22B-Instruct-2507",
        org="Qwen",
        license="apache-2.0",
        model_handler=QwenAPIHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "qwq-32b-FC": ModelConfig(
        model_name="qwq-32b",
        display_name="QwQ-32B (FC)",
        url="https://huggingface.co/Qwen/QwQ-32B",
        org="Qwen",
        license="apache-2.0",
        model_handler=QwenAPIHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "qwq-32b": ModelConfig(
        model_name="qwq-32b",
        display_name="QwQ-32B (Prompt)",
        url="https://huggingface.co/Qwen/QwQ-32B",
        org="Qwen",
        license="apache-2.0",
        model_handler=QwenAPIHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "xiaoming-14B": ModelConfig(
        model_name="xiaoming-14B",
        display_name="xiaoming-14B (FC)",
        url="https://www.mininglamp.com/",
        org="Mininglamp",
        license="Proprietary",
        model_handler=MiningHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=False,
    ),
    "DM-Cito-8B-v2": ModelConfig(
        model_name="DM-Cito-8B-v2",
        display_name="DM-Cito-8B-v2 (Prompt)",
        url="https://www.mininglamp.com/",
        org="Mininglamp",
        license="Proprietary",
        model_handler=DMCitoHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "Ling/ling-lite-v1.5": ModelConfig(
        model_name="Ling/ling-lite-v1.5",
        display_name="ling-lite-v1.5 (Prompt)",
        url="https://huggingface.co/inclusionAI/Ling-lite-1.5",
        org="Ling",
        license="MIT",
        model_handler=LingAPIHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "glm-4.5-FC": ModelConfig(
        model_name="glm-4.5",
        display_name="GLM-4.5 (FC)",
        url="https://huggingface.co/zai-org/GLM-4.5",
        org="Zhipu AI",
        license="MIT",
        model_handler=GLMAPIHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "glm-4.5-air-FC": ModelConfig(
        model_name="glm-4.5-air",
        display_name="GLM-4.5-Air (FC)",
        url="https://huggingface.co/zai-org/GLM-4.5-Air",
        org="Zhipu AI",
        license="MIT",
        model_handler=GLMAPIHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "kimi-k2-0711-preview-FC": ModelConfig(
        model_name="moonshotai/Kimi-K2-Instruct",
        display_name="Moonshotai-Kimi-K2-Instruct (FC)",
        url="https://huggingface.co/moonshotai/Kimi-K2-Instruct",
        org="MoonshotAI",
        license="modified-mit",
        model_handler=KimiHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "kimi-k2-0711-preview": ModelConfig(
        model_name="moonshotai/Kimi-K2-Instruct",
        display_name="Moonshotai-Kimi-K2-Instruct (Prompt)",
        url="https://huggingface.co/moonshotai/Kimi-K2-Instruct",
        org="MoonshotAI",
        license="modified-mit",
        model_handler=KimiHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
}

# Inference through local hosting
local_inference_model_map = {
    "deepseek-ai/DeepSeek-R1": ModelConfig(
        model_name="deepseek-ai/DeepSeek-R1",
        display_name="DeepSeek-R1 (Prompt) (Local)",
        url="https://huggingface.co/deepseek-ai/DeepSeek-R1",
        org="DeepSeek",
        license="MIT",
        model_handler=DeepseekReasoningHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "google/gemma-3-1b-it": ModelConfig(
        model_name="google/gemma-3-1b-it",
        display_name="Gemma-3-1b-it (Prompt)",
        url="https://blog.google/technology/developers/gemma-3/",
        org="Google",
        license="gemma-terms-of-use",
        model_handler=GemmaHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "google/gemma-3-4b-it": ModelConfig(
        model_name="google/gemma-3-4b-it",
        display_name="Gemma-3-4b-it (Prompt)",
        url="https://blog.google/technology/developers/gemma-3/",
        org="Google",
        license="gemma-terms-of-use",
        model_handler=GemmaHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "google/gemma-3-12b-it": ModelConfig(
        model_name="google/gemma-3-12b-it",
        display_name="Gemma-3-12b-it (Prompt)",
        url="https://blog.google/technology/developers/gemma-3/",
        org="Google",
        license="gemma-terms-of-use",
        model_handler=GemmaHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "google/gemma-3-27b-it": ModelConfig(
        model_name="google/gemma-3-27b-it",
        display_name="Gemma-3-27b-it (Prompt)",
        url="https://blog.google/technology/developers/gemma-3/",
        org="Google",
        license="gemma-terms-of-use",
        model_handler=GemmaHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "meta-llama/Llama-3.1-8B-Instruct-FC": ModelConfig(
        model_name="meta-llama/Llama-3.1-8B-Instruct",
        display_name="Llama-3.1-8B-Instruct (FC)",
        url="https://llama.meta.com/llama3",
        org="Meta",
        license="Meta Llama 3 Community",
        model_handler=LlamaHandler_3_1,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=False,
    ),
    "meta-llama/Llama-3.1-8B-Instruct": ModelConfig(
        model_name="meta-llama/Llama-3.1-8B-Instruct",
        display_name="Llama-3.1-8B-Instruct (Prompt)",
        url="https://llama.meta.com/llama3",
        org="Meta",
        license="Meta Llama 3 Community",
        model_handler=LlamaHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "meta-llama/Llama-3.1-70B-Instruct-FC": ModelConfig(
        model_name="meta-llama/Llama-3.1-70B-Instruct",
        display_name="Llama-3.1-70B-Instruct (FC)",
        url="https://llama.meta.com/llama3",
        org="Meta",
        license="Meta Llama 3 Community",
        model_handler=LlamaHandler_3_1,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=False,
    ),
    "meta-llama/Llama-3.1-70B-Instruct": ModelConfig(
        model_name="meta-llama/Llama-3.1-70B-Instruct",
        display_name="Llama-3.1-70B-Instruct (Prompt)",
        url="https://llama.meta.com/llama3",
        org="Meta",
        license="Meta Llama 3 Community",
        model_handler=LlamaHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "meta-llama/Llama-3.2-1B-Instruct-FC": ModelConfig(
        model_name="meta-llama/Llama-3.2-1B-Instruct",
        display_name="Llama-3.2-1B-Instruct (FC)",
        url="https://llama.meta.com/llama3",
        org="Meta",
        license="Meta Llama 3 Community",
        model_handler=LlamaHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=False,
    ),
    "meta-llama/Llama-3.2-3B-Instruct-FC": ModelConfig(
        model_name="meta-llama/Llama-3.2-3B-Instruct",
        display_name="Llama-3.2-3B-Instruct (FC)",
        url="https://llama.meta.com/llama3",
        org="Meta",
        license="Meta Llama 3 Community",
        model_handler=LlamaHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=False,
    ),
    "meta-llama/Llama-3.3-70B-Instruct-FC": ModelConfig(
        model_name="meta-llama/Llama-3.3-70B-Instruct",
        display_name="Llama-3.3-70B-Instruct (FC)",
        url="https://llama.meta.com/llama3",
        org="Meta",
        license="Meta Llama 3 Community",
        model_handler=LlamaHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=False,
    ),
    "meta-llama/Llama-4-Scout-17B-16E-Instruct-FC": ModelConfig(
        model_name="meta-llama/Llama-4-Scout-17B-16E-Instruct",
        display_name="Llama-4-Scout-17B-16E-Instruct (FC)",
        url="https://huggingface.co/meta-llama/Llama-4-Scout-17B-16E-Instruct",
        org="Meta",
        license="Meta Llama 4 Community",
        model_handler=LlamaHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=False,
    ),
    "meta-llama/Llama-4-Maverick-17B-128E-Instruct-FP8-FC": ModelConfig(
        model_name="meta-llama/Llama-4-Maverick-17B-128E-Instruct-FP8",
        display_name="Llama-4-Maverick-17B-128E-Instruct-FP8 (FC)",
        url="https://huggingface.co/meta-llama/Llama-4-Maverick-17B-128E-Instruct-FP8",
        org="Meta",
        license="Meta Llama 4 Community",
        model_handler=LlamaHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=False,
    ),
    "Salesforce/Llama-xLAM-2-70b-fc-r": ModelConfig(
        model_name="Salesforce/Llama-xLAM-2-70b-fc-r",
        display_name="xLAM-2-70b-fc-r (FC)",
        url="https://huggingface.co/Salesforce/Llama-xLAM-2-70b-fc-r",
        org="Salesforce",
        license="cc-by-nc-4.0",
        model_handler=SalesforceLlamaHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=False,
    ),
    "Salesforce/Llama-xLAM-2-8b-fc-r": ModelConfig(
        model_name="Salesforce/Llama-xLAM-2-8b-fc-r",
        display_name="xLAM-2-8b-fc-r (FC)",
        url="https://huggingface.co/Salesforce/Llama-xLAM-2-8b-fc-r",
        org="Salesforce",
        license="cc-by-nc-4.0",
        model_handler=SalesforceLlamaHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=False,
    ),
    "Salesforce/xLAM-2-32b-fc-r": ModelConfig(
        model_name="Salesforce/xLAM-2-32b-fc-r",
        display_name="xLAM-2-32b-fc-r (FC)",
        url="https://huggingface.co/Salesforce/xLAM-2-32b-fc-r",
        org="Salesforce",
        license="cc-by-nc-4.0",
        model_handler=SalesforceQwenHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=False,
    ),
    "Salesforce/xLAM-2-3b-fc-r": ModelConfig(
        model_name="Salesforce/xLAM-2-3b-fc-r",
        display_name="xLAM-2-3b-fc-r (FC)",
        url="https://huggingface.co/Salesforce/xLAM-2-3b-fc-r",
        org="Salesforce",
        license="cc-by-nc-4.0",
        model_handler=SalesforceQwenHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=False,
    ),
    "Salesforce/xLAM-2-1b-fc-r": ModelConfig(
        model_name="Salesforce/xLAM-2-1b-fc-r",
        display_name="xLAM-2-1b-fc-r (FC)",
        url="https://huggingface.co/Salesforce/xLAM-2-1b-fc-r",
        org="Salesforce",
        license="cc-by-nc-4.0",
        model_handler=SalesforceQwenHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=False,
    ),
    "mistralai/Ministral-8B-Instruct-2410": ModelConfig(
        model_name="mistralai/Ministral-8B-Instruct-2410",
        display_name="Ministral-8B-Instruct-2410 (FC)",
        url="https://huggingface.co/mistralai/Ministral-8B-Instruct-2410",
        org="Mistral AI",
        license="Mistral AI Research License",
        model_handler=MistralFCHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "microsoft/phi-4": ModelConfig(
        model_name="microsoft/phi-4",
        display_name="Phi-4 (Prompt)",
        url="https://huggingface.co/microsoft/phi-4",
        org="Microsoft",
        license="MIT",
        model_handler=PhiHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "microsoft/Phi-4-mini-instruct": ModelConfig(
        model_name="microsoft/Phi-4-mini-instruct",
        display_name="Phi-4-mini-instruct (Prompt)",
        url="https://huggingface.co/microsoft/Phi-4-mini-instruct",
        org="Microsoft",
        license="MIT",
        model_handler=PhiHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "microsoft/Phi-4-mini-instruct-FC": ModelConfig(
        model_name="microsoft/Phi-4-mini-instruct",
        display_name="Phi-4-mini-instruct (FC)",
        url="https://huggingface.co/microsoft/Phi-4-mini-instruct",
        org="Microsoft",
        license="MIT",
        model_handler=PhiFCHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=False,
    ),
    "ibm-granite/granite-3.2-8b-instruct": ModelConfig(
        model_name="ibm-granite/granite-3.2-8b-instruct",
        display_name="Granite-3.2-8B-Instruct (FC)",
        url="https://huggingface.co/ibm-granite/granite-3.2-8b-instruct",
        org="IBM",
        license="Apache-2.0",
        model_handler=Granite3FCHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=False,
    ),
    "ibm-granite/granite-3.1-8b-instruct": ModelConfig(
        model_name="ibm-granite/granite-3.1-8b-instruct",
        display_name="Granite-3.1-8B-Instruct (FC)",
        url="https://huggingface.co/ibm-granite/granite-3.1-8b-instruct",
        org="IBM",
        license="Apache-2.0",
        model_handler=Granite3FCHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=False,
    ),
    "ibm-granite/granite-20b-functioncalling": ModelConfig(
        model_name="ibm-granite/granite-20b-functioncalling",
        display_name="Granite-20b-FunctionCalling (FC)",
        url="https://huggingface.co/ibm-granite/granite-20b-functioncalling",
        org="IBM",
        license="Apache-2.0",
        model_handler=GraniteFunctionCallingHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "MadeAgents/Hammer2.1-7b": ModelConfig(
        model_name="MadeAgents/Hammer2.1-7b",
        display_name="Hammer2.1-7b (FC)",
        url="https://huggingface.co/MadeAgents/Hammer2.1-7b",
        org="MadeAgents",
        license="cc-by-nc-4.0",
        model_handler=HammerHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=False,
    ),
    "MadeAgents/Hammer2.1-3b": ModelConfig(
        model_name="MadeAgents/Hammer2.1-3b",
        display_name="Hammer2.1-3b (FC)",
        url="https://huggingface.co/MadeAgents/Hammer2.1-3b",
        org="MadeAgents",
        license="qwen-research",
        model_handler=HammerHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=False,
    ),
    "MadeAgents/Hammer2.1-1.5b": ModelConfig(
        model_name="MadeAgents/Hammer2.1-1.5b",
        display_name="Hammer2.1-1.5b (FC)",
        url="https://huggingface.co/MadeAgents/Hammer2.1-1.5b",
        org="MadeAgents",
        license="cc-by-nc-4.0",
        model_handler=HammerHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=False,
    ),
    "MadeAgents/Hammer2.1-0.5b": ModelConfig(
        model_name="MadeAgents/Hammer2.1-0.5b",
        display_name="Hammer2.1-0.5b (FC)",
        url="https://huggingface.co/MadeAgents/Hammer2.1-0.5b",
        org="MadeAgents",
        license="cc-by-nc-4.0",
        model_handler=HammerHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=False,
    ),
    "THUDM/glm-4-9b-chat": ModelConfig(
        model_name="THUDM/glm-4-9b-chat",
        display_name="GLM-4-9b-Chat (FC)",
        url="https://huggingface.co/THUDM/glm-4-9b-chat",
        org="THUDM",
        license="glm-4",
        model_handler=GLMHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "Qwen/Qwen3-0.6B-FC": ModelConfig(
        model_name="Qwen/Qwen3-0.6B",
        display_name="Qwen3-0.6B (FC)",
        url="https://huggingface.co/Qwen/Qwen3-0.6B",
        org="Qwen",
        license="apache-2.0",
        model_handler=QwenFCHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=False,
    ),
    "Qwen/Qwen3-0.6B": ModelConfig(
        model_name="Qwen/Qwen3-0.6B",
        display_name="Qwen3-0.6B (Prompt)",
        url="https://huggingface.co/Qwen/Qwen3-0.6B",
        org="Qwen",
        license="apache-2.0",
        model_handler=QwenHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "Qwen/Qwen3-1.7B-FC": ModelConfig(
        model_name="Qwen/Qwen3-1.7B",
        display_name="Qwen3-1.7B (FC)",
        url="https://huggingface.co/Qwen/Qwen3-1.7B",
        org="Qwen",
        license="apache-2.0",
        model_handler=QwenFCHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=False,
    ),
    "Qwen/Qwen3-1.7B": ModelConfig(
        model_name="Qwen/Qwen3-1.7B",
        display_name="Qwen3-1.7B (Prompt)",
        url="https://huggingface.co/Qwen/Qwen3-1.7B",
        org="Qwen",
        license="apache-2.0",
        model_handler=QwenHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "Qwen/Qwen3-4B-Instruct-2507-FC": ModelConfig(
        model_name="Qwen/Qwen3-4B-Instruct-2507",
        display_name="Qwen3-4B-Instruct-2507 (FC)",
        url="https://huggingface.co/Qwen/Qwen3-4B-Instruct-2507",
        org="Qwen",
        license="apache-2.0",
        model_handler=QwenFCHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=False,
    ),
    "Qwen/Qwen3-4B-Instruct-2507": ModelConfig(
        model_name="Qwen/Qwen3-4B-Instruct-2507",
        display_name="Qwen3-4B-Instruct-2507 (Prompt)",
        url="https://huggingface.co/Qwen/Qwen3-4B-Instruct-2507",
        org="Qwen",
        license="apache-2.0",
        model_handler=QwenHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "Qwen/Qwen3-8B-FC": ModelConfig(
        model_name="Qwen/Qwen3-8B",
        display_name="Qwen3-8B (FC)",
        url="https://huggingface.co/Qwen/Qwen3-8B",
        org="Qwen",
        license="apache-2.0",
        model_handler=QwenFCHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=False,
    ),
    "Qwen/Qwen3-8B": ModelConfig(
        model_name="Qwen/Qwen3-8B",
        display_name="Qwen3-8B (Prompt)",
        url="https://huggingface.co/Qwen/Qwen3-8B",
        org="Qwen",
        license="apache-2.0",
        model_handler=QwenHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "Qwen/Qwen3-14B-FC": ModelConfig(
        model_name="Qwen/Qwen3-14B",
        display_name="Qwen3-14B (FC)",
        url="https://huggingface.co/Qwen/Qwen3-14B",
        org="Qwen",
        license="apache-2.0",
        model_handler=QwenFCHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=False,
    ),
    "Qwen/Qwen3-14B": ModelConfig(
        model_name="Qwen/Qwen3-14B",
        display_name="Qwen3-14B (Prompt)",
        url="https://huggingface.co/Qwen/Qwen3-14B",
        org="Qwen",
        license="apache-2.0",
        model_handler=QwenHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "Qwen/Qwen3-32B-FC": ModelConfig(
        model_name="Qwen/Qwen3-32B",
        display_name="Qwen3-32B (FC)",
        url="https://huggingface.co/Qwen/Qwen3-32B",
        org="Qwen",
        license="apache-2.0",
        model_handler=QwenFCHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=False,
    ),
    "Qwen/Qwen3-32B": ModelConfig(
        model_name="Qwen/Qwen3-32B",
        display_name="Qwen3-32B (Prompt)",
        url="https://huggingface.co/Qwen/Qwen3-32B",
        org="Qwen",
        license="apache-2.0",
        model_handler=QwenHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "Qwen/Qwen3-30B-A3B-Instruct-2507-FC": ModelConfig(
        model_name="Qwen/Qwen3-30B-A3B-Instruct-2507",
        display_name="Qwen3-30B-A3B-Instruct-2507 (FC)",
        url="https://huggingface.co/Qwen/Qwen3-30B-A3B-Instruct-2507",
        org="Qwen",
        license="apache-2.0",
        model_handler=QwenFCHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=False,
    ),
    "Qwen/Qwen3-30B-A3B-Instruct-2507": ModelConfig(
        model_name="Qwen/Qwen3-30B-A3B-Instruct-2507",
        display_name="Qwen3-30B-A3B-Instruct-2507 (Prompt)",
        url="https://huggingface.co/Qwen/Qwen3-30B-A3B-Instruct-2507",
        org="Qwen",
        license="apache-2.0",
        model_handler=QwenHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "Qwen/Qwen3-235B-A22B-Instruct-2507-FC": ModelConfig(
        model_name="Qwen/Qwen3-235B-A22B-Instruct-2507",
        display_name="Qwen3-235B-A22B-Instruct-2507 (FC)",
        url="https://huggingface.co/Qwen/Qwen3-235B-A22B-Instruct-2507",
        org="Qwen",
        license="apache-2.0",
        model_handler=QwenFCHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=False,
    ),
    "Qwen/Qwen3-235B-A22B-Instruct-2507": ModelConfig(
        model_name="Qwen/Qwen3-235B-A22B-Instruct-2507",
        display_name="Qwen3-235B-A22B-Instruct-2507 (Prompt)",
        url="https://huggingface.co/Qwen/Qwen3-235B-A22B-Instruct-2507",
        org="Qwen",
        license="apache-2.0",
        model_handler=QwenHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "Team-ACE/ToolACE-2-8B": ModelConfig(
        model_name="Team-ACE/ToolACE-2-8B",
        display_name="ToolACE-2-8B (FC)",
        url="https://huggingface.co/Team-ACE/ToolACE-2-8B",
        org="Huawei Noah & USTC",
        license="Apache-2.0",
        model_handler=LlamaHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "openbmb/MiniCPM3-4B": ModelConfig(
        model_name="openbmb/MiniCPM3-4B",
        display_name="MiniCPM3-4B (Prompt)",
        url="https://huggingface.co/openbmb/MiniCPM3-4B",
        org="openbmb",
        license="Apache-2.0",
        model_handler=MiniCPMHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "openbmb/MiniCPM3-4B-FC": ModelConfig(
        model_name="openbmb/MiniCPM3-4B",
        display_name="MiniCPM3-4B-FC (FC)",
        url="https://huggingface.co/openbmb/MiniCPM3-4B",
        org="openbmb",
        license="Apache-2.0",
        model_handler=MiniCPMFCHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "watt-ai/watt-tool-8B": ModelConfig(
        model_name="watt-ai/watt-tool-8B",
        display_name="watt-tool-8B (FC)",
        url="https://huggingface.co/watt-ai/watt-tool-8B/",
        org="Watt AI Lab",
        license="Apache-2.0",
        model_handler=LlamaHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "watt-ai/watt-tool-70B": ModelConfig(
        model_name="watt-ai/watt-tool-70B",
        display_name="watt-tool-70B (FC)",
        url="https://huggingface.co/watt-ai/watt-tool-70B/",
        org="Watt AI Lab",
        license="Apache-2.0",
        model_handler=LlamaHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "ZJared/Haha-7B": ModelConfig(
        model_name="ZJared/Haha-7B",
        display_name="Haha-7B",
        url="https://huggingface.co/ZJared/Haha-7B",
        org="TeleAI",
        license="Apache 2.0",
        model_handler=QwenHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "speakleash/Bielik-11B-v2.3-Instruct": ModelConfig(
        model_name="speakleash/Bielik-11B-v2.3-Instruct",
        display_name="Bielik-11B-v2.3-Instruct (Prompt)",
        url="https://huggingface.co/speakleash/Bielik-11B-v2.3-Instruct",
        org="SpeakLeash & ACK Cyfronet AGH",
        license="Apache 2.0",
        model_handler=BielikHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "NovaSky-AI/Sky-T1-32B-Preview": ModelConfig(
        model_name="NovaSky-AI/Sky-T1-32B-Preview",
        display_name="Sky-T1-32B-Preview (Prompt)",
        url="https://huggingface.co/NovaSky-AI/Sky-T1-32B-Preview",
        org="NovaSky-AI",
        license="apache-2.0",
        model_handler=QwenHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "tiiuae/Falcon3-10B-Instruct-FC": ModelConfig(
        model_name="tiiuae/Falcon3-10B-Instruct",
        display_name="Falcon3-10B-Instruct (FC)",
        url="https://huggingface.co/tiiuae/Falcon3-10B-Instruct",
        org="TII UAE",
        license="falcon-llm-license",
        model_handler=Falcon3FCHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=False,
    ),
    "tiiuae/Falcon3-7B-Instruct-FC": ModelConfig(
        model_name="tiiuae/Falcon3-7B-Instruct",
        display_name="Falcon3-7B-Instruct (FC)",
        url="https://huggingface.co/tiiuae/Falcon3-7B-Instruct",
        org="TII UAE",
        license="falcon-llm-license",
        model_handler=Falcon3FCHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=False,
    ),
    "tiiuae/Falcon3-3B-Instruct-FC": ModelConfig(
        model_name="tiiuae/Falcon3-3B-Instruct",
        display_name="Falcon3-3B-Instruct (FC)",
        url="https://huggingface.co/tiiuae/Falcon3-3B-Instruct",
        org="TII UAE",
        license="falcon-llm-license",
        model_handler=Falcon3FCHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=False,
    ),
    "tiiuae/Falcon3-1B-Instruct-FC": ModelConfig(
        model_name="tiiuae/Falcon3-1B-Instruct",
        display_name="Falcon3-1B-Instruct (FC)",
        url="https://huggingface.co/tiiuae/Falcon3-1B-Instruct",
        org="TII UAE",
        license="falcon-llm-license",
        model_handler=Falcon3FCHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=False,
    ),
    "uiuc-convai/CoALM-8B": ModelConfig(
        model_name="uiuc-convai/CoALM-8B",
        display_name="CoALM-8B",
        url="https://huggingface.co/uiuc-convai/CoALM-8B",
        org="UIUC + Oumi",
        license="Meta Llama 3 Community",
        model_handler=LlamaHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "uiuc-convai/CoALM-70B": ModelConfig(
        model_name="uiuc-convai/CoALM-70B",
        display_name="CoALM-70B",
        url="https://huggingface.co/uiuc-convai/CoALM-70B",
        org="UIUC + Oumi",
        license="Meta Llama 3 Community",
        model_handler=LlamaHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "uiuc-convai/CoALM-405B": ModelConfig(
        model_name="uiuc-convai/CoALM-405B",
        display_name="CoALM-405B",
        url="https://huggingface.co/uiuc-convai/CoALM-405B",
        org="UIUC + Oumi",
        license="Meta Llama 3 Community",
        model_handler=LlamaHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "katanemo/Arch-Agent-1.5B": ModelConfig(
        model_name="katanemo/Arch-Agent-1.5B",
        display_name="Arch-Agent-1.5B",
        url="https://huggingface.co/katanemo/Arch-Agent-1.5B",
        org="katanemo",
        license="katanemo-research",
        model_handler=ArchHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=False,
    ),
    "katanemo/Arch-Agent-3B": ModelConfig(
        model_name="katanemo/Arch-Agent-3B",
        display_name="Arch-Agent-3B",
        url="https://huggingface.co/katanemo/Arch-Agent-3B",
        org="katanemo",
        license="katanemo-research",
        model_handler=ArchHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=False,
    ),
    "katanemo/Arch-Agent-7B": ModelConfig(
        model_name="katanemo/Arch-Agent-7B",
        display_name="Arch-Agent-7B",
        url="https://huggingface.co/katanemo/Arch-Agent-7B",
        org="katanemo",
        license="katanemo-research",
        model_handler=ArchHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=False,
    ),
    "katanemo/Arch-Agent-32B": ModelConfig(
        model_name="katanemo/Arch-Agent-32B",
        display_name="Arch-Agent-32B",
        url="https://huggingface.co/katanemo/Arch-Agent-32B",
        org="katanemo",
        license="katanemo-research",
        model_handler=ArchHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=False,
    ),
    "BitAgent/BitAgent-8B": ModelConfig(
        model_name="BitAgent/BitAgent-8B",
        display_name="BitAgent-8B",
        url="https://huggingface.co/BitAgent/BitAgent-8B/",
        org="Bittensor",
        license="Apache-2.0",
        model_handler=LlamaHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "BitAgent/BitAgent-Bounty-8B": ModelConfig(
        model_name="BitAgent/BitAgent-Bounty-8B",
        display_name="BitAgent-Bounty-8B",
        url="https://huggingface.co/BitAgent/BitAgent-Bounty-8B",
        org="Bittensor",
        license="Apache-2.0",
        model_handler=BitAgentHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=False,
    ),
    "ThinkAgents/ThinkAgent-1B": ModelConfig(
        model_name="ThinkAgents/ThinkAgent-1B",
        display_name="ThinkAgent-1B (FC)",
        url="https://huggingface.co/ThinkAgents/ThinkAgent-1B",
        org="ThinkAgents",
        license="apache-2.0",
        model_handler=ThinkAgentHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=False,
    ),
    "phronetic-ai/RZN-T": ModelConfig(
        model_name="phronetic-ai/RZN-T",
        display_name="RZN-T (Prompt)",
        url="https://huggingface.co/phronetic-ai/RZN-T",
        org="Phronetic AI",
        license="apache-2.0",
        model_handler=QwenHandler,
        input_price=None,
        output_price=None,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
}

# Inference through third-party inference platforms for open-source models
third_party_inference_model_map = {
    # Via Novita AI Endpoint
    "meta-llama/llama-4-maverick-17b-128e-instruct-fp8-novita": ModelConfig(
        model_name="meta-llama/llama-4-maverick-17b-128e-instruct-fp8",
        display_name="Llama-4-Maverick-17B-128E-Instruct-FP8 (Prompt) (Novita)",
        url="https://huggingface.co/meta-llama/Llama-4-Maverick-17B-128E-Instruct-FP8",
        org="Meta",
        license="Meta Llama 4 Community",
        model_handler=NovitaHandler,
        input_price=0.2,
        output_price=0.85,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "meta-llama/llama-4-maverick-17b-128e-instruct-fp8-FC-novita": ModelConfig(
        model_name="meta-llama/llama-4-maverick-17b-128e-instruct-fp8",
        display_name="Llama-4-Maverick-17B-128E-Instruct-FP8 (FC) (Novita)",
        url="https://huggingface.co/meta-llama/Llama-4-Maverick-17B-128E-Instruct-FP8",
        org="Meta",
        license="Meta Llama 4 Community",
        model_handler=NovitaHandler,
        input_price=0.2,
        output_price=0.85,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "meta-llama/llama-4-scout-17b-16e-instruct-novita": ModelConfig(
        model_name="meta-llama/llama-4-scout-17b-16e-instruct",
        display_name="Llama-4-Scout-17B-16E-Instruct (Prompt) (Novita)",
        url="https://huggingface.co/meta-llama/Llama-4-Scout-17B-16E-Instruct",
        org="Meta",
        license="Meta Llama 4 Community",
        model_handler=NovitaHandler,
        input_price=0.1,
        output_price=0.5,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    "meta-llama/llama-4-scout-17b-16e-instruct-FC-novita": ModelConfig(
        model_name="meta-llama/llama-4-scout-17b-16e-instruct",
        display_name="Llama-4-Scout-17B-16E-Instruct (FC) (Novita)",
        url="https://huggingface.co/meta-llama/Llama-4-Scout-17B-16E-Instruct",
        org="Meta",
        license="Meta Llama 4 Community",
        model_handler=NovitaHandler,
        input_price=0.1,
        output_price=0.5,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "qwen/qwq-32b-FC-novita": ModelConfig(
        model_name="qwen/qwq-32b",
        display_name="Qwen/QwQ-32B (FC) (Novita)",
        url="https://huggingface.co/Qwen/QwQ-32B",
        org="Qwen",
        license="apache-2.0",
        model_handler=NovitaHandler,
        input_price=0.18,
        output_price=0.2,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "qwen/qwq-32b-novita": ModelConfig(
        model_name="qwen/qwq-32b",
        display_name="Qwen/QwQ-32B (Prompt) (Novita)",
        url="https://huggingface.co/Qwen/QwQ-32B",
        org="Qwen",
        license="apache-2.0",
        model_handler=NovitaHandler,
        input_price=0.18,
        output_price=0.2,
        is_fc_model=False,
        underscore_to_dot=False,
    ),
    # Via Qwen Agent Framework
    "qwen3-4b-think-FC": ModelConfig(
        model_name="qwen3-4b-think-FC",
        display_name="Qwen3-4B-Think (FC)",
        url="https://huggingface.co/Qwen/Qwen3-4B",
        org="Qwen",
        license="apache-2.0",
        model_handler=QwenAgentThinkHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
    "qwen3-4b-nothink-FC": ModelConfig(
        model_name="qwen3-4b-nothink-FC",
        display_name="Qwen3-4B-NoThink (FC)",
        url="https://huggingface.co/Qwen/Qwen3-4B",
        org="Qwen",
        license="apache-2.0",
        model_handler=QwenAgentNoThinkHandler,
        input_price=None,
        output_price=None,
        is_fc_model=True,
        underscore_to_dot=True,
    ),
}


MODEL_CONFIG_MAPPING = {
    **api_inference_model_map,
    **local_inference_model_map,
    **third_party_inference_model_map,
}

# Uncomment to get the supported_models.py file contents
# print(repr(list(MODEL_CONFIG_MAPPING.keys())))
