import os
from typing import Optional

import numpy as np
import soxr

from ..Audio.Audio import load_audio
from ..Chinese.ChineseG2P import chinese_clean_g2p_and_norm
from ..Chinese.ZhBert import compute_bert_phone_features
from ..English.EnglishG2P import english_to_phones
from ..Japanese.JapaneseG2P import japanese_to_phones
from ..ModelManager import model_manager
from ..Utils.Constants import BERT_FEATURE_DIM
from ..Utils.Shared import context
from ..Utils.Utils import LRUCacheDict

# Import SV extraction for v2Pro/v2ProPlus
try:
    from .SpeakerVector import extract_sv_embedding
    _SV_AVAILABLE = True
except ImportError:
    _SV_AVAILABLE = False


class ReferenceAudio:
    _prompt_cache: dict[tuple[str, str, str], "ReferenceAudio"] = LRUCacheDict(
        capacity=int(os.getenv("Max_Cached_Reference_Audio", "10"))
    )

    def __new__(cls, prompt_wav: str, prompt_text: str, language: str = "auto", model_version: str = 'v2'):
        # Cache key includes model_version to avoid conflicts
        key = (prompt_wav, (language or "auto"), model_version)
        if key in cls._prompt_cache:
            instance = cls._prompt_cache[key]
            if instance.text != prompt_text or instance.language != language:
                instance.set_text(prompt_text, language)
            return instance

        instance = super().__new__(cls)
        cls._prompt_cache[key] = instance
        return instance

    def __init__(self, prompt_wav: str, prompt_text: str, language: str = "auto", model_version: str = 'v2'):
        if hasattr(self, "_initialized"):
            return

        self.text: str = prompt_text
        self.language: str = language or "auto"
        self.model_version: str = model_version
        self.phonemes_seq: Optional[np.ndarray] = None
        self.text_bert: Optional[np.ndarray] = None
        self.sv_emb: Optional[np.ndarray] = None
        self.set_text(prompt_text, language)

        self.audio_32k: Optional[np.ndarray] = load_audio(
            audio_path=prompt_wav,
            target_sampling_rate=32000,
        )
        audio_16k: np.ndarray = soxr.resample(self.audio_32k, 32000, 16000, quality="hq")
        
        # Extract SSL content (always needed)
        audio_16k_batch = np.expand_dims(audio_16k, axis=0)
        if not model_manager.cn_hubert:
            model_manager.load_cn_hubert()
        self.ssl_content: Optional[np.ndarray] = model_manager.cn_hubert.run(
            None, {"input_values": audio_16k_batch}
        )[0]
        
        # Extract speaker vector for v2Pro/v2ProPlus
        if model_version in ['v2Pro', 'v2ProPlus']:
            if _SV_AVAILABLE:
                self.sv_emb = extract_sv_embedding(audio_16k)  # Pass 1D array
                if self.sv_emb is None:
                    import logging
                    logging.getLogger(__name__).warning(
                        f"Failed to extract speaker embedding for {model_version}. "
                        f"Inference may not work correctly."
                    )
                else:
                    # Validate SV embedding shape
                    import logging
                    logger = logging.getLogger(__name__)
                    if self.sv_emb.shape != (1, 20480):
                        logger.error(
                            f"Invalid speaker embedding shape: {self.sv_emb.shape}, expected (1, 20480). "
                            f"Please check ERes2NetV2 model compatibility."
                        )
                        raise ValueError(f"Speaker embedding shape mismatch: {self.sv_emb.shape} != (1, 20480)")
                    logger.debug(f"✓ Speaker embedding extracted: shape={self.sv_emb.shape}, "
                               f"range=[{self.sv_emb.min():.3f}, {self.sv_emb.max():.3f}]")
            else:
                import logging
                logging.getLogger(__name__).warning(
                    f"Speaker vector extraction not available for {model_version}. "
                    f"Please ensure torchaudio is installed."
                )

        self._initialized = True

    def set_text(self, prompt_text: str, language: str = "auto") -> None:
        self.text = prompt_text
        lang = _decide_language(prompt_text, language)
        self.language = lang

        if lang == "en":
            ids = english_to_phones(prompt_text)
            word2ph: list[int] = []
            norm_text = ""
        elif lang == "zh":
            ids, word2ph, norm_text = chinese_clean_g2p_and_norm(prompt_text)
        else:
            ids = japanese_to_phones(prompt_text)
            word2ph = []
            norm_text = ""

        self.phonemes_seq = np.array([ids], dtype=np.int64)
        bert_matrix = _compute_reference_bert(lang, norm_text, word2ph, len(ids))
        self.text_bert = bert_matrix

        if lang in {"ja", "en", "zh"}:
            context.current_language = lang

    @classmethod
    def clear_cache(cls) -> None:
        cls._prompt_cache.clear()


def _decide_language(text: str, language: Optional[str]) -> str:
    lang = (language or "auto").lower()
    if lang == "auto":
        if _looks_english(text):
            return "en"
        if _looks_chinese(text):
            return "zh"
        return "ja"
    if lang in {"ja", "en", "zh"}:
        return lang
    return "ja"


def _compute_reference_bert(
    language: str, norm_text: str, word2ph: list[int], phone_len: int
) -> np.ndarray:
    if language == "zh" and phone_len:
        bert = compute_bert_phone_features(norm_text, word2ph)
        if bert.shape[0] == phone_len:
            return bert.astype(np.float32)
    return np.zeros((phone_len, BERT_FEATURE_DIM), dtype=np.float32)


def _looks_english(text: str) -> bool:
    ascii_letters = sum(ch.isascii() and ch.isalpha() for ch in text)
    non_ascii = sum(not ch.isascii() and not ch.isspace() for ch in text)
    return ascii_letters > 0 and ascii_letters >= non_ascii


def _looks_chinese(text: str) -> bool:
    return any("\u4e00" <= ch <= "\u9fff" for ch in text)


