"""
Command line interface for PromptSim.
"""

import argparse
import json
import sys
from typing import Dict, Any

import sys
import os
sys.path.insert(0, os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

from promptsim.client import PromptSim
from promptsim.exceptions import PromptSimError


def create_prompt_command(args):
    """Create a new prompt."""
    client = PromptSim(storage=args.storage, db_path=args.db_path)
    
    try:
        prompt = client.create_prompt(
            key=args.key,
            template=args.template,
            name=args.name,
            model=args.model,
            description=args.description or ""
        )
        print(f"Created prompt '{prompt.key}' version {prompt.version}")
    except PromptSimError as e:
        print(f"Error: {e}", file=sys.stderr)
        sys.exit(1)


def list_prompts_command(args):
    """List prompts."""
    client = PromptSim(storage=args.storage, db_path=args.db_path)
    
    try:
        prompts = client.list_prompts(args.key)
        if not prompts:
            print("No prompts found")
            return
        
        for prompt in prompts:
            status = "ACTIVE" if prompt.is_active else "INACTIVE"
            print(f"{prompt.key} v{prompt.version} ({status}) - {prompt.name}")
            if args.verbose:
                print(f"  Model: {prompt.model}")
                print(f"  Created: {prompt.created_at}")
                print(f"  Template: {prompt.template[:100]}...")
                print()
    except PromptSimError as e:
        print(f"Error: {e}", file=sys.stderr)
        sys.exit(1)


def complete_command(args):
    """Complete a prompt."""
    client = PromptSim(storage=args.storage, db_path=args.db_path)
    
    # Parse variables
    variables = {}
    if args.variables:
        try:
            variables = json.loads(args.variables)
        except json.JSONDecodeError:
            print("Error: Variables must be valid JSON", file=sys.stderr)
            sys.exit(1)
    
    try:
        response = client.complete(
            prompt_key=args.key,
            variables=variables,
            user_id=args.user_id
        )
        
        print("Response:")
        print(response.text)
        print()
        print(f"Cost: ${response.cost_usd:.6f}")
        print(f"Tokens: {response.tokens_used}")
        print(f"Latency: {response.latency_ms}ms")
        print(f"Model: {response.model}")
        
    except PromptSimError as e:
        print(f"Error: {e}", file=sys.stderr)
        sys.exit(1)


def cost_summary_command(args):
    """Show cost summary."""
    client = PromptSim(storage=args.storage, db_path=args.db_path)
    
    try:
        summary = client.get_cost_summary(args.key, args.days)
        
        print(f"Cost Summary (last {summary['days_analyzed']} days):")
        print(f"Total Cost: ${summary['total_cost']:.6f}")
        print(f"Total Executions: {summary['total_executions']}")
        print(f"Total Tokens: {summary['total_tokens']:,}")
        print(f"Avg Cost/Execution: ${summary['avg_cost_per_execution']:.6f}")
        print(f"Avg Tokens/Execution: {summary['avg_tokens_per_execution']:.1f}")
        
        if summary['cost_by_model']:
            print("\nCost by Model:")
            for model, data in summary['cost_by_model'].items():
                print(f"  {model}: ${data['cost']:.6f} ({data['executions']} executions)")
        
    except PromptSimError as e:
        print(f"Error: {e}", file=sys.stderr)
        sys.exit(1)


def export_command(args):
    """Export data."""
    client = PromptSim(storage=args.storage, db_path=args.db_path)
    
    try:
        data = client.export_data()
        
        if args.output:
            with open(args.output, 'w') as f:
                json.dump(data, f, indent=2)
            print(f"Data exported to {args.output}")
        else:
            print(json.dumps(data, indent=2))
            
    except PromptSimError as e:
        print(f"Error: {e}", file=sys.stderr)
        sys.exit(1)


def import_command(args):
    """Import data."""
    client = PromptSim(storage=args.storage, db_path=args.db_path)
    
    try:
        with open(args.file, 'r') as f:
            data = json.load(f)
        
        client.import_data(data)
        print(f"Data imported from {args.file}")
        
    except (PromptSimError, FileNotFoundError, json.JSONDecodeError) as e:
        print(f"Error: {e}", file=sys.stderr)
        sys.exit(1)


def main():
    """Main CLI entry point."""
    parser = argparse.ArgumentParser(description="PromptSim CLI")
    parser.add_argument("--storage", default="local", choices=["local", "files"], 
                       help="Storage backend")
    parser.add_argument("--db-path", default="./promptsim.db", 
                       help="SQLite database path")
    
    subparsers = parser.add_subparsers(dest="command", help="Commands")
    
    # Create prompt
    create_parser = subparsers.add_parser("create", help="Create a new prompt")
    create_parser.add_argument("key", help="Prompt key")
    create_parser.add_argument("template", help="Prompt template")
    create_parser.add_argument("--name", help="Prompt name")
    create_parser.add_argument("--model", default="gpt-3.5-turbo", help="OpenAI model")
    create_parser.add_argument("--description", help="Prompt description")
    create_parser.set_defaults(func=create_prompt_command)
    
    # List prompts
    list_parser = subparsers.add_parser("list", help="List prompts")
    list_parser.add_argument("--key", help="Filter by prompt key")
    list_parser.add_argument("--verbose", "-v", action="store_true", help="Verbose output")
    list_parser.set_defaults(func=list_prompts_command)
    
    # Complete prompt
    complete_parser = subparsers.add_parser("complete", help="Complete a prompt")
    complete_parser.add_argument("key", help="Prompt key")
    complete_parser.add_argument("--variables", help="Variables as JSON")
    complete_parser.add_argument("--user-id", help="User ID for A/B testing")
    complete_parser.set_defaults(func=complete_command)
    
    # Cost summary
    cost_parser = subparsers.add_parser("cost", help="Show cost summary")
    cost_parser.add_argument("--key", help="Filter by prompt key")
    cost_parser.add_argument("--days", type=int, default=30, help="Number of days")
    cost_parser.set_defaults(func=cost_summary_command)
    
    # Export data
    export_parser = subparsers.add_parser("export", help="Export data")
    export_parser.add_argument("--output", "-o", help="Output file (default: stdout)")
    export_parser.set_defaults(func=export_command)
    
    # Import data
    import_parser = subparsers.add_parser("import", help="Import data")
    import_parser.add_argument("file", help="Input file")
    import_parser.set_defaults(func=import_command)
    
    args = parser.parse_args()
    
    if not args.command:
        parser.print_help()
        sys.exit(1)
    
    args.func(args)


if __name__ == "__main__":
    main()