"""
Main PromptSim client.
"""

import hashlib
import os
from typing import Dict, Any, Optional, List, Union
from datetime import datetime

from .models import PromptTemplate, PromptExecution, PromptResponse, Experiment
from .storage import StorageBackend, SQLiteStorage, FileStorage
from .openai_client import OpenAIClient
from .exceptions import PromptSimError, PromptNotFoundError, ValidationError


class PromptSim:
    """Main PromptSim client for prompt versioning, A/B testing, and cost tracking."""
    
    def __init__(
        self,
        storage: str = "local",
        db_path: str = "./promptsim.db",
        data_dir: str = "./prompt_data",
        api_key: Optional[str] = None,
        host: Optional[str] = None,
        openai_api_key: Optional[str] = None
    ):
        """
        Initialize PromptSim client.
        
        Args:
            storage: Storage backend ("local", "files", "cloud")
            db_path: Path to SQLite database (for "local" storage)
            data_dir: Directory for JSON files (for "files" storage)
            api_key: API key for cloud storage
            host: Host URL for cloud storage
            openai_api_key: OpenAI API key
        """
        self.storage_type = storage
        
        # Initialize storage backend
        if storage == "local":
            self.storage = SQLiteStorage(db_path)
        elif storage == "files":
            self.storage = FileStorage(data_dir)
        elif storage == "cloud":
            # TODO: Implement cloud storage backend
            raise NotImplementedError("Cloud storage not yet implemented")
        else:
            raise ValueError(f"Unknown storage type: {storage}")
        
        # Initialize OpenAI client
        openai_key = openai_api_key or os.getenv("OPENAI_API_KEY")
        if openai_key:
            try:
                self.openai = OpenAIClient(openai_key)
            except Exception:
                # OpenAI dependencies not available
                self.openai = None
        else:
            self.openai = None
    
    def create_prompt(
        self,
        key: str,
        template: str,
        name: Optional[str] = None,
        model: str = "gpt-3.5-turbo",
        parameters: Optional[Dict[str, Any]] = None,
        tags: Optional[List[str]] = None,
        description: str = "",
        created_by: str = "system"
    ) -> PromptTemplate:
        """
        Create a new prompt template.
        
        Args:
            key: Unique key for the prompt
            template: Prompt template with {variable} placeholders
            name: Human-readable name
            model: OpenAI model to use
            parameters: Model parameters (temperature, max_tokens, etc.)
            tags: List of tags for organization
            description: Description of the prompt
            created_by: Creator identifier
            
        Returns:
            Created PromptTemplate
        """
        if not key or not template:
            raise ValidationError("Key and template are required")
        
        # Get next version number
        existing_prompts = self.storage.list_prompts(key)
        next_version = max([p.version for p in existing_prompts], default=0) + 1
        
        # Deactivate previous versions
        for prompt in existing_prompts:
            if prompt.is_active:
                prompt.is_active = False
                self.storage.save_prompt(prompt)
        
        # Create new prompt
        prompt = PromptTemplate(
            key=key,
            name=name or key.replace("_", " ").title(),
            template=template,
            model=model,
            version=next_version,
            parameters=parameters or {},
            tags=tags or [],
            description=description,
            created_by=created_by,
            is_active=True
        )
        
        self.storage.save_prompt(prompt)
        return prompt
    
    def get_prompt(self, key: str, version: Optional[int] = None) -> PromptTemplate:
        """
        Get a prompt template by key and version.
        
        Args:
            key: Prompt key
            version: Specific version (None for active version)
            
        Returns:
            PromptTemplate
        """
        return self.storage.get_prompt(key, version)
    
    def list_prompts(self, key: Optional[str] = None) -> List[PromptTemplate]:
        """
        List prompt templates.
        
        Args:
            key: Filter by specific key (None for all)
            
        Returns:
            List of PromptTemplate
        """
        return self.storage.list_prompts(key)
    
    def complete(
        self,
        prompt_key: str,
        variables: Optional[Dict[str, Any]] = None,
        user_id: Optional[str] = None,
        model: Optional[str] = None,
        temperature: Optional[float] = None,
        max_tokens: Optional[int] = None,
        experiment_id: Optional[str] = None,
        **kwargs
    ) -> PromptResponse:
        """
        Complete a prompt with variables.
        
        Args:
            prompt_key: Key of the prompt to use
            variables: Variables to substitute in the template
            user_id: User identifier for A/B testing
            model: Override model
            temperature: Override temperature
            max_tokens: Override max_tokens
            experiment_id: Experiment ID for A/B testing
            **kwargs: Additional parameters for OpenAI
            
        Returns:
            PromptResponse with completion and metadata
        """
        if not self.openai:
            raise PromptSimError("OpenAI API key not configured")
        
        variables = variables or {}
        
        # Get prompt (with A/B testing if experiment is active)
        prompt, variant_name = self._get_prompt_for_user(prompt_key, user_id, experiment_id)
        
        # Format template with variables
        try:
            formatted_prompt = prompt.template.format(**variables)
        except KeyError as e:
            raise ValidationError(f"Missing variable in template: {e}")
        
        # Use prompt parameters as defaults, allow overrides
        final_model = model or prompt.model
        final_temperature = temperature if temperature is not None else prompt.parameters.get("temperature", 0.7)
        final_max_tokens = max_tokens or prompt.parameters.get("max_tokens")
        
        # Call OpenAI
        result = self.openai.complete(
            prompt=formatted_prompt,
            model=final_model,
            temperature=final_temperature,
            max_tokens=final_max_tokens,
            **kwargs
        )
        
        # Create execution record
        execution = PromptExecution(
            prompt_id=prompt.id,
            prompt_key=prompt.key,
            prompt_version=prompt.version,
            user_id=user_id,
            input_variables=variables,
            input_text=formatted_prompt,
            output_text=result["text"],
            model=result["model"],
            tokens_used=result["tokens_used"],
            cost_usd=result["cost_usd"],
            latency_ms=result["latency_ms"],
            experiment_id=experiment_id,
            variant_name=variant_name,
            metadata=result.get("usage", {})
        )
        
        # Save execution
        self.storage.save_execution(execution)
        
        # Return response
        return PromptResponse(
            text=result["text"],
            cost_usd=result["cost_usd"],
            tokens_used=result["tokens_used"],
            latency_ms=result["latency_ms"],
            model=result["model"],
            prompt_version=prompt.version,
            execution_id=execution.id,
            metadata=result.get("usage", {})
        )
    
    def _get_prompt_for_user(
        self, 
        prompt_key: str, 
        user_id: Optional[str], 
        experiment_id: Optional[str]
    ) -> tuple[PromptTemplate, Optional[str]]:
        """
        Get prompt for user, considering A/B testing.
        
        Returns:
            (PromptTemplate, variant_name)
        """
        # TODO: Implement A/B testing logic
        # For now, just return the active prompt
        prompt = self.get_prompt(prompt_key)
        return prompt, None
    
    def get_executions(
        self, 
        prompt_key: Optional[str] = None, 
        limit: int = 100
    ) -> List[PromptExecution]:
        """
        Get prompt executions.
        
        Args:
            prompt_key: Filter by prompt key
            limit: Maximum number of executions to return
            
        Returns:
            List of PromptExecution
        """
        return self.storage.get_executions(prompt_key, limit)
    
    def get_cost_summary(
        self, 
        prompt_key: Optional[str] = None,
        days: int = 30
    ) -> Dict[str, Any]:
        """
        Get cost summary for prompts.
        
        Args:
            prompt_key: Filter by prompt key
            days: Number of days to analyze
            
        Returns:
            Cost summary dictionary
        """
        executions = self.get_executions(prompt_key, limit=10000)
        
        # Filter by date
        from datetime import timedelta
        cutoff_date = datetime.utcnow().replace(hour=0, minute=0, second=0, microsecond=0)
        cutoff_date = cutoff_date - timedelta(days=days)
        
        recent_executions = [
            e for e in executions 
            if e.executed_at >= cutoff_date
        ]
        
        if not recent_executions:
            return {
                "total_cost": 0.0,
                "total_executions": 0,
                "total_tokens": 0,
                "avg_cost_per_execution": 0.0,
                "avg_tokens_per_execution": 0.0,
                "cost_by_model": {},
                "executions_by_day": {},
                "days_analyzed": days
            }
        
        total_cost = sum(e.cost_usd for e in recent_executions)
        total_tokens = sum(e.tokens_used for e in recent_executions)
        total_executions = len(recent_executions)
        
        # Cost by model
        cost_by_model = {}
        for execution in recent_executions:
            model = execution.model
            if model not in cost_by_model:
                cost_by_model[model] = {"cost": 0.0, "executions": 0, "tokens": 0}
            cost_by_model[model]["cost"] += execution.cost_usd
            cost_by_model[model]["executions"] += 1
            cost_by_model[model]["tokens"] += execution.tokens_used
        
        # Executions by day
        executions_by_day = {}
        for execution in recent_executions:
            day = execution.executed_at.date().isoformat()
            if day not in executions_by_day:
                executions_by_day[day] = {"executions": 0, "cost": 0.0, "tokens": 0}
            executions_by_day[day]["executions"] += 1
            executions_by_day[day]["cost"] += execution.cost_usd
            executions_by_day[day]["tokens"] += execution.tokens_used
        
        return {
            "total_cost": total_cost,
            "total_executions": total_executions,
            "total_tokens": total_tokens,
            "avg_cost_per_execution": total_cost / total_executions,
            "avg_tokens_per_execution": total_tokens / total_executions,
            "cost_by_model": cost_by_model,
            "executions_by_day": executions_by_day,
            "days_analyzed": days
        }
    
    def export_data(self) -> Dict[str, Any]:
        """
        Export all data for migration/backup.
        
        Returns:
            Dictionary with all prompts and executions
        """
        prompts = self.list_prompts()
        executions = self.get_executions(limit=100000)  # Get all executions
        
        return {
            "prompts": [
                {
                    "id": p.id,
                    "key": p.key,
                    "name": p.name,
                    "template": p.template,
                    "model": p.model,
                    "version": p.version,
                    "parameters": p.parameters,
                    "tags": p.tags,
                    "is_active": p.is_active,
                    "created_at": p.created_at.isoformat(),
                    "created_by": p.created_by,
                    "description": p.description
                }
                for p in prompts
            ],
            "executions": [
                {
                    "id": e.id,
                    "prompt_id": e.prompt_id,
                    "prompt_key": e.prompt_key,
                    "prompt_version": e.prompt_version,
                    "user_id": e.user_id,
                    "input_variables": e.input_variables,
                    "input_text": e.input_text,
                    "output_text": e.output_text,
                    "model": e.model,
                    "tokens_used": e.tokens_used,
                    "cost_usd": e.cost_usd,
                    "latency_ms": e.latency_ms,
                    "error": e.error,
                    "metadata": e.metadata,
                    "executed_at": e.executed_at.isoformat(),
                    "experiment_id": e.experiment_id,
                    "variant_name": e.variant_name
                }
                for e in executions
            ],
            "exported_at": datetime.utcnow().isoformat(),
            "version": "0.1.0"
        }
    
    def import_data(self, data: Dict[str, Any]) -> None:
        """
        Import data from export.
        
        Args:
            data: Data dictionary from export_data()
        """
        # Import prompts
        for prompt_data in data.get("prompts", []):
            prompt = PromptTemplate(
                id=prompt_data["id"],
                key=prompt_data["key"],
                name=prompt_data["name"],
                template=prompt_data["template"],
                model=prompt_data["model"],
                version=prompt_data["version"],
                parameters=prompt_data.get("parameters", {}),
                tags=prompt_data.get("tags", []),
                is_active=prompt_data.get("is_active", True),
                created_at=datetime.fromisoformat(prompt_data["created_at"]),
                created_by=prompt_data.get("created_by", "system"),
                description=prompt_data.get("description", "")
            )
            self.storage.save_prompt(prompt)
        
        # Import executions
        for exec_data in data.get("executions", []):
            execution = PromptExecution(
                id=exec_data["id"],
                prompt_id=exec_data["prompt_id"],
                prompt_key=exec_data["prompt_key"],
                prompt_version=exec_data["prompt_version"],
                user_id=exec_data.get("user_id"),
                input_variables=exec_data.get("input_variables", {}),
                input_text=exec_data["input_text"],
                output_text=exec_data["output_text"],
                model=exec_data["model"],
                tokens_used=exec_data["tokens_used"],
                cost_usd=exec_data["cost_usd"],
                latency_ms=exec_data["latency_ms"],
                error=exec_data.get("error"),
                metadata=exec_data.get("metadata", {}),
                executed_at=datetime.fromisoformat(exec_data["executed_at"]),
                experiment_id=exec_data.get("experiment_id"),
                variant_name=exec_data.get("variant_name")
            )
            self.storage.save_execution(execution)