"""
Data models for PromptSim.
"""

from datetime import datetime
from typing import Dict, Any, Optional, List
from dataclasses import dataclass, field
from uuid import uuid4


@dataclass
class PromptTemplate:
    """A versioned prompt template."""
    
    id: str = field(default_factory=lambda: str(uuid4()))
    key: str = ""
    name: str = ""
    template: str = ""
    model: str = "gpt-3.5-turbo"
    version: int = 1
    parameters: Dict[str, Any] = field(default_factory=dict)
    tags: List[str] = field(default_factory=list)
    is_active: bool = True
    created_at: datetime = field(default_factory=datetime.utcnow)
    created_by: str = "system"
    description: str = ""
    
    def __post_init__(self):
        """Validate and set defaults after initialization."""
        if not self.key:
            raise ValueError("Prompt key is required")
        if not self.template:
            raise ValueError("Prompt template is required")
        if not self.name:
            self.name = self.key.replace("_", " ").title()


@dataclass
class PromptExecution:
    """A record of a prompt execution."""
    
    id: str = field(default_factory=lambda: str(uuid4()))
    prompt_id: str = ""
    prompt_key: str = ""
    prompt_version: int = 1
    user_id: Optional[str] = None
    input_variables: Dict[str, Any] = field(default_factory=dict)
    input_text: str = ""
    output_text: str = ""
    model: str = ""
    tokens_used: int = 0
    cost_usd: float = 0.0
    latency_ms: int = 0
    error: Optional[str] = None
    metadata: Dict[str, Any] = field(default_factory=dict)
    executed_at: datetime = field(default_factory=datetime.utcnow)
    experiment_id: Optional[str] = None
    variant_name: Optional[str] = None


@dataclass
class PromptResponse:
    """Response from a prompt execution."""
    
    text: str
    cost_usd: float
    tokens_used: int
    latency_ms: int
    model: str
    prompt_version: int
    execution_id: str
    metadata: Dict[str, Any] = field(default_factory=dict)


@dataclass
class Experiment:
    """An A/B test experiment."""
    
    id: str = field(default_factory=lambda: str(uuid4()))
    name: str = ""
    prompt_key: str = ""
    description: str = ""
    variants: List[Dict[str, Any]] = field(default_factory=list)
    status: str = "draft"  # draft, running, completed, paused
    traffic_allocation: Dict[str, float] = field(default_factory=dict)
    start_date: Optional[datetime] = None
    end_date: Optional[datetime] = None
    created_at: datetime = field(default_factory=datetime.utcnow)
    created_by: str = "system"
    success_metrics: List[str] = field(default_factory=list)


@dataclass
class BusinessMetric:
    """A business metric linked to prompt executions."""
    
    id: str = field(default_factory=lambda: str(uuid4()))
    execution_id: str = ""
    user_id: str = ""
    metric_name: str = ""
    metric_value: float = 0.0
    metric_type: str = "conversion"  # conversion, revenue, satisfaction, etc.
    recorded_at: datetime = field(default_factory=datetime.utcnow)
    metadata: Dict[str, Any] = field(default_factory=dict)