#!/usr/bin/env python3
"""
Test runner for PromptSim.

This script can be run after installing PromptSim to verify everything works correctly.
"""

import sys
import os
import subprocess
import tempfile
from pathlib import Path

def run_basic_tests():
    """Run basic functionality tests without pytest."""
    print("=== PromptSim Basic Functionality Tests ===")
    print("Testing core functionality without external dependencies...\n")
    
    try:
        from promptsim import PromptSim
        from promptsim.models import PromptTemplate, PromptExecution
        from promptsim.exceptions import PromptNotFoundError, ValidationError
        print("✓ Import successful")
    except ImportError as e:
        print(f"✗ Import failed: {e}")
        return False
    
    # Test 1: Basic prompt creation and retrieval
    print("1. Testing prompt creation and retrieval...")
    try:
        with tempfile.NamedTemporaryFile(suffix=".db", delete=False) as tmp:
            db_path = tmp.name
        
        client = PromptSim(storage="local", db_path=db_path)
        
        # Create prompt
        prompt = client.create_prompt(
            key="test_prompt",
            template="Hello {name}!",
            name="Test Prompt"
        )
        
        assert prompt.key == "test_prompt"
        assert prompt.version == 1
        assert prompt.is_active == True
        
        # Retrieve prompt
        retrieved = client.get_prompt("test_prompt")
        assert retrieved.template == "Hello {name}!"
        
        os.unlink(db_path)
        print("✓ Prompt creation and retrieval works")
        
    except Exception as e:
        print(f"✗ Prompt creation failed: {e}")
        return False
    
    # Test 2: Versioning
    print("2. Testing prompt versioning...")
    try:
        with tempfile.NamedTemporaryFile(suffix=".db", delete=False) as tmp:
            db_path = tmp.name
        
        client = PromptSim(storage="local", db_path=db_path)
        
        # Create v1
        v1 = client.create_prompt(key="test", template="Version 1")
        assert v1.version == 1
        
        # Create v2
        v2 = client.create_prompt(key="test", template="Version 2")
        assert v2.version == 2
        
        # Check active version
        active = client.get_prompt("test")
        assert active.version == 2
        assert active.template == "Version 2"
        
        # Check v1 is inactive
        v1_retrieved = client.get_prompt("test", version=1)
        assert v1_retrieved.is_active == False
        
        os.unlink(db_path)
        print("✓ Prompt versioning works")
        
    except Exception as e:
        print(f"✗ Versioning failed: {e}")
        return False
    
    # Test 3: File storage
    print("3. Testing file storage...")
    try:
        with tempfile.TemporaryDirectory() as tmp_dir:
            client = PromptSim(storage="files", data_dir=tmp_dir)
            
            prompt = client.create_prompt(
                key="file_test",
                template="File storage test"
            )
            
            retrieved = client.get_prompt("file_test")
            assert retrieved.template == "File storage test"
            
        print("✓ File storage works")
        
    except Exception as e:
        print(f"✗ File storage failed: {e}")
        return False
    
    # Test 4: Data export/import
    print("4. Testing data export/import...")
    try:
        with tempfile.NamedTemporaryFile(suffix=".db", delete=False) as tmp1:
            db_path1 = tmp1.name
        with tempfile.NamedTemporaryFile(suffix=".db", delete=False) as tmp2:
            db_path2 = tmp2.name
        
        # Create data in first client
        client1 = PromptSim(storage="local", db_path=db_path1)
        client1.create_prompt(key="export_test", template="Export test")
        
        # Export data
        data = client1.export_data()
        assert len(data["prompts"]) == 1
        
        # Import to second client
        client2 = PromptSim(storage="local", db_path=db_path2)
        client2.import_data(data)
        
        # Verify import
        imported = client2.get_prompt("export_test")
        assert imported.template == "Export test"
        
        os.unlink(db_path1)
        os.unlink(db_path2)
        print("✓ Data export/import works")
        
    except Exception as e:
        print(f"✗ Export/import failed: {e}")
        return False
    
    # Test 5: Cost summary
    print("5. Testing cost summary...")
    try:
        with tempfile.NamedTemporaryFile(suffix=".db", delete=False) as tmp:
            db_path = tmp.name
        
        client = PromptSim(storage="local", db_path=db_path)
        
        # Test empty summary
        summary = client.get_cost_summary()
        assert summary["total_cost"] == 0.0
        assert summary["total_executions"] == 0
        
        os.unlink(db_path)
        print("✓ Cost summary works")
        
    except Exception as e:
        print(f"✗ Cost summary failed: {e}")
        return False
    
    # Test 6: Error handling
    print("6. Testing error handling...")
    try:
        with tempfile.NamedTemporaryFile(suffix=".db", delete=False) as tmp:
            db_path = tmp.name
        
        client = PromptSim(storage="local", db_path=db_path)
        
        # Test prompt not found
        try:
            client.get_prompt("nonexistent")
            assert False, "Should have raised PromptNotFoundError"
        except PromptNotFoundError:
            pass  # Expected
        
        # Test validation error
        try:
            client.create_prompt(key="", template="test")
            assert False, "Should have raised ValidationError"
        except ValidationError:
            pass  # Expected
        
        os.unlink(db_path)
        print("✓ Error handling works")
        
    except Exception as e:
        print(f"✗ Error handling failed: {e}")
        return False
    
    print("\n=== All Basic Tests Passed! ===")
    print("PromptSim is working correctly.")
    return True


def run_pytest_tests():
    """Run full test suite with pytest if available."""
    print("\n=== Running Full Test Suite ===")
    
    try:
        import pytest
    except ImportError:
        print("pytest not available. Install with: pip install pytest")
        print("Skipping full test suite.")
        return True
    
    # Find tests directory
    current_dir = Path(__file__).parent
    tests_dir = current_dir / "tests"
    
    if not tests_dir.exists():
        print("Tests directory not found. Running basic tests only.")
        return True
    
    print("Running pytest...")
    try:
        result = subprocess.run([
            sys.executable, "-m", "pytest", 
            str(tests_dir), 
            "-v", 
            "--tb=short"
        ], capture_output=True, text=True)
        
        print(result.stdout)
        if result.stderr:
            print("STDERR:", result.stderr)
        
        if result.returncode == 0:
            print("✓ All pytest tests passed!")
            return True
        else:
            print("✗ Some pytest tests failed.")
            return False
            
    except Exception as e:
        print(f"Error running pytest: {e}")
        return False


def main():
    """Main test runner."""
    print("PromptSim Test Runner")
    print("====================")
    print("This script verifies that PromptSim is installed and working correctly.\n")
    
    # Run basic tests (always)
    basic_success = run_basic_tests()
    
    if not basic_success:
        print("\n❌ Basic tests failed. Please check your installation.")
        sys.exit(1)
    
    # Run full test suite if pytest is available
    pytest_success = run_pytest_tests()
    
    if basic_success and pytest_success:
        print("\n🎉 All tests passed! PromptSim is ready to use.")
        print("\nNext steps:")
        print("1. Set your OpenAI API key: export OPENAI_API_KEY='your-key-here'")
        print("2. Try the examples: python -c \"from promptsim import PromptSim; print('Ready!')\"")
        print("3. Check the documentation for usage examples")
    else:
        print("\n⚠️  Some tests failed, but basic functionality works.")
        sys.exit(1)


if __name__ == "__main__":
    main()