"""
Test PromptSim client.
"""

import pytest
import tempfile
import os
from unittest.mock import Mock, patch
from promptsim import PromptSim
from promptsim.models import PromptTemplate, PromptResponse
from promptsim.exceptions import PromptSimError, PromptNotFoundError, ValidationError


class TestPromptSimClient:
    """Test PromptSim client functionality."""
    
    @pytest.fixture
    def client(self):
        """Create PromptSim client with temporary storage."""
        with tempfile.NamedTemporaryFile(suffix=".db", delete=False) as tmp:
            db_path = tmp.name
        
        client = PromptSim(storage="local", db_path=db_path)
        yield client
        
        # Cleanup
        os.unlink(db_path)
    
    @pytest.fixture
    def file_client(self):
        """Create PromptSim client with file storage."""
        with tempfile.TemporaryDirectory() as tmp_dir:
            client = PromptSim(storage="files", data_dir=tmp_dir)
            yield client
    
    def test_create_prompt(self, client):
        """Test creating a prompt."""
        prompt = client.create_prompt(
            key="test_prompt",
            template="Hello {name}!",
            name="Test Prompt",
            model="gpt-3.5-turbo",
            description="A test prompt"
        )
        
        assert prompt.key == "test_prompt"
        assert prompt.template == "Hello {name}!"
        assert prompt.name == "Test Prompt"
        assert prompt.model == "gpt-3.5-turbo"
        assert prompt.version == 1
        assert prompt.is_active == True
        assert prompt.description == "A test prompt"
    
    def test_create_prompt_validation(self, client):
        """Test prompt creation validation."""
        # Missing key
        with pytest.raises(ValidationError):
            client.create_prompt(key="", template="Hello!")
        
        # Missing template
        with pytest.raises(ValidationError):
            client.create_prompt(key="test", template="")
    
    def test_prompt_versioning(self, client):
        """Test prompt versioning."""
        # Create version 1
        v1 = client.create_prompt(
            key="test_prompt",
            template="Hello {name}!",
            name="Version 1"
        )
        assert v1.version == 1
        assert v1.is_active == True
        
        # Create version 2
        v2 = client.create_prompt(
            key="test_prompt",
            template="Hi {name}!",
            name="Version 2"
        )
        assert v2.version == 2
        assert v2.is_active == True
        
        # Check that v1 is now inactive
        v1_retrieved = client.get_prompt("test_prompt", version=1)
        assert v1_retrieved.is_active == False
        
        # Active version should be v2
        active = client.get_prompt("test_prompt")
        assert active.version == 2
        assert active.template == "Hi {name}!"
    
    def test_get_prompt(self, client):
        """Test getting prompts."""
        # Create a prompt
        original = client.create_prompt(
            key="test_prompt",
            template="Hello {name}!"
        )
        
        # Get active version
        retrieved = client.get_prompt("test_prompt")
        assert retrieved.key == original.key
        assert retrieved.template == original.template
        
        # Get specific version
        specific = client.get_prompt("test_prompt", version=1)
        assert specific.version == 1
    
    def test_get_nonexistent_prompt(self, client):
        """Test getting non-existent prompt."""
        with pytest.raises(PromptNotFoundError):
            client.get_prompt("nonexistent")
    
    def test_list_prompts(self, client):
        """Test listing prompts."""
        # Create multiple prompts
        client.create_prompt(key="prompt1", template="Template 1")
        client.create_prompt(key="prompt2", template="Template 2")
        client.create_prompt(key="prompt1", template="Template 1 v2")  # New version
        
        # List all prompts
        all_prompts = client.list_prompts()
        assert len(all_prompts) == 3  # 2 versions of prompt1 + 1 version of prompt2
        
        # List specific prompt
        prompt1_versions = client.list_prompts("prompt1")
        assert len(prompt1_versions) == 2
        assert all(p.key == "prompt1" for p in prompt1_versions)
    
    def test_complete_without_openai(self, client):
        """Test complete method without OpenAI configured."""
        client.create_prompt(
            key="test_prompt",
            template="Hello {name}!"
        )
        
        with pytest.raises(PromptSimError, match="OpenAI API key not configured"):
            client.complete(
                prompt_key="test_prompt",
                variables={"name": "Alice"}
            )
    
    @patch('promptsim.client.OpenAIClient')
    def test_complete_with_mock_openai(self, mock_openai_class, client):
        """Test complete method with mocked OpenAI."""
        # Setup mock
        mock_openai = Mock()
        mock_openai.complete.return_value = {
            "text": "Hello Alice!",
            "tokens_used": 10,
            "cost_usd": 0.00002,
            "latency_ms": 500,
            "model": "gpt-3.5-turbo",
            "usage": {"prompt_tokens": 5, "completion_tokens": 5}
        }
        mock_openai_class.return_value = mock_openai
        
        # Create client with OpenAI key
        client.openai = mock_openai
        
        # Create prompt
        client.create_prompt(
            key="test_prompt",
            template="Hello {name}!",
            model="gpt-3.5-turbo"
        )
        
        # Complete prompt
        response = client.complete(
            prompt_key="test_prompt",
            variables={"name": "Alice"},
            user_id="user123"
        )
        
        assert isinstance(response, PromptResponse)
        assert response.text == "Hello Alice!"
        assert response.tokens_used == 10
        assert response.cost_usd == 0.00002
        assert response.latency_ms == 500
        assert response.model == "gpt-3.5-turbo"
        assert response.prompt_version == 1
        
        # Verify OpenAI was called correctly
        mock_openai.complete.assert_called_once()
        call_args = mock_openai.complete.call_args
        assert "Hello Alice!" in call_args[1]["prompt"]
    
    def test_complete_missing_variables(self, client):
        """Test complete with missing template variables."""
        client.create_prompt(
            key="test_prompt",
            template="Hello {name} from {location}!"
        )
        
        # Mock OpenAI to avoid actual API call
        client.openai = Mock()
        
        with pytest.raises(ValidationError, match="Missing variable in template"):
            client.complete(
                prompt_key="test_prompt",
                variables={"name": "Alice"}  # Missing 'location'
            )
    
    def test_get_executions(self, client):
        """Test getting executions."""
        # Initially no executions
        executions = client.get_executions()
        assert len(executions) == 0
        
        # Create some mock executions by directly using storage
        from promptsim.models import PromptExecution
        
        exec1 = PromptExecution(
            prompt_key="prompt1",
            input_text="Hello!",
            output_text="Hi!",
            model="gpt-3.5-turbo",
            tokens_used=5,
            cost_usd=0.00001,
            latency_ms=200
        )
        
        exec2 = PromptExecution(
            prompt_key="prompt2",
            input_text="Goodbye!",
            output_text="Bye!",
            model="gpt-3.5-turbo",
            tokens_used=8,
            cost_usd=0.00002,
            latency_ms=300
        )
        
        client.storage.save_execution(exec1)
        client.storage.save_execution(exec2)
        
        # Get all executions
        all_executions = client.get_executions()
        assert len(all_executions) == 2
        
        # Get executions for specific prompt
        prompt1_executions = client.get_executions("prompt1")
        assert len(prompt1_executions) == 1
        assert prompt1_executions[0].prompt_key == "prompt1"
    
    def test_cost_summary_empty(self, client):
        """Test cost summary with no executions."""
        summary = client.get_cost_summary()
        
        assert summary["total_cost"] == 0.0
        assert summary["total_executions"] == 0
        assert summary["total_tokens"] == 0
        assert summary["avg_cost_per_execution"] == 0.0
        assert summary["avg_tokens_per_execution"] == 0.0
        assert summary["cost_by_model"] == {}
        assert summary["executions_by_day"] == {}
        assert summary["days_analyzed"] == 30
    
    def test_cost_summary_with_data(self, client):
        """Test cost summary with execution data."""
        from promptsim.models import PromptExecution
        from datetime import datetime
        
        # Create mock executions
        exec1 = PromptExecution(
            prompt_key="prompt1",
            input_text="Hello!",
            output_text="Hi!",
            model="gpt-3.5-turbo",
            tokens_used=10,
            cost_usd=0.00002,
            latency_ms=200,
            executed_at=datetime.utcnow()
        )
        
        exec2 = PromptExecution(
            prompt_key="prompt1",
            input_text="How are you?",
            output_text="I'm good!",
            model="gpt-4",
            tokens_used=15,
            cost_usd=0.00005,
            latency_ms=300,
            executed_at=datetime.utcnow()
        )
        
        client.storage.save_execution(exec1)
        client.storage.save_execution(exec2)
        
        # Get cost summary
        summary = client.get_cost_summary()
        
        assert summary["total_cost"] == 0.00007
        assert summary["total_executions"] == 2
        assert summary["total_tokens"] == 25
        assert summary["avg_cost_per_execution"] == 0.000035
        assert summary["avg_tokens_per_execution"] == 12.5
        
        # Check cost by model
        assert "gpt-3.5-turbo" in summary["cost_by_model"]
        assert "gpt-4" in summary["cost_by_model"]
        assert summary["cost_by_model"]["gpt-3.5-turbo"]["cost"] == 0.00002
        assert summary["cost_by_model"]["gpt-4"]["cost"] == 0.00005
    
    def test_export_data(self, client):
        """Test data export."""
        # Create some data
        client.create_prompt(
            key="test_prompt",
            template="Hello {name}!",
            name="Test Prompt"
        )
        
        # Export data
        data = client.export_data()
        
        assert "prompts" in data
        assert "executions" in data
        assert "exported_at" in data
        assert "version" in data
        
        assert len(data["prompts"]) == 1
        assert data["prompts"][0]["key"] == "test_prompt"
        assert data["prompts"][0]["template"] == "Hello {name}!"
        assert len(data["executions"]) == 0
    
    def test_import_data(self, client):
        """Test data import."""
        # Create export data
        export_data = {
            "prompts": [
                {
                    "id": "test-id",
                    "key": "imported_prompt",
                    "name": "Imported Prompt",
                    "template": "Hello {name}!",
                    "model": "gpt-3.5-turbo",
                    "version": 1,
                    "parameters": {},
                    "tags": [],
                    "is_active": True,
                    "created_at": "2024-01-01T00:00:00",
                    "created_by": "system",
                    "description": "Imported prompt"
                }
            ],
            "executions": [],
            "exported_at": "2024-01-01T00:00:00",
            "version": "0.1.0"
        }
        
        # Import data
        client.import_data(export_data)
        
        # Verify import
        prompts = client.list_prompts()
        assert len(prompts) == 1
        assert prompts[0].key == "imported_prompt"
        assert prompts[0].name == "Imported Prompt"
        assert prompts[0].template == "Hello {name}!"
    
    def test_file_storage_client(self, file_client):
        """Test client with file storage."""
        # Create prompt
        prompt = file_client.create_prompt(
            key="file_prompt",
            template="Hello from file storage!"
        )
        
        assert prompt.key == "file_prompt"
        
        # Retrieve prompt
        retrieved = file_client.get_prompt("file_prompt")
        assert retrieved.template == "Hello from file storage!"
        
        # List prompts
        prompts = file_client.list_prompts()
        assert len(prompts) == 1
        assert prompts[0].key == "file_prompt"