"""
Integration tests for PromptSim.
"""

import pytest
import tempfile
import os
import json
from promptsim import PromptSim


class TestIntegration:
    """Integration tests for complete workflows."""
    
    @pytest.fixture
    def client(self):
        """Create PromptSim client with temporary storage."""
        with tempfile.NamedTemporaryFile(suffix=".db", delete=False) as tmp:
            db_path = tmp.name
        
        client = PromptSim(storage="local", db_path=db_path)
        yield client
        
        # Cleanup
        os.unlink(db_path)
    
    def test_complete_workflow(self, client):
        """Test complete prompt management workflow."""
        # 1. Create initial prompt
        prompt_v1 = client.create_prompt(
            key="welcome_message",
            template="Hello {name}! Welcome to {product}.",
            name="Welcome Message",
            description="Initial welcome message"
        )
        
        assert prompt_v1.version == 1
        assert prompt_v1.is_active == True
        
        # 2. Create improved version
        prompt_v2 = client.create_prompt(
            key="welcome_message",
            template="Hi {name}! 👋 Welcome to {product}! How can I help you today?",
            name="Welcome Message v2",
            description="Improved welcome message with emoji"
        )
        
        assert prompt_v2.version == 2
        assert prompt_v2.is_active == True
        
        # 3. Verify v1 is now inactive
        v1_retrieved = client.get_prompt("welcome_message", version=1)
        assert v1_retrieved.is_active == False
        
        # 4. Active version should be v2
        active = client.get_prompt("welcome_message")
        assert active.version == 2
        assert "👋" in active.template
        
        # 5. List all versions
        versions = client.list_prompts("welcome_message")
        assert len(versions) == 2
        
        # 6. Export all data
        export_data = client.export_data()
        assert len(export_data["prompts"]) == 2
        assert export_data["prompts"][0]["key"] == "welcome_message"
        assert export_data["prompts"][1]["key"] == "welcome_message"
        
        # 7. Create new client and import data
        with tempfile.NamedTemporaryFile(suffix=".db", delete=False) as tmp2:
            db_path2 = tmp2.name
        
        try:
            client2 = PromptSim(storage="local", db_path=db_path2)
            client2.import_data(export_data)
            
            # Verify import worked
            imported_prompts = client2.list_prompts()
            assert len(imported_prompts) == 2
            
            active_imported = client2.get_prompt("welcome_message")
            assert active_imported.version == 2
            assert "👋" in active_imported.template
            
        finally:
            os.unlink(db_path2)
    
    def test_multi_prompt_workflow(self, client):
        """Test workflow with multiple different prompts."""
        # Create different types of prompts
        prompts_to_create = [
            {
                "key": "customer_support",
                "template": "You are a helpful customer support agent. User question: {question}",
                "name": "Customer Support",
                "model": "gpt-3.5-turbo"
            },
            {
                "key": "sales_assistant",
                "template": "You are a sales assistant for {product}. Customer inquiry: {inquiry}",
                "name": "Sales Assistant",
                "model": "gpt-4"
            },
            {
                "key": "content_writer",
                "template": "Write a {content_type} about {topic} in {tone} tone.",
                "name": "Content Writer",
                "model": "gpt-3.5-turbo"
            }
        ]
        
        created_prompts = []
        for prompt_data in prompts_to_create:
            prompt = client.create_prompt(**prompt_data)
            created_prompts.append(prompt)
        
        # Verify all prompts were created
        all_prompts = client.list_prompts()
        assert len(all_prompts) == 3
        
        # Verify each prompt can be retrieved
        for original in created_prompts:
            retrieved = client.get_prompt(original.key)
            assert retrieved.key == original.key
            assert retrieved.template == original.template
            assert retrieved.model == original.model
        
        # Create new versions for some prompts
        client.create_prompt(
            key="customer_support",
            template="You are a friendly customer support agent. Question: {question}. Please be helpful and professional.",
            name="Customer Support v2"
        )
        
        # Verify versioning worked
        cs_versions = client.list_prompts("customer_support")
        assert len(cs_versions) == 2
        
        active_cs = client.get_prompt("customer_support")
        assert active_cs.version == 2
        assert "friendly" in active_cs.template
    
    def test_storage_backend_compatibility(self):
        """Test compatibility between different storage backends."""
        # Create data with SQLite storage
        with tempfile.NamedTemporaryFile(suffix=".db", delete=False) as tmp:
            db_path = tmp.name
        
        try:
            sqlite_client = PromptSim(storage="local", db_path=db_path)
            
            # Create some prompts
            sqlite_client.create_prompt(
                key="test_prompt",
                template="Hello {name}!",
                name="Test Prompt"
            )
            
            # Export data
            export_data = sqlite_client.export_data()
            
        finally:
            os.unlink(db_path)
        
        # Import data into file storage
        with tempfile.TemporaryDirectory() as tmp_dir:
            file_client = PromptSim(storage="files", data_dir=tmp_dir)
            file_client.import_data(export_data)
            
            # Verify data was imported correctly
            prompts = file_client.list_prompts()
            assert len(prompts) == 1
            assert prompts[0].key == "test_prompt"
            assert prompts[0].template == "Hello {name}!"
            
            # Export from file storage
            file_export = file_client.export_data()
            
            # Data should be equivalent
            assert len(file_export["prompts"]) == len(export_data["prompts"])
            assert file_export["prompts"][0]["key"] == export_data["prompts"][0]["key"]
    
    def test_cost_tracking_workflow(self, client):
        """Test cost tracking and analysis workflow."""
        from promptsim.models import PromptExecution
        from datetime import datetime, timedelta
        
        # Create a prompt
        client.create_prompt(
            key="test_prompt",
            template="Hello {name}!",
            model="gpt-3.5-turbo"
        )
        
        # Simulate some executions over different days
        base_date = datetime.utcnow()
        
        executions = [
            PromptExecution(
                prompt_key="test_prompt",
                input_text="Hello Alice!",
                output_text="Hi Alice!",
                model="gpt-3.5-turbo",
                tokens_used=10,
                cost_usd=0.00002,
                latency_ms=200,
                executed_at=base_date
            ),
            PromptExecution(
                prompt_key="test_prompt",
                input_text="Hello Bob!",
                output_text="Hi Bob!",
                model="gpt-3.5-turbo",
                tokens_used=12,
                cost_usd=0.000024,
                latency_ms=250,
                executed_at=base_date - timedelta(hours=1)
            ),
            PromptExecution(
                prompt_key="test_prompt",
                input_text="Hello Charlie!",
                output_text="Hi Charlie!",
                model="gpt-4",
                tokens_used=8,
                cost_usd=0.0001,
                latency_ms=300,
                executed_at=base_date - timedelta(days=1)
            )
        ]
        
        # Save executions
        for execution in executions:
            client.storage.save_execution(execution)
        
        # Get cost summary
        summary = client.get_cost_summary(days=7)
        
        assert summary["total_executions"] == 3
        assert summary["total_tokens"] == 30
        assert summary["total_cost"] == 0.000144  # Sum of all costs
        
        # Check cost by model
        assert "gpt-3.5-turbo" in summary["cost_by_model"]
        assert "gpt-4" in summary["cost_by_model"]
        assert summary["cost_by_model"]["gpt-3.5-turbo"]["executions"] == 2
        assert summary["cost_by_model"]["gpt-4"]["executions"] == 1
        
        # Get executions for specific prompt
        prompt_executions = client.get_executions("test_prompt")
        assert len(prompt_executions) == 3
        
        # Verify executions are sorted by date (newest first)
        assert prompt_executions[0].executed_at >= prompt_executions[1].executed_at
        assert prompt_executions[1].executed_at >= prompt_executions[2].executed_at