"""
Test data models.
"""

import pytest
from datetime import datetime
from promptsim.models import PromptTemplate, PromptExecution, PromptResponse, Experiment, BusinessMetric


class TestPromptTemplate:
    """Test PromptTemplate model."""
    
    def test_create_prompt_template(self):
        """Test creating a prompt template."""
        prompt = PromptTemplate(
            key="test_prompt",
            template="Hello {name}!",
            model="gpt-3.5-turbo"
        )
        
        assert prompt.key == "test_prompt"
        assert prompt.template == "Hello {name}!"
        assert prompt.model == "gpt-3.5-turbo"
        assert prompt.version == 1
        assert prompt.is_active == True
        assert prompt.name == "Test Prompt"  # Auto-generated from key
        assert isinstance(prompt.created_at, datetime)
        assert len(prompt.id) > 0
    
    def test_prompt_template_validation(self):
        """Test prompt template validation."""
        # Missing key should raise error
        with pytest.raises(ValueError, match="Prompt key is required"):
            PromptTemplate(key="", template="Hello!")
        
        # Missing template should raise error
        with pytest.raises(ValueError, match="Prompt template is required"):
            PromptTemplate(key="test", template="")
    
    def test_prompt_template_defaults(self):
        """Test default values."""
        prompt = PromptTemplate(
            key="test_prompt",
            template="Hello {name}!"
        )
        
        assert prompt.model == "gpt-3.5-turbo"
        assert prompt.version == 1
        assert prompt.parameters == {}
        assert prompt.tags == []
        assert prompt.is_active == True
        assert prompt.created_by == "system"
        assert prompt.description == ""


class TestPromptExecution:
    """Test PromptExecution model."""
    
    def test_create_execution(self):
        """Test creating a prompt execution."""
        execution = PromptExecution(
            prompt_id="prompt-123",
            prompt_key="test_prompt",
            prompt_version=1,
            user_id="user-456",
            input_text="Hello Alice!",
            output_text="Hi there!",
            model="gpt-3.5-turbo",
            tokens_used=10,
            cost_usd=0.00002,
            latency_ms=500
        )
        
        assert execution.prompt_id == "prompt-123"
        assert execution.prompt_key == "test_prompt"
        assert execution.user_id == "user-456"
        assert execution.tokens_used == 10
        assert execution.cost_usd == 0.00002
        assert execution.latency_ms == 500
        assert isinstance(execution.executed_at, datetime)
        assert len(execution.id) > 0


class TestPromptResponse:
    """Test PromptResponse model."""
    
    def test_create_response(self):
        """Test creating a prompt response."""
        response = PromptResponse(
            text="Hello there!",
            cost_usd=0.00002,
            tokens_used=10,
            latency_ms=500,
            model="gpt-3.5-turbo",
            prompt_version=1,
            execution_id="exec-123"
        )
        
        assert response.text == "Hello there!"
        assert response.cost_usd == 0.00002
        assert response.tokens_used == 10
        assert response.latency_ms == 500
        assert response.model == "gpt-3.5-turbo"
        assert response.prompt_version == 1
        assert response.execution_id == "exec-123"


class TestExperiment:
    """Test Experiment model."""
    
    def test_create_experiment(self):
        """Test creating an experiment."""
        experiment = Experiment(
            name="Test A/B",
            prompt_key="test_prompt",
            description="Testing two variants"
        )
        
        assert experiment.name == "Test A/B"
        assert experiment.prompt_key == "test_prompt"
        assert experiment.description == "Testing two variants"
        assert experiment.status == "draft"
        assert experiment.variants == []
        assert isinstance(experiment.created_at, datetime)
        assert len(experiment.id) > 0


class TestBusinessMetric:
    """Test BusinessMetric model."""
    
    def test_create_business_metric(self):
        """Test creating a business metric."""
        metric = BusinessMetric(
            execution_id="exec-123",
            user_id="user-456",
            metric_name="conversion_rate",
            metric_value=0.15,
            metric_type="conversion"
        )
        
        assert metric.execution_id == "exec-123"
        assert metric.user_id == "user-456"
        assert metric.metric_name == "conversion_rate"
        assert metric.metric_value == 0.15
        assert metric.metric_type == "conversion"
        assert isinstance(metric.recorded_at, datetime)
        assert len(metric.id) > 0