"""
Test OpenAI client integration.
"""

import pytest
from unittest.mock import Mock, patch
from promptsim.openai_client import OpenAIClient
from promptsim.exceptions import OpenAIError


class TestOpenAIClient:
    """Test OpenAI client functionality."""
    
    def test_openai_not_available(self):
        """Test behavior when OpenAI is not available."""
        with patch('promptsim.openai_client.OPENAI_AVAILABLE', False):
            with pytest.raises(OpenAIError, match="OpenAI and tiktoken packages are required"):
                OpenAIClient("test-key")
    
    @patch('promptsim.openai_client.OPENAI_AVAILABLE', True)
    @patch('promptsim.openai_client.tiktoken')
    @patch('promptsim.openai_client.OpenAI')
    def test_token_counting(self, mock_openai_class, mock_tiktoken):
        """Test token counting functionality."""
        # Setup mocks
        mock_encoding = Mock()
        mock_encoding.encode.return_value = [1, 2, 3, 4, 5]  # 5 tokens
        mock_tiktoken.encoding_for_model.return_value = mock_encoding
        
        client = OpenAIClient("test-key")
        
        # Test token counting
        token_count = client.count_tokens("Hello world!", "gpt-3.5-turbo")
        
        assert token_count == 5
        mock_tiktoken.encoding_for_model.assert_called_with("gpt-3.5-turbo")
        mock_encoding.encode.assert_called_with("Hello world!")
    
    @patch('promptsim.openai_client.OPENAI_AVAILABLE', True)
    @patch('promptsim.openai_client.tiktoken')
    @patch('promptsim.openai_client.OpenAI')
    def test_token_counting_fallback(self, mock_openai_class, mock_tiktoken):
        """Test token counting with unknown model fallback."""
        # Setup mocks
        mock_encoding = Mock()
        mock_encoding.encode.return_value = [1, 2, 3]  # 3 tokens
        mock_tiktoken.encoding_for_model.side_effect = KeyError("Unknown model")
        mock_tiktoken.get_encoding.return_value = mock_encoding
        
        client = OpenAIClient("test-key")
        
        # Test token counting with unknown model
        token_count = client.count_tokens("Hello!", "unknown-model")
        
        assert token_count == 3
        mock_tiktoken.get_encoding.assert_called_with("cl100k_base")
    
    def test_cost_calculation(self):
        """Test cost calculation."""
        with patch('promptsim.openai_client.OPENAI_AVAILABLE', True):
            with patch('promptsim.openai_client.OpenAI'):
                client = OpenAIClient("test-key")
                
                # Test GPT-3.5-turbo pricing
                cost = client.calculate_cost(1000, 500, "gpt-3.5-turbo")
                expected = (1000/1000 * 0.0015) + (500/1000 * 0.002)  # Input + output
                assert cost == expected
                
                # Test GPT-4 pricing
                cost = client.calculate_cost(1000, 500, "gpt-4")
                expected = (1000/1000 * 0.03) + (500/1000 * 0.06)
                assert cost == expected
                
                # Test unknown model (should use GPT-3.5-turbo pricing)
                cost = client.calculate_cost(1000, 500, "unknown-model")
                expected = (1000/1000 * 0.0015) + (500/1000 * 0.002)
                assert cost == expected
    
    @patch('promptsim.openai_client.OPENAI_AVAILABLE', True)
    @patch('promptsim.openai_client.tiktoken')
    @patch('promptsim.openai_client.OpenAI')
    def test_complete_success(self, mock_openai_class, mock_tiktoken):
        """Test successful completion."""
        # Setup mocks
        mock_encoding = Mock()
        mock_encoding.encode.side_effect = lambda text: [1] * len(text.split())  # 1 token per word
        mock_tiktoken.encoding_for_model.return_value = mock_encoding
        
        mock_response = Mock()
        mock_response.choices = [Mock()]
        mock_response.choices[0].message.content = "Hello there!"
        mock_response.usage.model_dump.return_value = {"prompt_tokens": 2, "completion_tokens": 2}
        
        mock_openai = Mock()
        mock_openai.chat.completions.create.return_value = mock_response
        mock_openai_class.return_value = mock_openai
        
        client = OpenAIClient("test-key")
        
        # Test completion
        result = client.complete("Hello world!", model="gpt-3.5-turbo")
        
        assert result["text"] == "Hello there!"
        assert result["model"] == "gpt-3.5-turbo"
        assert "tokens_used" in result
        assert "cost_usd" in result
        assert "latency_ms" in result
        assert result["latency_ms"] > 0
        
        # Verify OpenAI was called
        mock_openai.chat.completions.create.assert_called_once()
        call_args = mock_openai.chat.completions.create.call_args
        assert call_args[1]["model"] == "gpt-3.5-turbo"
        assert call_args[1]["messages"][0]["content"] == "Hello world!"
    
    @patch('promptsim.openai_client.OPENAI_AVAILABLE', True)
    @patch('promptsim.openai_client.OpenAI')
    def test_complete_error(self, mock_openai_class):
        """Test completion with error."""
        mock_openai = Mock()
        mock_openai.chat.completions.create.side_effect = Exception("API Error")
        mock_openai_class.return_value = mock_openai
        
        client = OpenAIClient("test-key")
        
        with pytest.raises(OpenAIError, match="OpenAI API call failed"):
            client.complete("Hello world!")
    
    @patch('promptsim.openai_client.OPENAI_AVAILABLE', True)
    @patch('promptsim.openai_client.tiktoken')
    @patch('promptsim.openai_client.OpenAI')
    def test_complete_with_messages(self, mock_openai_class, mock_tiktoken):
        """Test completion with message list."""
        # Setup mocks
        mock_encoding = Mock()
        mock_encoding.encode.return_value = [1, 2]  # 2 tokens
        mock_tiktoken.encoding_for_model.return_value = mock_encoding
        
        mock_response = Mock()
        mock_response.choices = [Mock()]
        mock_response.choices[0].message.content = "Response"
        mock_response.usage.model_dump.return_value = {"prompt_tokens": 2, "completion_tokens": 1}
        
        mock_openai = Mock()
        mock_openai.chat.completions.create.return_value = mock_response
        mock_openai_class.return_value = mock_openai
        
        client = OpenAIClient("test-key")
        
        messages = [
            {"role": "user", "content": "Hello"},
            {"role": "assistant", "content": "Hi there!"}
        ]
        
        result = client.complete_with_messages(messages)
        
        assert result["text"] == "Response"
        assert "tokens_used" in result
        assert "cost_usd" in result
        
        # Verify OpenAI was called with messages
        mock_openai.chat.completions.create.assert_called_once()
        call_args = mock_openai.chat.completions.create.call_args
        assert call_args[1]["messages"] == messages