"""
Test storage backends.
"""

import pytest
import tempfile
import os
from datetime import datetime
from promptsim.storage import SQLiteStorage, FileStorage
from promptsim.models import PromptTemplate, PromptExecution
from promptsim.exceptions import PromptNotFoundError, StorageError


class TestSQLiteStorage:
    """Test SQLite storage backend."""
    
    @pytest.fixture
    def storage(self):
        """Create temporary SQLite storage."""
        with tempfile.NamedTemporaryFile(suffix=".db", delete=False) as tmp:
            db_path = tmp.name
        
        storage = SQLiteStorage(db_path)
        yield storage
        
        # Cleanup
        os.unlink(db_path)
    
    def test_save_and_get_prompt(self, storage):
        """Test saving and retrieving a prompt."""
        prompt = PromptTemplate(
            key="test_prompt",
            template="Hello {name}!",
            name="Test Prompt",
            model="gpt-3.5-turbo"
        )
        
        # Save prompt
        storage.save_prompt(prompt)
        
        # Retrieve prompt
        retrieved = storage.get_prompt("test_prompt")
        
        assert retrieved.key == prompt.key
        assert retrieved.template == prompt.template
        assert retrieved.name == prompt.name
        assert retrieved.model == prompt.model
        assert retrieved.version == prompt.version
        assert retrieved.is_active == prompt.is_active
    
    def test_prompt_versioning(self, storage):
        """Test prompt versioning."""
        # Create version 1
        prompt_v1 = PromptTemplate(
            key="test_prompt",
            template="Hello {name}!",
            version=1
        )
        storage.save_prompt(prompt_v1)
        
        # Create version 2
        prompt_v2 = PromptTemplate(
            key="test_prompt",
            template="Hi {name}!",
            version=2,
            is_active=True
        )
        storage.save_prompt(prompt_v2)
        
        # Get active version (should be v2)
        active = storage.get_prompt("test_prompt")
        assert active.version == 2
        assert active.template == "Hi {name}!"
        
        # Get specific version
        v1 = storage.get_prompt("test_prompt", version=1)
        assert v1.version == 1
        assert v1.template == "Hello {name}!"
    
    def test_list_prompts(self, storage):
        """Test listing prompts."""
        # Create multiple prompts
        prompt1 = PromptTemplate(key="prompt1", template="Template 1")
        prompt2 = PromptTemplate(key="prompt2", template="Template 2")
        
        storage.save_prompt(prompt1)
        storage.save_prompt(prompt2)
        
        # List all prompts
        all_prompts = storage.list_prompts()
        assert len(all_prompts) == 2
        
        # List specific prompt
        prompt1_versions = storage.list_prompts("prompt1")
        assert len(prompt1_versions) == 1
        assert prompt1_versions[0].key == "prompt1"
    
    def test_prompt_not_found(self, storage):
        """Test prompt not found error."""
        with pytest.raises(PromptNotFoundError):
            storage.get_prompt("nonexistent")
    
    def test_save_and_get_execution(self, storage):
        """Test saving and retrieving executions."""
        execution = PromptExecution(
            prompt_id="prompt-123",
            prompt_key="test_prompt",
            prompt_version=1,
            input_text="Hello Alice!",
            output_text="Hi there!",
            model="gpt-3.5-turbo",
            tokens_used=10,
            cost_usd=0.00002,
            latency_ms=500
        )
        
        # Save execution
        storage.save_execution(execution)
        
        # Retrieve executions
        executions = storage.get_executions()
        assert len(executions) == 1
        
        retrieved = executions[0]
        assert retrieved.prompt_key == execution.prompt_key
        assert retrieved.input_text == execution.input_text
        assert retrieved.output_text == execution.output_text
        assert retrieved.tokens_used == execution.tokens_used
        assert retrieved.cost_usd == execution.cost_usd
    
    def test_get_executions_by_prompt(self, storage):
        """Test filtering executions by prompt key."""
        exec1 = PromptExecution(
            prompt_key="prompt1",
            input_text="Input 1",
            output_text="Output 1",
            model="gpt-3.5-turbo",
            tokens_used=5,
            cost_usd=0.00001,
            latency_ms=300
        )
        
        exec2 = PromptExecution(
            prompt_key="prompt2",
            input_text="Input 2",
            output_text="Output 2",
            model="gpt-3.5-turbo",
            tokens_used=8,
            cost_usd=0.00002,
            latency_ms=400
        )
        
        storage.save_execution(exec1)
        storage.save_execution(exec2)
        
        # Get executions for prompt1
        prompt1_executions = storage.get_executions("prompt1")
        assert len(prompt1_executions) == 1
        assert prompt1_executions[0].prompt_key == "prompt1"
        
        # Get all executions
        all_executions = storage.get_executions()
        assert len(all_executions) == 2


class TestFileStorage:
    """Test file storage backend."""
    
    @pytest.fixture
    def storage(self):
        """Create temporary file storage."""
        with tempfile.TemporaryDirectory() as tmp_dir:
            storage = FileStorage(tmp_dir)
            yield storage
    
    def test_save_and_get_prompt(self, storage):
        """Test saving and retrieving a prompt."""
        prompt = PromptTemplate(
            key="test_prompt",
            template="Hello {name}!",
            name="Test Prompt"
        )
        
        # Save prompt
        storage.save_prompt(prompt)
        
        # Retrieve prompt
        retrieved = storage.get_prompt("test_prompt")
        
        assert retrieved.key == prompt.key
        assert retrieved.template == prompt.template
        assert retrieved.name == prompt.name
    
    def test_prompt_versioning(self, storage):
        """Test prompt versioning with file storage."""
        # Create version 1
        prompt_v1 = PromptTemplate(
            key="test_prompt",
            template="Hello {name}!",
            version=1
        )
        storage.save_prompt(prompt_v1)
        
        # Create version 2
        prompt_v2 = PromptTemplate(
            key="test_prompt",
            template="Hi {name}!",
            version=2
        )
        storage.save_prompt(prompt_v2)
        
        # Get latest version
        latest = storage.get_prompt("test_prompt")
        assert latest.version == 2
        assert latest.template == "Hi {name}!"
        
        # Get specific version
        v1 = storage.get_prompt("test_prompt", version=1)
        assert v1.version == 1
        assert v1.template == "Hello {name}!"
    
    def test_list_prompts(self, storage):
        """Test listing prompts."""
        prompt1 = PromptTemplate(key="prompt1", template="Template 1")
        prompt2 = PromptTemplate(key="prompt2", template="Template 2")
        
        storage.save_prompt(prompt1)
        storage.save_prompt(prompt2)
        
        prompts = storage.list_prompts()
        assert len(prompts) == 2
        
        keys = [p.key for p in prompts]
        assert "prompt1" in keys
        assert "prompt2" in keys
    
    def test_save_and_get_execution(self, storage):
        """Test saving and retrieving executions."""
        execution = PromptExecution(
            prompt_key="test_prompt",
            input_text="Hello!",
            output_text="Hi!",
            model="gpt-3.5-turbo",
            tokens_used=5,
            cost_usd=0.00001,
            latency_ms=200
        )
        
        storage.save_execution(execution)
        
        executions = storage.get_executions()
        assert len(executions) == 1
        assert executions[0].prompt_key == "test_prompt"