"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PolicyStatementWithActions = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const common_substrings_1 = require("@udondan/common-substrings");
const RegexParser = require("regex-parser");
const access_level_1 = require("../access-level");
const _2_conditions_1 = require("./2-conditions");
/**
 * Adds "action" functionality to the Policy Statement
 */
class PolicyStatementWithActions extends _2_conditions_1.PolicyStatementWithCondition {
    constructor() {
        super(...arguments);
        this.accessLevelList = {};
        this.useNotAction = false;
        this.floydActions = [];
        this.cdkActionsApplied = false;
        this.isCompact = false;
    }
    /**
     * Injects actions into the statement.
     *
     * Only relevant for the main package. In CDK mode this only calls super.
     */
    toJSON() {
        // @ts-ignore only available after swapping 1-base
        if (typeof this.addResources == 'function') {
            this.cdkApplyActions();
            return super.toJSON();
        }
        const mode = this.useNotAction ? 'NotAction' : 'Action';
        const statement = super.toJSON();
        const self = this;
        if (this.hasActions()) {
            if (this.isCompact) {
                this.compactActions();
            }
            const actions = this.floydActions
                .filter((elem, pos) => {
                return self.floydActions.indexOf(elem) == pos;
            })
                .sort();
            statement[mode] = actions.length > 1 ? actions : actions[0];
        }
        return statement;
    }
    toStatementJson() {
        this.cdkApplyActions();
        // @ts-ignore only available after swapping 1-base
        return super.toStatementJson();
    }
    cdkApplyActions() {
        if (!this.cdkActionsApplied) {
            const mode = this.useNotAction ? 'addNotActions' : 'addActions';
            const self = this;
            if (this.isCompact) {
                this.compactActions();
            }
            const uniqueActions = this.floydActions
                .filter((elem, pos) => {
                return self.floydActions.indexOf(elem) == pos;
            })
                .sort();
            // @ts-ignore only available after swapping 1-base
            this[mode](...uniqueActions);
        }
        this.cdkActionsApplied = true;
    }
    /**
     * Switches the statement to use [`NotAction`](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notaction.html).
     */
    notAction() {
        this.useNotAction = true;
        return this;
    }
    /**
     * Checks weather actions have been applied to the policy.
     */
    hasActions() {
        return this.floydActions.length > 0;
    }
    /**
     * Adds actions by name.
     *
     * Depending on the "mode", actions will be either added to the list of [`Actions`](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_action.html) or [`NotAction`](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notaction.html).
     *
     * The mode can be switched by calling `notAction()`.
     *
     * If the action does not contain a colon, the action will be prefixed with the service prefix of the class, e.g. `ec2:`
     *
     * @param action Actions that will be added to the statement.
     */
    to(action) {
        if (this.servicePrefix.length && action.indexOf(':') < 0) {
            action = this.servicePrefix + ':' + action;
        }
        this.floydActions.push(action);
        return this;
    }
    /**
     * Adds all actions of the statement provider to the statement, e.g. `actions: 'ec2:*'`
     */
    allActions() {
        if (this.servicePrefix.length) {
            this.to(`${this.servicePrefix}:*`);
        }
        else {
            this.to('*');
        }
        return this;
    }
    /**
     * Adds all actions that match one of the given regular expressions.
     *
     * @param expressions One or more regular expressions. The regular expressions need to be in [Perl/JavaScript literal style](https://developer.mozilla.org/en-US/docs/Web/JavaScript/Guide/Regular_Expressions) and need to be passed as strings,
     * For example:
     * ```typescript
     * allMatchingActions('/vpn/i')
     * ```
     */
    allMatchingActions(...expressions) {
        expressions.forEach((expression) => {
            for (const [_, actions] of Object.entries(this.accessLevelList)) {
                actions.forEach((action) => {
                    if (action.match(RegexParser(expression))) {
                        this.to(`${this.servicePrefix}:${action}`);
                    }
                });
            }
        });
        return this;
    }
    /**
     * Adds all actions with [access level](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_understand-policy-summary-access-level-summaries.html#access_policies_access-level) LIST to the statement
     *
     * Permission to list resources within the service to determine whether an object exists.
     *
     * Actions with this level of access can list objects but cannot see the contents of a resource.
     *
     * For example, the Amazon S3 action `ListBucket` has the List access level.
     */
    allListActions() {
        return this.addAccessLevel(access_level_1.AccessLevel.LIST);
    }
    /**
     * Adds all actions with [access level](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_understand-policy-summary-access-level-summaries.html#access_policies_access-level) READ to the statement
     *
     * Permission to read but not edit the contents and attributes of resources in the service.
     *
     * For example, the Amazon S3 actions `GetObject` and `GetBucketLocation` have the Read access level.
     */
    allReadActions() {
        return this.addAccessLevel(access_level_1.AccessLevel.READ);
    }
    /**
     * Adds all actions with [access level](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_understand-policy-summary-access-level-summaries.html#access_policies_access-level) WRITE to the statement
     *
     * Permission to create, delete, or modify resources in the service.
     *
     * For example, the Amazon S3 actions `CreateBucket`, `DeleteBucket` and `PutObject` have the Write access level.
     *
     * Write actions might also allow modifying a resource tag. However, an action that allows only changes to tags has the Tagging access level.
     */
    allWriteActions() {
        return this.addAccessLevel(access_level_1.AccessLevel.WRITE);
    }
    /**
     * Adds all actions with [access level](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_understand-policy-summary-access-level-summaries.html#access_policies_access-level) PERMISSION MANAGEMENT to the statement
     *
     * Permission to grant or modify resource permissions in the service.
     *
     * For example, most IAM and AWS Organizations actions, as well as actions like the Amazon S3 actions `PutBucketPolicy` and `DeleteBucketPolicy` have the Permissions management access level.
     */
    allPermissionManagementActions() {
        return this.addAccessLevel(access_level_1.AccessLevel.PERMISSION_MANAGEMENT);
    }
    /**
     * Adds all actions with [access level](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_understand-policy-summary-access-level-summaries.html#access_policies_access-level) TAGGING to the statement
     *
     * Permission to perform actions that only change the state of resource tags.
     *
     * For example, the IAM actions `TagRole` and `UntagRole` have the Tagging access level because they allow only tagging or untagging a role. However, the `CreateRole` action allows tagging a role resource when you create that role. Because the action does not only add a tag, it has the Write access level.
     */
    allTaggingActions() {
        return this.addAccessLevel(access_level_1.AccessLevel.TAGGING);
    }
    addAccessLevel(accessLevel) {
        if (accessLevel in this.accessLevelList) {
            this.accessLevelList[accessLevel].forEach((action) => {
                this.to(`${this.servicePrefix}:${action}`);
            });
        }
        return this;
    }
    /**
     * Condense action list down to a list of patterns.
     *
     * Using this method can help to reduce the policy size.
     *
     * For example, all actions with access level `list` could be reduced to a small pattern `List*`.
     */
    compact() {
        this.isCompact = true;
        return this;
    }
    compactActions() {
        // actions that will be included, service prefix is removed
        const includeActions = this.floydActions.map((elem) => {
            return elem.substr(elem.indexOf(':') + 1);
        });
        // actions that will not be included, the opposite of includeActions
        const excludeActions = [];
        for (const [_, actions] of Object.entries(this.accessLevelList)) {
            actions.forEach((action) => {
                if (!includeActions.includes(action)) {
                    excludeActions.push(`^${action}$`);
                }
            });
        }
        // will contain the index of elements that are covered by substrings
        let covered = [];
        const subs = common_substrings_1.default(includeActions.map((action) => {
            return `^${action}$`;
        }), {
            minLength: 3,
            minOccurrence: 2,
        })
            .filter((sub) => {
            // remove all substrings, that match an action we have not selected
            for (let action of excludeActions) {
                if (action.includes(sub.name)) {
                    return false;
                }
            }
            return true;
        })
            .sort((a, b) => {
            // sort list by the number of matches
            if (a.source.length < b.source.length)
                return 1;
            if (a.source.length > b.source.length)
                return -1;
            return 0;
        })
            .filter((sub) => {
            // removes substrings, that have already been covered by other substrings
            const sources = sub.source.filter((source) => {
                return !covered.includes(source);
            });
            if (sources.length > 1) {
                //add list of sources to the global list of covered actions
                covered = covered.concat(sources);
                return true;
            }
            return false;
        });
        // stores the list of patterns
        const compactActionList = [];
        subs.forEach((sub) => {
            compactActionList.push(`${this.servicePrefix}:*${sub.name}*`
                .replace('*^', '')
                .replace('$*', ''));
            sub.source.forEach((source) => {
                includeActions[source] = ''; // invalidate, will be filtered later
            });
        });
        includeActions
            .filter((action) => {
            // remove elements that have been covered by patterns, we invalidated them above
            return action.length > 0;
        })
            .forEach((action) => {
            // add actions that have not been covered by patterns to the new action list
            compactActionList.push(`${this.servicePrefix}:${action}`);
        });
        // we're done. override action list
        this.floydActions = compactActionList;
    }
}
exports.PolicyStatementWithActions = PolicyStatementWithActions;
_a = JSII_RTTI_SYMBOL_1;
PolicyStatementWithActions[_a] = { fqn: "iam-floyd.PolicyStatementWithActions", version: "0.386.0" };
//# sourceMappingURL=data:application/json;base64,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