import threading
from typing import Literal, Optional, Dict, Any, Type

from dbqueue.sources import PostgresWalSource
from dbqueue.transports import RabbitMQTransport
from dbqueue.formatters import (
    BaseFormatter,
    SimpleFormatter,
    DebeziumLikeFormatter,
)
from dbqueue.utils import log
from dbqueue.core.consumer import RabbitConsumer


EventFormat = Literal["simple", "debezium_like"]

FORMATTER_MAP: Dict[EventFormat, Type[BaseFormatter]] = {
    "simple": SimpleFormatter,
    "debezium_like": DebeziumLikeFormatter,
}


class DBQueueWatcher:
    """
    PostgreSQL WAL → RabbitMQ publish → RabbitMQ handler consumer.
    Fully pluggable + developer friendly.
    """

    def __init__(
        self,
        db_dsn: str,
        rabbit_url: str,
        *,
        slot_name: str = "dbqueue_slot",
        publication: str = "dbqueue_pub",
        output_plugin: str = "pgoutput",
        event_format: EventFormat = "simple",
        routing_key_template: str = "{schema}.{table}.{op}",
        exchange: str = "dbqueue.events",
    ) -> None:
        self.db_dsn = db_dsn
        self.rabbit_url = rabbit_url
        self.slot_name = slot_name
        self.publication = publication
        self.output_plugin = output_plugin
        self.event_format = event_format
        self.routing_key_template = routing_key_template
        self.exchange = exchange

        # Developer Handler API
        self.handlers = {
            "INSERT": {},
            "UPDATE": {},
            "DELETE": {},
        }

        # Formatter
        formatter_cls = FORMATTER_MAP[self.event_format]
        self._formatter: BaseFormatter = formatter_cls()

        # RabbitMQ publisher
        self._transport = RabbitMQTransport(
            url=self.rabbit_url,
            exchange=self.exchange,
            routing_key_template=self.routing_key_template,
        )

        # WAL stream source
        self._source = PostgresWalSource(
            dsn=self.db_dsn,
            slot_name=self.slot_name,
            publication=self.publication,
            output_plugin=self.output_plugin,
            on_event=self._handle_raw_event,
        )

        # RabbitMQ consumer (handler router)
        self._consumer = RabbitConsumer(
            url=self.rabbit_url,
            exchange=self.exchange,
            routing_key_template=self.routing_key_template,
            handlers=self.handlers,
        )

        self._thread: Optional[threading.Thread] = None

    # ----------------------------------------------------------------------
    # -> Developer Handler API
    # ----------------------------------------------------------------------
    def on_insert(self, table: str):
        def decorator(fn):
            self.handlers["INSERT"][table.lower()] = fn
            return fn
        return decorator

    def on_update(self, table: str):
        def decorator(fn):
            self.handlers["UPDATE"][table.lower()] = fn
            return fn
        return decorator

    def on_delete(self, table: str):
        def decorator(fn):
            self.handlers["DELETE"][table.lower()] = fn
            return fn
        return decorator

    # ----------------------------------------------------------------------
    # -> CDC event publish stage
    # ----------------------------------------------------------------------
    def _handle_raw_event(self, event: Dict[str, Any]) -> None:
        formatted = self._formatter.format(event)
        self._transport.publish(formatted)

    # ----------------------------------------------------------------------
    # -> Pipeline control
    # ----------------------------------------------------------------------
    def start(self, background: bool = False) -> None:
        log("🚀 DBQueueWatcher starting...")

        # 🟢 Start consumer
        self._consumer.start()

        if background:
            self._thread = threading.Thread(
                target=self._source.start,
                daemon=True,
            )
            self._thread.start()
            log("🎯 Running in background thread.")
        else:
            log("🎯 Running in blocking mode.")
            self._source.start()

    def stop(self) -> None:
        self._source.stop()
        self._consumer.stop()
        log("⏹ DBQueueWatcher stopped.")

