from typing import List, Dict, Any, Tuple
from struct import unpack_from

from .base import BaseDecoder


class PgOutputDecoder(BaseDecoder):
    """
    Minimal pgoutput decoder (PostgreSQL 17, proto_version=1)

    Şu anda desteklenenler:
      - R (RELATION): tablo ve kolon bilgisini cache'ler
      - I (INSERT): after row üretir
      - U (UPDATE): before (varsa) + after üretir
      - D (DELETE): before row üretir

    Not: Şimdilik tüm kolonlar text olarak okunuyor (type cast yok),
    bu haliyle CDC event'leri JSON string olarak taşımak için yeterli.
    """

    def __init__(self) -> None:
        # relation_oid -> {"schema": str, "table": str, "columns": [name, ...]}
        self.relations: Dict[int, Dict[str, Any]] = {}

    # ------------------------------------------------------------------

    def decode(self, msg) -> List[Dict[str, Any]]:
        payload: bytes = msg.payload
        if not payload:
            return []

        tag = chr(payload[0])

        if tag == "R":  # Relation
            self._parse_relation(payload)
            return []

        if tag == "I":  # Insert
            evt = self._parse_insert(payload, msg)
            return [evt] if evt else []

        if tag == "U":  # Update
            evt = self._parse_update(payload, msg)
            return [evt] if evt else []

        if tag == "D":  # Delete
            evt = self._parse_delete(payload, msg)
            return [evt] if evt else []

        # BEGIN (B), COMMIT (C), TRUNCATE (T) vb. şimdilik ignore
        return []

    # ------------------------------------------------------------------
    # RELATION
    # ------------------------------------------------------------------

    def _parse_relation(self, payload: bytes) -> None:
        """
        R (Relation) message format (simplified):

        Byte1: 'R'
        Int32: relation oid
        String: namespace
        String: relation name
        Byte1: replica identity
        Int16: number of columns
        loop:
          Byte1: flags
          String: column name
          Int32: type oid
          Int32: type mod
        """
        idx = 1  # skip 'R'

        oid = self._read_int32(payload, idx)
        idx += 4

        namespace, idx = self._read_cstring(payload, idx)
        relname, idx = self._read_cstring(payload, idx)

        replica_id = payload[idx]
        idx += 1

        ncols = self._read_int16(payload, idx)
        idx += 2

        columns = []
        for _ in range(ncols):
            flags = payload[idx]
            idx += 1

            colname, idx = self._read_cstring(payload, idx)

            type_oid = self._read_int32(payload, idx)
            idx += 4

            type_mod = self._read_int32(payload, idx)
            idx += 4

            columns.append(colname)

        self.relations[oid] = {
            "schema": namespace,
            "table": relname,
            "columns": columns,
            "replica_id": replica_id,
        }

    # ------------------------------------------------------------------
    # INSERT
    # ------------------------------------------------------------------

    def _parse_insert(self, payload: bytes, msg) -> Dict[str, Any] | None:
        """
        I (Insert) message format:

        Byte1: 'I'
        Int32: relation oid
        Byte1: 'N'
        TupleData (yeni satır)
        """
        idx = 1  # skip 'I'

        relid = self._read_int32(payload, idx)
        idx += 4

        if relid not in self.relations:
            return None

        rel = self.relations[relid]

        tag = chr(payload[idx])
        idx += 1
        if tag != "N":
            return None

        after, idx = self._read_tuple(payload, idx, rel["columns"])

        return {
            "op": "insert",
            "schema": rel["schema"],
            "table": rel["table"],
            "before": None,
            "after": after,
            "meta": {
                "lsn": msg.data_start,
            },
        }

    # ------------------------------------------------------------------
    # UPDATE
    # ------------------------------------------------------------------

    def _parse_update(self, payload: bytes, msg) -> Dict[str, Any] | None:
        """
        U (Update) message format (simplified):

        Byte1: 'U'
        Int32: relation oid
        Optional:
          Byte1: 'K' (old key) veya 'O' (old tuple)
          TupleData (old)
        Required:
          Byte1: 'N'
          TupleData (new)
        """
        idx = 1  # skip 'U'

        relid = self._read_int32(payload, idx)
        idx += 4

        if relid not in self.relations:
            return None

        rel = self.relations[relid]
        before = None
        after = None

        # İlk tag 'K' veya 'O' olabilir (eski değerleri taşır)
        tag = chr(payload[idx])
        idx += 1

        if tag in ("K", "O"):
            before, idx = self._read_tuple(payload, idx, rel["columns"])
            # Sonrasında 'N' ile yeni tuple gelir
            if idx >= len(payload):
                return None
            tag = chr(payload[idx])
            idx += 1

        # Şu anki tag 'N' ise yeni row
        if tag == "N":
            after, idx = self._read_tuple(payload, idx, rel["columns"])

        return {
            "op": "update",
            "schema": rel["schema"],
            "table": rel["table"],
            "before": before,
            "after": after,
            "meta": {
                "lsn": msg.data_start,
            },
        }

    # ------------------------------------------------------------------
    # DELETE
    # ------------------------------------------------------------------

    def _parse_delete(self, payload: bytes, msg) -> Dict[str, Any] | None:
        """
        D (Delete) message format (simplified):

        Byte1: 'D'
        Int32: relation oid
        Byte1: 'K' (key) veya 'O' (old tuple)
        TupleData (old row)
        """
        idx = 1  # skip 'D'

        relid = self._read_int32(payload, idx)
        idx += 4

        if relid not in self.relations:
            return None

        rel = self.relations[relid]

        tag = chr(payload[idx])
        idx += 1
        if tag not in ("K", "O"):
            return None

        before, idx = self._read_tuple(payload, idx, rel["columns"])

        return {
            "op": "delete",
            "schema": rel["schema"],
            "table": rel["table"],
            "before": before,
            "after": None,
            "meta": {
                "lsn": msg.data_start,
            },
        }

    # ------------------------------------------------------------------
    # TUPLE PARSING
    # ------------------------------------------------------------------

    def _read_tuple(
        self,
        buf: bytes,
        offset: int,
        col_names: List[str],
    ) -> Tuple[Dict[str, Any], int]:
        """
        TupleData format:

        Int16: number of columns (ncols)
        loop ncols:
          Byte1: kind ('t' text, 'n' NULL, 'u' unchanged toast)
          if kind == 't':
            Int32: length
            Bytes: data
        """
        idx = offset
        ncols = self._read_int16(buf, idx)
        idx += 2

        values: Dict[str, Any] = {}
        for i in range(ncols):
            kind = chr(buf[idx])
            idx += 1

            col_name = col_names[i] if i < len(col_names) else f"col_{i}"

            if kind == "t":  # text
                length = self._read_int32(buf, idx)
                idx += 4
                data = buf[idx: idx + length]
                idx += length
                values[col_name] = data.decode("utf-8")
            elif kind == "n":  # NULL
                values[col_name] = None
            elif kind == "u":  # unchanged toast
                # Şimdilik unchanged toast için de None veriyoruz
                values[col_name] = None
            else:
                # Bilinmeyen tip: None geç
                values[col_name] = None

        return values, idx

    # ------------------------------------------------------------------
    # Binary yardımcıları
    # ------------------------------------------------------------------

    def _read_int16(self, buf: bytes, offset: int) -> int:
        return unpack_from("!h", buf, offset)[0]

    def _read_int32(self, buf: bytes, offset: int) -> int:
        return unpack_from("!i", buf, offset)[0]

    def _read_cstring(self, buf: bytes, offset: int) -> tuple[str, int]:
        end = buf.index(0, offset)
        s = buf[offset:end].decode("utf-8")
        return s, end + 1
