import json
from typing import List, Dict, Any
from psycopg2.extras import ReplicationMessage
from .base import BaseDecoder


class Wal2JsonDecoder(BaseDecoder):
    def decode(self, msg: ReplicationMessage) -> List[Dict[str, Any]]:
        payload_str = msg.payload
        data = json.loads(payload_str)

        changes = data.get("change", [])
        events = []

        for ch in changes:
            evt = self._normalize_change(ch, data)
            events.append(evt)

        return events

    def _normalize_change(
        self,
        change: Dict[str, Any],
        envelope: Dict[str, Any],
    ) -> Dict[str, Any]:

        kind = change.get("kind")  # insert | update | delete
        schema = change.get("schema")
        table = change.get("table")

        col_names = change.get("columnnames") or []
        col_values = change.get("columnvalues") or []
        after_row = dict(zip(col_names, col_values)) if col_names else None

        old = change.get("oldkeys") or {}
        old_names = old.get("keynames") or []
        old_values = old.get("keyvalues") or []
        before_row = dict(zip(old_names, old_values)) if old_names else None

        ts = envelope.get("timestamp")
        xid = envelope.get("xid")

        return {
            "op": kind,
            "schema": schema,
            "table": table,
            "before": before_row,
            "after": after_row if kind in ("insert", "update") else None,
            "meta": {
                "xid": xid,
                "timestamp": ts,
            },
        }
