import json
from typing import Any, Dict

import pika

from dbqueue.utils import log


class RabbitMQTransport:
    """
    Simple transport that publishes events to a RabbitMQ exchange.
    """

    def __init__(
        self,
        url: str,
        exchange: str = "dbqueue.events",
        exchange_type: str = "topic",
        durable: bool = True,
        routing_key_template: str = "{schema}.{table}.{op}",
    ) -> None:
        """
        :param url: AMQP URL, e.g., amqp://guest:guest@localhost:5672/
        :param exchange: Exchange name
        :param exchange_type: topic / direct / fanout etc.
        :param durable: Exchange durability
        :param routing_key_template: routing key template
        """
        self.url = url
        self.exchange = exchange
        self.exchange_type = exchange_type
        self.durable = durable
        self.routing_key_template = routing_key_template

        self.routing_key_template = routing_key_template
        self._connection = None
        self._channel = None

        self._connect()

    def _connect(self) -> None:
        """
        Establishes connection. Retry mechanism can be added if it fails.
        For now, let's try a simple while loop or let the caller manage it.
        Since there is no 'retry' parameter here, we try directly.
        """
        try:
            params = pika.URLParameters(self.url)
            self._connection = pika.BlockingConnection(params)
            self._channel = self._connection.channel()

            self._channel.exchange_declare(
                exchange=self.exchange,
                exchange_type=self.exchange_type,
                durable=self.durable,
            )
            log(f"🐇 RabbitMQ connected, exchange={self.exchange}")
        except Exception as e:
            log(f"❌ RabbitMQ connection failed: {e}")
            raise

    def _ensure_connection(self):
        if self._connection and not self._connection.is_closed:
            return

        log("⚠️ RabbitMQ connection lost, reconnecting...")
        import time
        while True:
            try:
                self._connect()
                break
            except Exception:
                time.sleep(5)

    def _build_routing_key(self, event: Dict[str, Any]) -> str:
        op = event.get("op")
        schema = event.get("schema")
        table = event.get("table")
        return self.routing_key_template.format(
            op=op or "",
            schema=schema or "",
            table=table or "",
        )

    def publish(self, event: Dict[str, Any]) -> None:
        routing_key = self._build_routing_key(event)
        body = json.dumps(event, default=str).encode("utf-8")

        try:
            self._publish_internal(routing_key, body)
        except (pika.exceptions.ConnectionClosed, pika.exceptions.AMQPConnectionError):
            log("⚠️ Connection error during publish, retrying...")
            self._ensure_connection()
            self._publish_internal(routing_key, body)

    def _publish_internal(self, routing_key: str, body: bytes) -> None:
        if not self._channel or self._channel.is_closed:
            self._ensure_connection()

        self._channel.basic_publish(
            exchange=self.exchange,
            routing_key=routing_key,
            body=body,
            properties=pika.BasicProperties(
                delivery_mode=2,  # persistent
                content_type="application/json",
            ),
        )
        log(f"📤 Published to RabbitMQ: rk={routing_key}")
