import pytest
from datetime import datetime
from decimal import Decimal
from dbqueue.formatters import SimpleFormatter, DebeziumLikeFormatter

def test_simple_formatter():
    formatter = SimpleFormatter()
    raw_event = {
        "schema": "public",
        "table": "users",
        "op": "INSERT",
        "after": {"id": 1, "name": "Mete", "created_at": "2023-01-01 12:00:00"},
        "before": None,
        "meta": {}
    }
    formatted = formatter.format(raw_event)
    
    assert formatted["op"] == "INSERT"
    assert formatted["table"] == "users"
    assert formatted["after"]["name"] == "Mete"

def test_debezium_like_formatter():
    formatter = DebeziumLikeFormatter()
    raw_event = {
        "schema": "public",
        "table": "users",
        "op": "update",  # Debezium formatter expects lowercase op from wal2json decoder logic? 
                       # Wait, wal2json decoder returns 'kind' which is usually lowercase 'insert', 'update'.
                       # Let's check wal2json decoder again. 
                       # Yes: kind = change.get("kind") # insert | update | delete
        "after": {"id": 1, "name": "Mete Updated"},
        "before": {"id": 1, "name": "Mete"},
        "meta": {"timestamp": 123456789, "xid": 100}
    }
    formatted = formatter.format(raw_event)
    
    assert formatted["op"] == "u"
    assert formatted["after"]["name"] == "Mete Updated"
    assert formatted["before"]["name"] == "Mete"
    assert formatted["source"]["table"] == "users"

def test_decimal_serialization():
    formatter = SimpleFormatter()
    raw_event = {
        "schema": "public",
        "table": "orders",
        "op": "INSERT",
        "after": {"amount": Decimal("10.50")},
        "before": None
    }
    formatted = formatter.format(raw_event)
    assert formatted["after"]["amount"] == Decimal("10.50")
