import pytest
from unittest.mock import MagicMock, patch
from dbqueue.sources import PostgresWalSource

def test_source_connect(mock_db_conn):
    conn, cursor = mock_db_conn
    with patch("psycopg2.connect", return_value=conn) as mock_connect:
        source = PostgresWalSource(dsn="postgresql://localhost/db")
        source.start()
        
        # start() calls _connect() then _ensure_slot() then starts replication
        mock_connect.assert_called_once()
        cursor.create_replication_slot.assert_called()
        cursor.start_replication.assert_called()

def test_source_consume_event(mock_db_conn):
    conn, cursor = mock_db_conn
    
    # Mock a WAL message
    mock_msg = MagicMock()
    mock_msg.payload = '{"change": []}' # Dummy payload
    mock_msg.data_start = 100
    mock_msg.cursor = cursor
    
    # Mock decoder
    mock_decoder = MagicMock()
    mock_decoder.decode.return_value = [{"op": "INSERT", "table": "test"}]
    
    with patch("psycopg2.connect", return_value=conn):
        # We need to patch where the decoder is imported or assigned
        # Since it's assigned inside start(), it's harder to mock directly without partial mocking
        # For simplicity, let's mock the decoder class in the module
        
        with patch("dbqueue.sources.decoders.wal2json.Wal2JsonDecoder", return_value=mock_decoder):
            source = PostgresWalSource(dsn="postgresql://localhost/db", output_plugin="wal2json")
            
            # Manually trigger connect to set up cursor
            source._connect()
            source.decoder = mock_decoder # Inject mock decoder
            
            # Trigger consume
            source._consume(mock_msg)
            
            mock_decoder.decode.assert_called_with(mock_msg)
            cursor.send_feedback.assert_called_with(flush_lsn=100)
