# First Powermake

## Installation & updates

First of all, make sure PowerMake is up to date, new versions are often out each two weeks.  
For this tutorial, you need `powermake >= 2.4.0`

```sh
pip install -U powermake
```


## File hierarchy

We will create a very very small project that look like this:

```
- my_project
    - src
        main.c
```

> [!NOTE]  
> PowerMake doesn't require you to have a specific file hierarchy, so if you don't like this one, just change it, but you will have to adapt everything according to your file hierarchy.

In the `main.c` file, you should put a small program, for example:
```c
#include <stdio.h>

int main()
{
    printf("Hello World !");
    return 0;
}
```

Once your file hierarchy is created, go in the `my_project` folder and run this command:
```sh
powermake
```

> [!IMPORTANT]  
> If the command `powermake` is not found, it's because the python script dir (on linux often `~/.local/bin`) is not in your path.  
> You can either add the python script dir in your path or run `python -m powermake` which does exactly the same as `powermake`

This command should have generated a file named makefile.py just under `my_project`
```
- my_project
    - src
        main.c
    - makefile.py
```

> [!NOTE]  
> You can rename the file `makefile.py` the name you like the most, but for the rest, we'll assume that this file is called makefile.py.


Open makefile.py in your favorite editor, the file should look like this:

```py
import powermake

def on_build(config: powermake.Config):
    config.add_flags("-Wall", "-Wextra")

    config.add_includedirs("./")

    files = powermake.get_files("**/*.c",
        "**/*.cpp", "**/*.cc", "**/*.C",
        "**/*.asm", "**/*.s", "**/*.rc")

    objects = powermake.compile_files(config, files)

    powermake.link_files(config, objects)

powermake.run("my_project", build_callback=on_build)
```

## Running your first makefile

Go in the same directory as the `makefile.py` file and run:

```sh
python makefile.py -rv
```

- The `-r` option is here to specify that we want to rebuild everything each time (like this PowerMake will not skip any already compiled file and you will see everything that happens even if you enter this command multiple time.)
- The `-v` option enable the verbose mode, like this you will see which command PowerMake is actually running.

At this time, if you have followed the steps carefully, you should now have a new build folder
```
- my_project
    - src
        main.c
    - build
        - ...
    - makefile.py
```

In this build folder you will found the generated binary in an architecture like this:
```
build/Linux/x64/release/bin/my_project
```
Obviously, the exact path will depend on your platform.

> [!NOTE]  
> If you've looked carefully the commands generated by PowerMake, you may be wondering why is PowerMake using the `g++` linker while we are only compiling a C file.  
> That's intended. `g++` is able to link both C and C++ compiled files and when using `g++` on a C compiled object, the behavior is the same as when using `gcc`.

## Running your newly compiled program

Obviously, if you run:
```sh
./build/Linux/x64/release/bin/my_project
```
(with the path updated for your platform)

You will see a beautiful
```
Hello World !
```

However, it might not be practical to have to find your program in the build folder.

And easier way to run your program is to use the builtin `test` callback in powermake.

Just run:
```
python makefile.py -t
```
And you will see the same beautiful `Hello World !`.

> [!WARNING]  
> You will have issues if you compile and run your program with different arguments, let's say you want to compile in debug mode, like this:
> ```sh
> python makefile.py -rvd
> ```
> In debug mode, your program will go in the debug build folder (`build/Linux/x64/debug/bin/my_project`).
> 
> But the trap here is that `python makefile.py -t` will run the release version.  
> To run the debug version, you must use `python makefile.py -dt`.
>
> To avoid mistakes like this, we highly encourage you to compile and run in the same command, like this:
> ```sh
> python makefile.py -rvt
> ```
> And if you don't want to rebuild your project each time, you can use:
> ```sh
> python makefile.py -bvt
> ```


## Deep dive in makefile.py

So far, you were able to compile and test a small C project.  
Maybe if you played a little with the command line, you were even able to compile in debug and release, in verbose or not and with or without rebuilding everything each time.

However, you haven't written a single line in the `makefile.py`, we just used the default template and by chance it worked.

Let's take a look inside this file.

```py
import powermake

def on_build(config: powermake.Config):
    config.add_flags("-Wall", "-Wextra")

    config.add_includedirs("./")

    files = powermake.get_files("**/*.c",
        "**/*.cpp", "**/*.cc", "**/*.C",
        "**/*.asm", "**/*.s", "**/*.rc")

    objects = powermake.compile_files(config, files)

    powermake.link_files(config, objects)

powermake.run("my_project", build_callback=on_build)
```

It may look like a lot to compile a single file and it is, because this template is already able to compile a rather big project.


The entry point of this file is this line:
```py
powermake.run("my_project", build_callback=on_build)
```

This small line will be present in almost every powermake makefile and does a ton of things for you.

It's here that the command line is parsed, that the optional PowerMake config files are parsed, that the compilation environment is configured, that your compiler is detected and setup and finally it's this line that will launch specific actions like cleaning, compiling, testing, installing, etc...

Here, we only have set up a build callback.  
This means that the function `on_build` will be called when it's compilation time and default callbacks will be called for all other jobs (cleaning, installing, testing).


The actual compilation happens in this function:
```py
def on_build(config: powermake.Config):
```

This function can be called as you like, but should always take a `config` parameter.  
This parameter is generated by `powermake.run` and passed to the function when it's called.

The config parameter will contain everything required for the compilation. The compiler object, the flags to use, the build directories, etc...

With these two lines, we modify the config to add new flags and add a new include directory.
```py
config.add_flags("-Wall", "-Wextra")
config.add_includedirs("./")
```
We don't really need these two lines for our mini-project, you can remove them and see that everything is still working.  
The first line enable most warnings for our compilation, we may want to keep that, it will never hurt.  
The second line is irrelevant here, we should just remove it.

Then comes this complex line:
```py
files = powermake.get_files("**/*.c",
    "**/*.cpp", "**/*.cc", "**/*.C",
    "**/*.asm", "**/*.s", "**/*.rc")
```
Here, we are just scanning every subfolder of the project to create a set of files that can be compiled.  
This is way too complex for our need, just this is way enough:
```py
files = powermake.get_files("**/*.c")
```
We can also limit the search to the `src` subfolder:
```py
files = powermake.get_files("src/*.c")
```
And if you want you can also just hardcode the file like this:
```py
files = {"src/main.c"}
```
Though this is way less scalable as you will need to edit the makefile every time you add a file.


Finally, we compile every .c file in an .o with this line:
```py
objects = powermake.compile_files(config, files)
```

And we link all these objects together in a single executable
```py
powermake.link_files(config, objects)
```

Here is a recap of the simplified makefile:
```py
import powermake

def on_build(config: powermake.Config):
    config.add_flags("-Wall", "-Wextra")

    files = powermake.get_files("**/*.c")

    objects = powermake.compile_files(config, files)

    powermake.link_files(config, objects)

powermake.run("my_project", build_callback=on_build)
```

## Play with the command line

Do you feel a little more confortable with the basic makefile ?

You should now play a little bit with the command line, try every option and try to see what they does.

Use `python makefile.py --help` to display all available options.  
Use the `-v` flag to really see what is happening (unless you are testing the verbosity flags of course)

Don't worry if you don't manage to understand every single one of them, they will all eventually be explained in the tutorial.


## [Optional] Set up your IDE

If you are using vscode, you are lucky because PowerMake can configure your IDE for you.  
Just install the [Microsoft C/C++ extension](https://marketplace.visualstudio.com/items?itemName=ms-vscode.cpptools-extension-pack) and then run:
```sh
python makefile.py --generate-vscode
```

Then you will be able to put a breakpoint in your main.c file using F9 and then debug your program using F5

See [Compatibility with vscode](../../documentation.md#visual-studio-code) for more details.


If you are not using vscode, it's likely that your IDE supports `compile_commands.json` files.
You can generate this file with PowerMake like this:

```sh
python makefile.py -rvd -o .
```


### [-> Next tutorial (Crossplatform library)](../02-crossplatform-library/README.md)