"""
Section plugin tests
"""

from django.db import transaction

from cms.api import add_plugin
from cms.models import Placeholder

from richie.apps.core.tests.utils import CMSPluginTestCase
from richie.plugins.section.cms_plugins import SectionPlugin
from richie.plugins.section.factories import SectionFactory


# pylint: disable=too-many-ancestors
class SectionCMSPluginsTestCase(CMSPluginTestCase):
    """Section plugin tests case"""

    @staticmethod
    @transaction.atomic
    def test_cms_plugins_section_title_required():
        """
        The "title" field is not required when instantiating a section.

        Use ``@transaction.atomic`` decorator because of CMSTestCase behavior,
        see details in CMSPluginTestCase docstring.
        """
        SectionFactory(title=None)

    def test_cms_plugins_section_create_success(self):
        """
        Section plugin creation success
        """
        section = SectionFactory(title="Foo")
        self.assertEqual("Foo", section.title)

    def test_cms_plugins_section_context_and_html(self):
        """
        Instanciating this plugin with an instance should populate the context
        and render in the template.
        """
        placeholder = Placeholder.objects.create(slot="test")

        # Create random values for parameters with a factory
        section = SectionFactory()

        model_instance = add_plugin(
            placeholder, SectionPlugin, "en", title=section.title
        )
        plugin_instance = model_instance.get_plugin_class_instance()
        plugin_context = plugin_instance.render({}, model_instance, None)

        # Check if "instance" is in plugin context
        self.assertIn("instance", plugin_context)

        # Check if parameters, generated by the factory, are correctly set in
        # "instance" of plugin context
        self.assertEqual(plugin_context["instance"].title, section.title)

        # Template context
        context = self.get_practical_plugin_context()

        # Get generated html for section title
        html = context["cms_content_renderer"].render_plugin(model_instance, {})

        # Check rendered title
        self.assertIn(section.title, html)

    def test_cms_plugins_section_render(self):
        """
        Full rendering for a Section with default title level and nested
        plugin content
        """
        # Title template with default title level
        title_format = """<h2 class="section__title">{}</h2>"""

        # Dummy slot where to include plugin
        placeholder = Placeholder.objects.create(slot="test")

        # Create random values for parameters with a factory
        section = SectionFactory()

        # Template context
        context = self.get_practical_plugin_context()

        # Init base Section plugin with required title
        base_plugin_instance = add_plugin(
            placeholder, SectionPlugin, "en", title=section.title
        )

        # Nest a Text plugin into the Section plugin
        nested_content = "<p>Lorem ipsum</p>"
        add_plugin(
            placeholder,
            plugin_type="CKEditorPlugin",
            language="en",
            target=base_plugin_instance,
            body="<p>Lorem ipsum</p>",
        )

        # Render placeholder so plugin is fully rendered in real situation
        html = context["cms_content_renderer"].render_placeholder(
            placeholder, context=context, language="en"
        )

        expected_title = title_format.format(section.title)

        # Check expected title and nested content are there
        self.assertInHTML(expected_title, html)
        self.assertInHTML(nested_content, html)

    def test_cms_plugins_section_header_level(self):
        """
        Header level can be changed from context variable 'header_level'.
        """
        # We deliberately use level '10' since it can be substituted from any
        # reasonable default level.
        header_format = """<h10 class="section__title">{}</h10>"""

        # Dummy slot where to include plugin
        placeholder = Placeholder.objects.create(slot="test")

        # Create random values for parameters with a factory
        section = SectionFactory()

        # Template context with additional variable to define a custom header
        # level for header markup
        context = self.get_practical_plugin_context({"header_level": 10})

        # Init base Section plugin with required title
        add_plugin(placeholder, SectionPlugin, "en", title=section.title)

        # Render placeholder so plugin is fully rendered in real situation
        html = context["cms_content_renderer"].render_placeholder(
            placeholder, context=context, language="en"
        )

        expected_header = header_format.format(section.title)

        # Expected header markup should match given 'header_level' context
        # variable
        self.assertInHTML(expected_header, html)
