"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.resolve = resolve;
const util_1 = require("util");
function isResolvable(obj) {
    return obj.toJSON !== undefined;
}
function resolve(value, options = {}) {
    const args = options.args ?? [];
    const omitEmpty = options.omitEmpty ?? false;
    if (value == null) {
        return value;
    }
    if (isResolvable(value)) {
        const resolved = value.toJSON();
        return resolve(resolved, options);
    }
    // Special resolution for few JavaScript built-in types
    // that by default would be stringified as empty objects ('{}')
    // as they are missing a `toJSON` implementation.
    switch (true) {
        case util_1.types.isRegExp(value):
            if (value.flags)
                throw new Error("RegExp with flags should be explicitly converted to a string");
            return value.source;
        case util_1.types.isSet(value):
            return resolve(Array.from(value), options);
        case util_1.types.isMap(value):
            return resolve(Object.fromEntries(value), options);
        case util_1.types.isBoxedPrimitive(value):
            return resolve(value.valueOf(), options);
    }
    // if value is a function, call it and resolve the result.
    if (typeof value === "function") {
        const resolved = value.apply(undefined, args);
        return resolve(resolved, options);
    }
    if (typeof value !== "object") {
        return value;
    }
    if (Array.isArray(value)) {
        if (omitEmpty && value.length === 0) {
            return undefined;
        }
        return value.map((x) => resolve(x, options)).filter((x) => x != null); // filter undefined/null/omitted
    }
    // only allow data types (i.e. objects without constructors)
    if (value.constructor && value.constructor?.name !== "Object") {
        throw new Error(`only data types can be resolved. trying to resolve object of type ${value.constructor?.name}`);
    }
    const result = {};
    for (const [k, v] of Object.entries(value)) {
        const resolved = resolve(v, options);
        // skip undefined values
        if (resolved === undefined) {
            continue;
        }
        result[k] = resolved;
    }
    if (omitEmpty && Object.keys(result).length === 0) {
        return undefined;
    }
    return result;
}
//# sourceMappingURL=data:application/json;base64,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