"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsCdkTypeScriptApp = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const auto_discover_1 = require("./auto-discover");
const awscdk_deps_js_1 = require("./awscdk-deps-js");
const cdk_config_1 = require("./cdk-config");
const cdk_tasks_1 = require("./cdk-tasks");
const integ_runner_1 = require("./integ-runner");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
const javascript_1 = require("../javascript");
const typescript_1 = require("../typescript");
/**
 * AWS CDK app in TypeScript
 *
 * @pjid awscdk-app-ts
 */
class AwsCdkTypeScriptApp extends typescript_1.TypeScriptAppProject {
    /**
     * The CDK version this app is using.
     */
    get cdkVersion() {
        return this.cdkDeps.cdkVersion;
    }
    constructor(options) {
        super({
            ...options,
            sampleCode: false,
            bundlerOptions: {
                ...options.bundlerOptions,
                // we invoke the "bundle" task as part of the build step in cdk.json so
                // we don't want it to be added to the pre-compile phase.
                runBundleTask: javascript_1.RunBundleTask.MANUAL,
            },
        });
        this.cdkDeps = new awscdk_deps_js_1.AwsCdkDepsJs(this, {
            dependencyType: dependencies_1.DependencyType.RUNTIME,
            ...options,
        });
        this.appEntrypoint = options.appEntrypoint ?? "main.ts";
        // CLI
        this.addDevDeps(`aws-cdk@${this.cdkDeps.cdkCliVersion}`);
        // no compile step because we do all of it in typescript directly
        this.compileTask.reset();
        this.cdkTasks = new cdk_tasks_1.CdkTasks(this);
        // add synth to the build
        this.postCompileTask.spawn(this.cdkTasks.synthSilent);
        const tsConfigFile = this.tsconfig?.fileName;
        if (!tsConfigFile) {
            throw new Error("Expecting tsconfig.json");
        }
        this.cdkConfig = new cdk_config_1.CdkConfig(this, {
            app: `npx ts-node -P ${tsConfigFile} --prefer-ts-exts ${path.posix.join(this.srcdir, this.appEntrypoint)}`,
            featureFlags: this.cdkDeps.cdkMajorVersion < 2,
            buildCommand: this.runTaskCommand(this.bundler.bundleTask),
            watchIncludes: [`${this.srcdir}/**/*.ts`, `${this.testdir}/**/*.ts`],
            watchExcludes: [
                "README.md",
                "cdk*.json",
                "**/*.d.ts",
                "**/*.js",
                "tsconfig.json",
                "package*.json",
                "yarn.lock",
                "node_modules",
            ],
            ...options,
        });
        this.gitignore.exclude(".parcel-cache/");
        this.npmignore?.exclude(`${this.cdkConfig.cdkout}/`);
        this.npmignore?.exclude(".cdk.staging/");
        if (this.tsconfig) {
            this.tsconfig.exclude.push(this.cdkConfig.cdkout);
        }
        this.addDevDeps("ts-node");
        if (options.sampleCode ?? true) {
            new SampleCode(this, this.cdkDeps.cdkMajorVersion);
        }
        new auto_discover_1.AutoDiscover(this, {
            srcdir: this.srcdir,
            testdir: this.testdir,
            lambdaOptions: options.lambdaOptions,
            tsconfigPath: this.tsconfigDev.fileName,
            cdkDeps: this.cdkDeps,
            lambdaAutoDiscover: options.lambdaAutoDiscover ?? true,
            edgeLambdaAutoDiscover: options.edgeLambdaAutoDiscover ?? true,
            lambdaExtensionAutoDiscover: options.lambdaExtensionAutoDiscover ?? true,
            integrationTestAutoDiscover: options.integrationTestAutoDiscover ?? true,
        });
        if (options.experimentalIntegRunner) {
            new integ_runner_1.IntegRunner(this);
        }
    }
    /**
     * Adds an AWS CDK module dependencies
     * @param modules The list of modules to depend on
     */
    addCdkDependency(...modules) {
        return this.cdkDeps.addV1Dependencies(...modules);
    }
}
exports.AwsCdkTypeScriptApp = AwsCdkTypeScriptApp;
_a = JSII_RTTI_SYMBOL_1;
AwsCdkTypeScriptApp[_a] = { fqn: "projen.awscdk.AwsCdkTypeScriptApp", version: "0.91.13" };
class SampleCode extends component_1.Component {
    constructor(project, cdkMajorVersion) {
        super(project);
        this.cdkMajorVersion = cdkMajorVersion;
        this.appProject = project;
    }
    synthesize() {
        const outdir = this.project.outdir;
        const srcdir = path.join(outdir, this.appProject.srcdir);
        if (fs.existsSync(srcdir) &&
            fs.readdirSync(srcdir).filter((x) => x.endsWith(".ts"))) {
            return;
        }
        const srcImports = new Array();
        if (this.cdkMajorVersion < 2) {
            srcImports.push("import { App, Construct, Stack, StackProps } from '@aws-cdk/core';");
        }
        else {
            srcImports.push("import { App, Stack, StackProps } from 'aws-cdk-lib';");
            srcImports.push("import { Construct } from 'constructs';");
        }
        const srcCode = `${srcImports.join("\n")}

export class MyStack extends Stack {
  constructor(scope: Construct, id: string, props: StackProps = {}) {
    super(scope, id, props);

    // define resources here...
  }
}

// for development, use account/region from cdk cli
const devEnv = {
  account: process.env.CDK_DEFAULT_ACCOUNT,
  region: process.env.CDK_DEFAULT_REGION,
};

const app = new App();

new MyStack(app, '${this.project.name}-dev', { env: devEnv });
// new MyStack(app, '${this.project.name}-prod', { env: prodEnv });

app.synth();`;
        fs.mkdirSync(srcdir, { recursive: true });
        fs.writeFileSync(path.join(srcdir, this.appProject.appEntrypoint), srcCode);
        const testdir = path.join(outdir, this.appProject.testdir);
        if (fs.existsSync(testdir) &&
            fs.readdirSync(testdir).filter((x) => x.endsWith(".ts"))) {
            return;
        }
        const testImports = new Array();
        if (this.cdkMajorVersion < 2) {
            testImports.push("import { App } from '@aws-cdk/core';");
            testImports.push("import { Template } from '@aws-cdk/assertions';");
        }
        else {
            testImports.push("import { App } from 'aws-cdk-lib';");
            testImports.push("import { Template } from 'aws-cdk-lib/assertions';");
        }
        const appEntrypointName = path.basename(this.appProject.appEntrypoint, ".ts");
        const testCode = `${testImports.join("\n")}
import { MyStack } from '../${this.appProject.srcdir}/${appEntrypointName}';

test('Snapshot', () => {
  const app = new App();
  const stack = new MyStack(app, 'test');

  const template = Template.fromStack(stack);
  expect(template.toJSON()).toMatchSnapshot();
});`;
        fs.mkdirSync(testdir, { recursive: true });
        fs.writeFileSync(path.join(testdir, `${appEntrypointName}.test.ts`), testCode);
    }
}
//# sourceMappingURL=data:application/json;base64,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