"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BuildWorkflow = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const component_1 = require("../component");
const github_1 = require("../github");
const constants_1 = require("../github/constants");
const util_1 = require("../github/private/util");
const workflow_actions_1 = require("../github/workflow-actions");
const workflows_model_1 = require("../github/workflows-model");
const javascript_1 = require("../javascript");
const runner_options_1 = require("../runner-options");
const consts_1 = require("./private/consts");
const name_1 = require("../util/name");
const path_1 = require("../util/path");
class BuildWorkflow extends component_1.Component {
    constructor(project, options) {
        super(project);
        this._postBuildJobs = [];
        const github = github_1.GitHub.of(this.project.root);
        if (!github) {
            throw new Error("BuildWorkflow is currently only supported for GitHub projects");
        }
        this.github = github;
        this.preBuildSteps = options.preBuildSteps ?? [];
        this.postBuildSteps = options.postBuildSteps ?? [];
        this.gitIdentity = options.gitIdentity ?? constants_1.DEFAULT_GITHUB_ACTIONS_USER;
        this.buildTask = options.buildTask;
        this.artifactsDirectory =
            options.artifactsDirectory ?? consts_1.DEFAULT_ARTIFACTS_DIRECTORY;
        (0, util_1.ensureNotHiddenPath)(this.artifactsDirectory, "artifactsDirectory");
        this.name = options.name ?? (0, name_1.workflowNameForProject)("build", this.project);
        const mutableBuilds = options.mutableBuild ?? true;
        this.workflow = new github_1.GithubWorkflow(github, this.name);
        this.workflow.on(options.workflowTriggers ?? {
            pullRequest: {},
            workflowDispatch: {}, // allow manual triggering
        });
        this.addBuildJob(options);
        if (mutableBuilds) {
            this.addSelfMutationJob(options);
        }
        if (project instanceof javascript_1.NodeProject) {
            project.addPackageIgnore(constants_1.PERMISSION_BACKUP_FILE);
        }
    }
    addBuildJob(options) {
        const projectPathRelativeToRoot = path.relative(this.project.root.outdir, this.project.outdir);
        const jobConfig = {
            ...(0, runner_options_1.filteredRunsOnOptions)(options.runsOn, options.runsOnGroup),
            container: options.containerImage
                ? { image: options.containerImage }
                : undefined,
            env: {
                CI: "true",
                ...options.env,
            },
            permissions: {
                contents: workflows_model_1.JobPermission.WRITE,
                ...options.permissions,
            },
            defaults: this.project.parent // is subproject,
                ? {
                    run: {
                        workingDirectory: (0, path_1.ensureRelativePathStartsWithDot)(projectPathRelativeToRoot),
                    },
                }
                : undefined,
            steps: (() => this.renderBuildSteps(projectPathRelativeToRoot)),
            outputs: {
                [consts_1.SELF_MUTATION_HAPPENED_OUTPUT]: {
                    stepId: consts_1.SELF_MUTATION_STEP,
                    outputName: consts_1.SELF_MUTATION_HAPPENED_OUTPUT,
                },
            },
        };
        this.workflow.addJob(consts_1.BUILD_JOBID, jobConfig);
    }
    /**
     * Returns a list of job IDs that are part of the build.
     */
    get buildJobIds() {
        return [consts_1.BUILD_JOBID, ...this._postBuildJobs];
    }
    /**
     * Adds steps that are executed after the build.
     * @param steps The job steps
     */
    addPostBuildSteps(...steps) {
        this.postBuildSteps.push(...steps);
    }
    /**
     * Adds another job to the build workflow which is executed after the build
     * job succeeded.
     *
     * Jobs are executed _only_ if the build did NOT self mutate. If the build
     * self-mutate, the branch will either be updated or the build will fail (in
     * forks), so there is no point in executing the post-build job.
     *
     * @param id The id of the new job
     * @param job The job specification
     */
    addPostBuildJob(id, job) {
        const steps = [];
        steps.push({
            name: "Download build artifacts",
            uses: "actions/download-artifact@v4",
            with: {
                name: constants_1.BUILD_ARTIFACT_NAME,
                path: this.artifactsDirectory,
            },
        }, {
            name: "Restore build artifact permissions",
            continueOnError: true,
            run: [
                `cd ${this.artifactsDirectory} && setfacl --restore=${constants_1.PERMISSION_BACKUP_FILE}`,
            ].join("\n"),
        });
        steps.push(...(job.steps ?? []));
        this.workflow.addJob(id, {
            needs: [consts_1.BUILD_JOBID],
            // only run if build did not self-mutate
            if: `\${{ !${consts_1.SELF_MUTATION_CONDITION} }}`,
            ...job,
            steps: steps,
        });
        // add to the list of build job IDs
        this._postBuildJobs.push(id);
    }
    /**
     * Run a task as a job within the build workflow which is executed after the
     * build job succeeded.
     *
     * The job will have access to build artifacts and will install project
     * dependencies in order to be able to run any commands used in the tasks.
     *
     * Jobs are executed _only_ if the build did NOT self mutate. If the build
     * self-mutate, the branch will either be updated or the build will fail (in
     * forks), so there is no point in executing the post-build job.
     *
     * @param options Specify tools and other options
     */
    addPostBuildJobTask(task, options = {}) {
        this.addPostBuildJobCommands(`post-build-${task.name}`, [`${this.project.projenCommand} ${task.name}`], {
            checkoutRepo: true,
            installDeps: true,
            tools: options.tools,
            ...(0, runner_options_1.filteredRunsOnOptions)(options.runsOn, options.runsOnGroup),
        });
    }
    /**
     * Run a sequence of commands as a job within the build workflow which is
     * executed after the build job succeeded.
     *
     * Jobs are executed _only_ if the build did NOT self mutate. If the build
     * self-mutate, the branch will either be updated or the build will fail (in
     * forks), so there is no point in executing the post-build job.
     *
     * @param options Specify tools and other options
     */
    addPostBuildJobCommands(id, commands, options) {
        const steps = [];
        if (options?.checkoutRepo) {
            steps.push(github_1.WorkflowSteps.checkout({
                with: {
                    ref: consts_1.PULL_REQUEST_REF,
                    repository: consts_1.PULL_REQUEST_REPOSITORY,
                    ...(this.github.downloadLfs ? { lfs: true } : {}),
                },
            }));
        }
        if (options?.checkoutRepo &&
            options?.installDeps &&
            this.project instanceof javascript_1.NodeProject) {
            steps.push({
                name: "Install dependencies",
                run: `${this.project.package.installCommand}`,
            });
        }
        steps.push({ run: commands.join("\n") });
        this.addPostBuildJob(id, {
            permissions: {
                contents: workflows_model_1.JobPermission.READ,
            },
            tools: options?.tools,
            ...(0, runner_options_1.filteredRunsOnOptions)(options?.runsOn, options?.runsOnGroup),
            steps,
        });
    }
    addSelfMutationJob(options) {
        this.workflow.addJob("self-mutation", {
            ...(0, runner_options_1.filteredRunsOnOptions)(options.runsOn, options.runsOnGroup),
            permissions: {
                contents: workflows_model_1.JobPermission.WRITE,
            },
            needs: [consts_1.BUILD_JOBID],
            if: `always() && ${consts_1.SELF_MUTATION_CONDITION} && ${consts_1.NOT_FORK}`,
            steps: [
                ...this.workflow.projenCredentials.setupSteps,
                ...workflow_actions_1.WorkflowActions.checkoutWithPatch({
                    // we need to use a PAT so that our push will trigger the build workflow
                    token: this.workflow.projenCredentials.tokenRef,
                    ref: consts_1.PULL_REQUEST_REF,
                    repository: consts_1.PULL_REQUEST_REPOSITORY,
                    lfs: this.github.downloadLfs,
                }),
                github_1.WorkflowSteps.setupGitIdentity({ gitIdentity: this.gitIdentity }),
                {
                    name: "Push changes",
                    env: {
                        PULL_REQUEST_REF: consts_1.PULL_REQUEST_REF,
                    },
                    run: [
                        "git add .",
                        'git commit -s -m "chore: self mutation"',
                        `git push origin HEAD:$PULL_REQUEST_REF`,
                    ].join("\n"),
                },
            ],
        });
    }
    /**
     * Called (lazily) during synth to render the build job steps.
     */
    renderBuildSteps(projectPathRelativeToRoot) {
        return [
            github_1.WorkflowSteps.checkout({
                with: {
                    ref: consts_1.PULL_REQUEST_REF,
                    repository: consts_1.PULL_REQUEST_REPOSITORY,
                    ...(this.github.downloadLfs ? { lfs: true } : {}),
                },
            }),
            ...this.preBuildSteps,
            {
                name: this.buildTask.name,
                run: this.github.project.runTaskCommand(this.buildTask),
            },
            ...this.postBuildSteps,
            // check for mutations and upload a git patch file as an artifact
            ...workflow_actions_1.WorkflowActions.uploadGitPatch({
                stepId: consts_1.SELF_MUTATION_STEP,
                outputName: consts_1.SELF_MUTATION_HAPPENED_OUTPUT,
                mutationError: "Files were changed during build (see build log). If this was triggered from a fork, you will need to update your branch.",
            }),
            // upload the build artifact only if we have post-build jobs (otherwise, there's no point)
            ...(this._postBuildJobs.length == 0
                ? []
                : [
                    {
                        name: "Backup artifact permissions",
                        continueOnError: true,
                        run: `cd ${this.artifactsDirectory} && getfacl -R . > ${constants_1.PERMISSION_BACKUP_FILE}`,
                    },
                    github_1.WorkflowSteps.uploadArtifact({
                        with: {
                            name: constants_1.BUILD_ARTIFACT_NAME,
                            path: this.project.parent
                                ? `${projectPathRelativeToRoot}/${this.artifactsDirectory}`
                                : this.artifactsDirectory,
                        },
                    }),
                ]),
        ];
    }
}
exports.BuildWorkflow = BuildWorkflow;
_a = JSII_RTTI_SYMBOL_1;
BuildWorkflow[_a] = { fqn: "projen.build.BuildWorkflow", version: "0.91.13" };
//# sourceMappingURL=data:application/json;base64,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