"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.JsiiProject = exports.Stability = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const consts_1 = require("./consts");
const jsii_docgen_1 = require("./jsii-docgen");
const consts_2 = require("../build/private/consts");
const workflow_steps_1 = require("../github/workflow-steps");
const workflows_model_1 = require("../github/workflows-model");
const javascript_1 = require("../javascript");
const runner_options_1 = require("../runner-options");
const typescript_1 = require("../typescript");
const util_1 = require("../util");
const EMAIL_REGEX = /(?:[a-z0-9!#$%&'*+/=?^_`{|}~-]+(?:\.[a-z0-9!#$%&'*+/=?^_`{|}~-]+)*|"(?:[\x01-\x08\x0b\x0c\x0e-\x1f\x21\x23-\x5b\x5d-\x7f]|\\[\x01-\x09\x0b\x0c\x0e-\x7f])*")@(?:(?:[a-z0-9](?:[a-z0-9-]*[a-z0-9])?\.)+[a-z0-9](?:[a-z0-9-]*[a-z0-9])?|\[(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?|[a-z0-9-]*[a-z0-9]:(?:[\x01-\x08\x0b\x0c\x0e-\x1f\x21-\x5a\x53-\x7f]|\\[\x01-\x09\x0b\x0c\x0e-\x7f])+)\])/;
const URL_REGEX = /((([A-Za-z]{3,9}:(?:\/\/)?)(?:[\-;:&=\+\$,\w]+@)?[A-Za-z0-9\.\-]+|(?:www\.|[\-;:&=\+\$,\w]+@)[A-Za-z0-9\.\-]+)((?:\/[\+~%\/\.\w\-_]*)?\??(?:[\-\+=&;%@\.\w_]*)#?(?:[\.\!\/\\\w]*))?)/;
const REPO_TEMP_DIRECTORY = ".repo";
const BUILD_ARTIFACT_OLD_DIR = "dist.old";
var Stability;
(function (Stability) {
    Stability["EXPERIMENTAL"] = "experimental";
    Stability["STABLE"] = "stable";
    Stability["DEPRECATED"] = "deprecated";
})(Stability || (exports.Stability = Stability = {}));
/**
 * Multi-language jsii library project
 *
 * @pjid jsii
 */
class JsiiProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        const { authorEmail, authorUrl } = parseAuthorAddress(options);
        const jsiiVersion = options.jsiiVersion ?? "~5.6.0";
        const defaultOptions = {
            repository: options.repositoryUrl,
            authorName: options.author,
            authorEmail,
            authorUrl,
        };
        const forcedOptions = {
            releaseToNpm: false, // we have a jsii release workflow
            disableTsconfig: true, // jsii generates its own tsconfig.json
            docgen: false, // we use jsii-docgen here so disable typescript docgen
        };
        const mergedOptions = (0, util_1.deepMerge)([
            defaultOptions,
            options,
            forcedOptions,
        ]);
        super(mergedOptions);
        const srcdir = this.srcdir;
        const libdir = this.libdir;
        this.addFields({ types: `${libdir}/index.d.ts` });
        const compressAssembly = options.compressAssembly ?? false;
        // this is an unhelpful warning
        const jsiiFlags = ["--silence-warnings=reserved-word"];
        if (compressAssembly) {
            jsiiFlags.push("--compress-assembly");
        }
        const compatIgnore = options.compatIgnore ?? ".compatignore";
        this.addFields({ stability: options.stability ?? Stability.STABLE });
        if (options.stability === Stability.DEPRECATED) {
            this.addFields({ deprecated: true });
        }
        const compatTask = this.addTask("compat", {
            description: "Perform API compatibility check against latest version",
            exec: `jsii-diff npm:$(node -p "require(\'./package.json\').name") -k --ignore-file ${compatIgnore} || (echo "\nUNEXPECTED BREAKING CHANGES: add keys such as \'removed:constructs.Node.of\' to ${compatIgnore} to skip.\n" && exit 1)`,
        });
        const compat = options.compat ?? false;
        if (compat) {
            this.compileTask.spawn(compatTask);
        }
        this.compileTask.reset(["jsii", ...jsiiFlags].join(" "));
        this.watchTask.reset(["jsii", "-w", ...jsiiFlags].join(" "));
        // Create a new package:all task, it will be filled with language targets later
        this.packageAllTask = this.addTask("package-all", {
            description: "Packages artifacts for all target languages",
        });
        // in jsii we consider the entire repo (post build) as the build artifact
        // which is then used to create the language bindings in separate jobs.
        // we achieve this by doing a checkout and overwrite with the files from the js package.
        this.packageJsTask = this.addPackagingTask("js");
        // When running inside CI we initially only package js. Other targets are packaged in separate jobs.
        // Outside of CI (i.e locally) we simply package all targets.
        this.packageTask.reset();
        this.packageTask.spawn(this.packageJsTask, {
            // Only run in CI
            condition: `node -e "if (!process.env.CI) process.exit(1)"`,
        });
        this.packageTask.spawn(this.packageAllTask, {
            // Don't run in CI
            condition: `node -e "if (process.env.CI) process.exit(1)"`,
        });
        const targets = {};
        const jsii = {
            outdir: this.artifactsDirectory,
            targets,
            tsc: {
                outDir: libdir,
                rootDir: srcdir,
            },
        };
        if (options.excludeTypescript) {
            jsii.excludeTypescript = options.excludeTypescript;
        }
        this.addFields({ jsii });
        const extraJobOptions = {
            ...this.getJobRunsOnConfig(options),
            ...(options.workflowContainerImage
                ? { container: { image: options.workflowContainerImage } }
                : {}),
        };
        if (options.releaseToNpm != false) {
            const npmjs = {
                registry: this.package.npmRegistry,
                npmTokenSecret: this.package.npmTokenSecret,
                npmProvenance: this.package.npmProvenance,
                codeArtifactOptions: options.codeArtifactOptions,
            };
            this.addTargetToBuild("js", this.packageJsTask, extraJobOptions);
            this.addTargetToRelease("js", this.packageJsTask, npmjs);
        }
        // we cannot call an option `java` because the java code generated by jsii
        // does not compile due to a conflict between this option name and the `java`
        // package (e.g. when `java.util.Objects` is referenced).
        if ("java" in options) {
            throw new Error('the "java" option is now called "publishToMaven"');
        }
        const maven = options.publishToMaven;
        if (maven) {
            targets.java = {
                package: maven.javaPackage,
                maven: {
                    groupId: maven.mavenGroupId,
                    artifactId: maven.mavenArtifactId,
                },
            };
            const task = this.addPackagingTask("java");
            this.addTargetToBuild("java", task, extraJobOptions);
            this.addTargetToRelease("java", task, maven);
        }
        const pypi = options.publishToPypi ?? options.python;
        if (pypi) {
            targets.python = {
                distName: pypi.distName,
                module: pypi.module,
            };
            const task = this.addPackagingTask("python");
            this.addTargetToBuild("python", task, extraJobOptions);
            this.addTargetToRelease("python", task, pypi);
        }
        const nuget = options.publishToNuget ?? options.dotnet;
        if (nuget) {
            targets.dotnet = {
                namespace: nuget.dotNetNamespace,
                packageId: nuget.packageId,
                iconUrl: nuget.iconUrl,
            };
            const task = this.addPackagingTask("dotnet");
            this.addTargetToBuild("dotnet", task, extraJobOptions);
            this.addTargetToRelease("dotnet", task, nuget);
        }
        const golang = options.publishToGo;
        if (golang) {
            targets.go = {
                moduleName: golang.moduleName,
                packageName: golang.packageName,
                versionSuffix: golang.versionSuffix,
            };
            const task = this.addPackagingTask("go");
            this.addTargetToBuild("go", task, extraJobOptions);
            this.addTargetToRelease("go", task, golang);
        }
        // If jsiiVersion is "*", don't specify anything so the user can manage.
        // Otherwise, use `jsiiVersion`
        const jsiiSuffix = jsiiVersion === "*" ? "" : `@${jsiiVersion}`;
        this.addDevDeps(`jsii${jsiiSuffix}`, `jsii-rosetta${jsiiSuffix}`, "jsii-diff", "jsii-pacmak");
        this.gitignore.exclude(".jsii", "tsconfig.json");
        this.npmignore?.include(".jsii");
        if (options.docgen ?? true) {
            // If jsiiVersion is "*", don't specify anything so the user can manage.
            // Otherwise use a version that is compatible with all supported jsii releases.
            const docgenVersion = options.jsiiVersion === "*" ? "*" : "^10.5.0";
            new jsii_docgen_1.JsiiDocgen(this, {
                version: docgenVersion,
                filePath: options.docgenFilePath,
            });
        }
        // jsii updates .npmignore, so we make it writable
        if (this.npmignore) {
            this.npmignore.readonly = false;
        }
    }
    /**
     * Adds a target language to the release workflow.
     * @param language
     * @returns
     */
    addTargetToRelease(language, packTask, target) {
        if (!this.release) {
            return;
        }
        const pacmak = this.pacmakForLanguage(language, packTask);
        const prePublishSteps = [
            ...pacmak.bootstrapSteps,
            workflow_steps_1.WorkflowSteps.checkout({
                with: {
                    path: REPO_TEMP_DIRECTORY,
                    ...(this.github?.downloadLfs ? { lfs: true } : {}),
                },
            }),
            ...pacmak.packagingSteps,
        ];
        const commonPublishOptions = {
            publishTools: pacmak.publishTools,
            prePublishSteps,
        };
        const handler = publishTo[language];
        this.release?.publisher[handler]({
            ...commonPublishOptions,
            ...target,
        });
    }
    /**
     * Adds a target language to the build workflow
     * @param language
     * @returns
     */
    addTargetToBuild(language, packTask, extraJobOptions) {
        if (!this.buildWorkflow) {
            return;
        }
        const pacmak = this.pacmakForLanguage(language, packTask);
        this.buildWorkflow.addPostBuildJob(`package-${language}`, {
            ...(0, runner_options_1.filteredRunsOnOptions)(extraJobOptions.runsOn, extraJobOptions.runsOnGroup),
            permissions: {
                contents: workflows_model_1.JobPermission.READ,
            },
            tools: {
                node: { version: this.nodeVersion ?? "lts/*" },
                ...pacmak.publishTools,
            },
            steps: [
                ...pacmak.bootstrapSteps,
                workflow_steps_1.WorkflowSteps.checkout({
                    with: {
                        path: REPO_TEMP_DIRECTORY,
                        ref: consts_2.PULL_REQUEST_REF,
                        repository: consts_2.PULL_REQUEST_REPOSITORY,
                        ...(this.github?.downloadLfs ? { lfs: true } : {}),
                    },
                }),
                ...pacmak.packagingSteps,
            ],
            ...extraJobOptions,
        });
    }
    addPackagingTask(language) {
        const packageTargetTask = this.tasks.addTask(`package:${language}`, {
            description: `Create ${language} language bindings`,
        });
        const commandParts = ["jsii-pacmak", "-v"];
        if (this.package.packageManager === javascript_1.NodePackageManager.PNPM) {
            commandParts.push("--pack-command 'pnpm pack'");
        }
        commandParts.push(`--target ${language}`);
        packageTargetTask.exec(commandParts.join(" "));
        this.packageAllTask.spawn(packageTargetTask);
        return packageTargetTask;
    }
    pacmakForLanguage(target, packTask) {
        const bootstrapSteps = [];
        const packagingSteps = [];
        // Generic bootstrapping for all target languages
        bootstrapSteps.push(...this.workflowBootstrapSteps);
        if (this.package.packageManager === javascript_1.NodePackageManager.PNPM) {
            bootstrapSteps.push({
                name: "Setup pnpm",
                uses: "pnpm/action-setup@v3",
                with: { version: this.package.pnpmVersion },
            });
        }
        else if (this.package.packageManager === javascript_1.NodePackageManager.BUN) {
            bootstrapSteps.push({
                name: "Setup bun",
                uses: "oven-sh/setup-bun@v2",
                with: { "bun-version": this.package.bunVersion },
            });
        }
        // Installation steps before packaging, but after checkout
        packagingSteps.push({
            name: "Install Dependencies",
            run: `cd ${REPO_TEMP_DIRECTORY} && ${this.package.installCommand}`,
        }, {
            name: "Extract build artifact",
            run: `tar --strip-components=1 -xzvf ${this.artifactsDirectory}/js/*.tgz -C ${REPO_TEMP_DIRECTORY}`,
        }, {
            name: `Move build artifact out of the way`,
            run: `mv ${this.artifactsDirectory} ${BUILD_ARTIFACT_OLD_DIR}`,
        }, {
            name: `Create ${target} artifact`,
            run: `cd ${REPO_TEMP_DIRECTORY} && ${this.runTaskCommand(packTask)}`,
        }, {
            name: `Collect ${target} artifact`,
            run: `mv ${REPO_TEMP_DIRECTORY}/${this.artifactsDirectory} ${this.artifactsDirectory}`,
        });
        return {
            publishTools: consts_1.JSII_TOOLCHAIN[target],
            bootstrapSteps,
            packagingSteps,
        };
    }
    /**
     * Generates the runs-on config for Jobs.
     * Throws error if 'runsOn' and 'runsOnGroup' are both set.
     *
     * @param options - 'runsOn' or 'runsOnGroup'.
     */
    getJobRunsOnConfig(options) {
        return options.workflowRunsOnGroup
            ? { runsOnGroup: options.workflowRunsOnGroup }
            : options.workflowRunsOn
                ? { runsOn: options.workflowRunsOn }
                : {};
    }
}
exports.JsiiProject = JsiiProject;
_a = JSII_RTTI_SYMBOL_1;
JsiiProject[_a] = { fqn: "projen.cdk.JsiiProject", version: "0.91.13" };
const publishTo = {
    js: "publishToNpm",
    java: "publishToMaven",
    python: "publishToPyPi",
    dotnet: "publishToNuget",
    go: "publishToGo",
};
function parseAuthorAddress(options) {
    let authorEmail = options.authorEmail;
    let authorUrl = options.authorUrl;
    if (options.authorAddress) {
        if (options.authorEmail && options.authorEmail !== options.authorAddress) {
            throw new Error("authorEmail is deprecated and cannot be used in conjunction with authorAddress");
        }
        if (options.authorUrl && options.authorUrl !== options.authorAddress) {
            throw new Error("authorUrl is deprecated and cannot be used in conjunction with authorAddress.");
        }
        if (EMAIL_REGEX.test(options.authorAddress)) {
            authorEmail = options.authorAddress;
        }
        else if (URL_REGEX.test(options.authorAddress)) {
            authorUrl = options.authorAddress;
        }
        else {
            throw new Error(`authorAddress must be either an email address or a URL: ${options.authorAddress}`);
        }
    }
    return { authorEmail, authorUrl };
}
//# sourceMappingURL=data:application/json;base64,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