"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.isObjectContainingFieldExactly = exports.Circleci = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const component_1 = require("../component");
const util_1 = require("../util");
const yaml_1 = require("../yaml");
/**
 * Circleci Class to manage `.circleci/config.yml`.
 * Check projen's docs for more information.
 *
 * @see https://circleci.com/docs/2.0/configuration-reference/
 */
class Circleci extends component_1.Component {
    constructor(project, options = {}) {
        super(project);
        /**
         * reduce objects with `identifier` field for WorkflowJobs.
         * A workflow job can contain `orbParameter` which are passed to orbs.
         * This map is directly added as `Record<string,any>` to job.
         * So we gonna add those after the default field of WorkflowJob.
         * @see https://circleci.com/developer/orbs/orb/circleci/node#usage-install_nodejs
         * @param jobs
         */
        this.renderJobs = (jobs = []) => {
            let result = [];
            for (const job of jobs ?? []) {
                const { identifier, orbParameters, ...reduced } = job;
                if ((0, exports.isObjectContainingFieldExactly)(job, "identifier")) {
                    result = [...result, identifier];
                }
                else {
                    result = [
                        ...result,
                        { [identifier]: { ...reduced, ...orbParameters } },
                    ];
                }
            }
            return result;
        };
        /**
         * Snake case for listed keys. There are too many exceptions needed to do it recursive without a whitelist.
         * This list needs to be updated once we add field with snake case like `aws_auth`.
         * @param input
         */
        this.snakeCase = (input) => {
            const snakeCaseKeyWords = [
                "awsAuth",
                "workingDirectory",
                "resourceClass",
                "dockerLayerCaching",
                "noOutputTimeout",
            ];
            return (0, util_1.snakeCaseKeys)(input, true, snakeCaseKeyWords);
        };
        this.options = options;
        this.orbs = options.orbs ?? {};
        this.workflows = options.workflows ?? [];
        this.jobs = options.jobs ?? [];
        this.file = new yaml_1.YamlFile(project, ".circleci/config.yml", {
            obj: () => this.renderCircleCi(),
        });
    }
    /**
     * function to prepare the rendering of the yaml file.
     * Objects with dynamic keys like workflows and jobs required the field `identifier` to be set.
     * Those object will be reduced by `identifier` field before rendering
     * @private
     */
    renderCircleCi() {
        // render dynamic keys for workflow
        const workflowRecords = {};
        for (const workflow of this.workflows) {
            const { identifier, ...reduced } = workflow;
            reduced.jobs = this.renderJobs(workflow.jobs);
            workflowRecords[identifier] = this.snakeCase(reduced);
        }
        // render dynamic keys for jobs
        const jobRecords = {};
        for (const job of this.jobs) {
            const { identifier, ...reduced } = job;
            jobRecords[identifier] = this.snakeCase(reduced);
        }
        return {
            version: this.options.version || 2.1,
            setup: this.options.setup,
            orbs: this.orbs,
            jobs: Object.keys(jobRecords).length > 0 ? jobRecords : undefined,
            workflows: workflowRecords,
        };
    }
    /**
     * add new workflow to existing pipeline
     * @param workflow
     */
    addWorkflow(workflow) {
        this.workflows = [...this.workflows, workflow];
    }
    /**
     * Add a Circleci Orb to pipeline. Will throw error if the orb already exists
     * @param name
     * @param orb
     */
    addOrb(name, orb) {
        if (this.orbs[name] !== undefined) {
            throw new Error(`Circleci Config already contains an orb named ${name}.`);
        }
        this.orbs[name] = orb;
    }
}
exports.Circleci = Circleci;
_a = JSII_RTTI_SYMBOL_1;
Circleci[_a] = { fqn: "projen.circleci.Circleci", version: "0.91.13" };
const isObjectContainingFieldExactly = (obj, field) => {
    return Object.keys(obj).length == 1 && Object.keys(obj).includes(field);
};
exports.isObjectContainingFieldExactly = isObjectContainingFieldExactly;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2lyY2xlY2kuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi9zcmMvY2lyY2xlY2kvY2lyY2xlY2kudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFDQSw0Q0FBeUM7QUFFekMsa0NBQXdDO0FBQ3hDLGtDQUFtQztBQWdFbkM7Ozs7O0dBS0c7QUFDSCxNQUFhLFFBQVMsU0FBUSxxQkFBUztJQTBCckMsWUFBWSxPQUFnQixFQUFFLFVBQXlCLEVBQUU7UUFDdkQsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBaURqQjs7Ozs7OztXQU9HO1FBQ0ssZUFBVSxHQUFHLENBQUMsT0FBc0IsRUFBRSxFQUFPLEVBQUU7WUFDckQsSUFBSSxNQUFNLEdBQVEsRUFBRSxDQUFDO1lBQ3JCLEtBQUssTUFBTSxHQUFHLElBQUksSUFBSSxJQUFJLEVBQUUsRUFBRSxDQUFDO2dCQUM3QixNQUFNLEVBQUUsVUFBVSxFQUFFLGFBQWEsRUFBRSxHQUFHLE9BQU8sRUFBRSxHQUFHLEdBQUcsQ0FBQztnQkFDdEQsSUFBSSxJQUFBLHNDQUE4QixFQUFDLEdBQUcsRUFBRSxZQUFZLENBQUMsRUFBRSxDQUFDO29CQUN0RCxNQUFNLEdBQUcsQ0FBQyxHQUFHLE1BQU0sRUFBRSxVQUFVLENBQUMsQ0FBQztnQkFDbkMsQ0FBQztxQkFBTSxDQUFDO29CQUNOLE1BQU0sR0FBRzt3QkFDUCxHQUFHLE1BQU07d0JBQ1QsRUFBRSxDQUFDLFVBQVUsQ0FBQyxFQUFFLEVBQUUsR0FBRyxPQUFPLEVBQUUsR0FBRyxhQUFhLEVBQUUsRUFBRTtxQkFDbkQsQ0FBQztnQkFDSixDQUFDO1lBQ0gsQ0FBQztZQUNELE9BQU8sTUFBTSxDQUFDO1FBQ2hCLENBQUMsQ0FBQztRQWNGOzs7O1dBSUc7UUFDSyxjQUFTLEdBQUcsQ0FBQyxLQUFVLEVBQU8sRUFBRTtZQUN0QyxNQUFNLGlCQUFpQixHQUFHO2dCQUN4QixTQUFTO2dCQUNULGtCQUFrQjtnQkFDbEIsZUFBZTtnQkFDZixvQkFBb0I7Z0JBQ3BCLGlCQUFpQjthQUNsQixDQUFDO1lBQ0YsT0FBTyxJQUFBLG9CQUFhLEVBQUMsS0FBSyxFQUFFLElBQUksRUFBRSxpQkFBaUIsQ0FBQyxDQUFDO1FBQ3ZELENBQUMsQ0FBQztRQWxHQSxJQUFJLENBQUMsT0FBTyxHQUFHLE9BQU8sQ0FBQztRQUN2QixJQUFJLENBQUMsSUFBSSxHQUFHLE9BQU8sQ0FBQyxJQUFJLElBQUksRUFBRSxDQUFDO1FBQy9CLElBQUksQ0FBQyxTQUFTLEdBQUcsT0FBTyxDQUFDLFNBQVMsSUFBSSxFQUFFLENBQUM7UUFDekMsSUFBSSxDQUFDLElBQUksR0FBRyxPQUFPLENBQUMsSUFBSSxJQUFJLEVBQUUsQ0FBQztRQUMvQixJQUFJLENBQUMsSUFBSSxHQUFHLElBQUksZUFBUSxDQUFDLE9BQU8sRUFBRSxzQkFBc0IsRUFBRTtZQUN4RCxHQUFHLEVBQUUsR0FBRyxFQUFFLENBQUMsSUFBSSxDQUFDLGNBQWMsRUFBRTtTQUNqQyxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7O09BS0c7SUFDSyxjQUFjO1FBQ3BCLG1DQUFtQztRQUNuQyxNQUFNLGVBQWUsR0FBd0IsRUFBRSxDQUFDO1FBQ2hELEtBQUssTUFBTSxRQUFRLElBQUksSUFBSSxDQUFDLFNBQVMsRUFBRSxDQUFDO1lBQ3RDLE1BQU0sRUFBRSxVQUFVLEVBQUUsR0FBRyxPQUFPLEVBQUUsR0FBRyxRQUFRLENBQUM7WUFDNUMsT0FBTyxDQUFDLElBQUksR0FBRyxJQUFJLENBQUMsVUFBVSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUM5QyxlQUFlLENBQUMsVUFBVSxDQUFDLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUN4RCxDQUFDO1FBRUQsK0JBQStCO1FBQy9CLE1BQU0sVUFBVSxHQUF3QixFQUFFLENBQUM7UUFDM0MsS0FBSyxNQUFNLEdBQUcsSUFBSSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7WUFDNUIsTUFBTSxFQUFFLFVBQVUsRUFBRSxHQUFHLE9BQU8sRUFBRSxHQUFHLEdBQUcsQ0FBQztZQUN2QyxVQUFVLENBQUMsVUFBVSxDQUFDLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUNuRCxDQUFDO1FBRUQsT0FBTztZQUNMLE9BQU8sRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLE9BQU8sSUFBSSxHQUFHO1lBQ3BDLEtBQUssRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLEtBQUs7WUFDekIsSUFBSSxFQUFFLElBQUksQ0FBQyxJQUFJO1lBQ2YsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxTQUFTO1lBQ2pFLFNBQVMsRUFBRSxlQUFlO1NBQzNCLENBQUM7SUFDSixDQUFDO0lBRUQ7OztPQUdHO0lBQ0ksV0FBVyxDQUFDLFFBQWtCO1FBQ25DLElBQUksQ0FBQyxTQUFTLEdBQUcsQ0FBQyxHQUFHLElBQUksQ0FBQyxTQUFTLEVBQUUsUUFBUSxDQUFDLENBQUM7SUFDakQsQ0FBQztJQTBCRDs7OztPQUlHO0lBQ0ksTUFBTSxDQUFDLElBQVksRUFBRSxHQUFXO1FBQ3JDLElBQUksSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxTQUFTLEVBQUUsQ0FBQztZQUNsQyxNQUFNLElBQUksS0FBSyxDQUFDLGlEQUFpRCxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQzVFLENBQUM7UUFDRCxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLEdBQUcsQ0FBQztJQUN4QixDQUFDOztBQTlHSCw0QkErSEM7OztBQUVNLE1BQU0sOEJBQThCLEdBQUcsQ0FDNUMsR0FBUSxFQUNSLEtBQWEsRUFDSixFQUFFO0lBQ1gsT0FBTyxNQUFNLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDLE1BQU0sSUFBSSxDQUFDLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLENBQUM7QUFDMUUsQ0FBQyxDQUFDO0FBTFcsUUFBQSw4QkFBOEIsa0NBS3pDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgV29ya2Zsb3dKb2IsIFdvcmtmbG93LCBKb2IgfSBmcm9tIFwiLi9tb2RlbFwiO1xuaW1wb3J0IHsgQ29tcG9uZW50IH0gZnJvbSBcIi4uL2NvbXBvbmVudFwiO1xuaW1wb3J0IHsgUHJvamVjdCB9IGZyb20gXCIuLi9wcm9qZWN0XCI7XG5pbXBvcnQgeyBzbmFrZUNhc2VLZXlzIH0gZnJvbSBcIi4uL3V0aWxcIjtcbmltcG9ydCB7IFlhbWxGaWxlIH0gZnJvbSBcIi4uL3lhbWxcIjtcblxuLyoqXG4gKiBPcHRpb25zIGZvciBjbGFzcyB7QGxpbmsgQ2lyY2xlY2l9XG4gKlxuICogQHNlZSBodHRwczovL2NpcmNsZWNpLmNvbS9kb2NzLzIuMC9jb25maWd1cmF0aW9uLXJlZmVyZW5jZS9cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBDaXJjbGVDaVByb3BzIHtcbiAgLyoqXG4gICAqIENvbnRhaW5zIGEgbWFwIG9mIENpcmNsQ2kgT3Jic1xuICAgKiBgYGBqc29uXG4gICAqIG9yYnM6IHtcbiAgICogIG5vZGU6IFwiY2lyY2xlY2kvbm9kZUA1LjAuMVwiXG4gICAqICBzbGFjazogXCJjaXJjbGVjaS9zbGFja0A0LjguM1wiXG4gICAqIH1cbiAgICogYGBgXG4gICAqL1xuICByZWFkb25seSBvcmJzPzogUmVjb3JkPHN0cmluZywgc3RyaW5nPjtcbiAgLyoqXG4gICAqIHBpcGVsaW5lIHZlcnNpb25cbiAgICpcbiAgICogQGRlZmF1bHQgMi4xXG4gICAqIEBzZWUgaHR0cHM6Ly9jaXJjbGVjaS5jb20vZG9jcy8yLjAvY29uZmlndXJhdGlvbi1yZWZlcmVuY2UvI3ZlcnNpb25cbiAgICovXG4gIHJlYWRvbmx5IHZlcnNpb24/OiBudW1iZXI7XG4gIC8qKlxuICAgKiBMaXN0IG9mIFdvcmtmbG93cyBvZiBwaXBlbGluZSwgZS5nLlxuICAgKiBgYGBqc29uXG4gICAqIHdvcmtmbG93czoge1xuICAgKiAgIHtcbiAgICogICAgIGlkZW50aWZpZXI6IFwiYnVpbGRcIixcbiAgICogICAgICAgam9iczogW3tcbiAgICogICAgICAgICAgaWRlbnRpZmllcjogXCJub2RlL2luc3RhbGxcIixcbiAgICogICAgICAgICAgY29udGV4dDogW1wibnBtXCJdLFxuICAgKiAgICAgICB9XVxuICAgKiAgIH1cbiAgICogfVxuICAgKiBgYGBcbiAgICpcbiAgICogQHNlZSBodHRwczovL2NpcmNsZWNpLmNvbS9kb2NzLzIuMC9jb25maWd1cmF0aW9uLXJlZmVyZW5jZS8jd29ya2Zsb3dzXG4gICAqL1xuICByZWFkb25seSB3b3JrZmxvd3M/OiBXb3JrZmxvd1tdO1xuICAvKipcbiAgICogTGlzdCBvZiBKb2JzIHRvIGNyZWF0ZSB1bmlxdWUgc3RlcHMgcGVyIHBpcGVsaW5lLCBlLmcuXG4gICAqIGBgYGpzb25cbiAgICogam9iczogW3tcbiAgICogIGlkZW50aWZpZXI6IFwiY29tcGlsZVwiLFxuICAgKiAgZG9ja2VyOiB7IGltYWdlOiBcImdvbGFuZzphbHBpbmVcIiB9XG4gICAqICBzdGVwczogW1wiY2hlY2tvdXRcIiwgcnVuOiB7Y29tbWFuZDogXCJnbyBidWlsZCAuXCJ9XVxuICAgKiB9XVxuICAgKiBgYGBcbiAgICpcbiAgICogQHNlZSBodHRwczovL2NpcmNsZWNpLmNvbS9kb2NzLzIuMC9jb25maWd1cmF0aW9uLXJlZmVyZW5jZS8jam9ic1xuICAgKi9cbiAgcmVhZG9ubHkgam9icz86IEpvYltdO1xuICAvKipcbiAgICogVGhlIHNldHVwIGZpZWxkIGVuYWJsZXMgeW91IHRvIGNvbmRpdGlvbmFsbHkgdHJpZ2dlciBjb25maWd1cmF0aW9ucyBmcm9tIG91dHNpZGVcbiAgICogdGhlIHByaW1hcnkgLmNpcmNsZWNpIHBhcmVudCBkaXJlY3RvcnksIHVwZGF0ZSBwaXBlbGluZSBwYXJhbWV0ZXJzLCBvciBnZW5lcmF0ZSBjdXN0b21pemVkIGNvbmZpZ3VyYXRpb25zLlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vY2lyY2xlY2kuY29tL2RvY3MvMi4wL2NvbmZpZ3VyYXRpb24tcmVmZXJlbmNlLyNzZXR1cFxuICAgKi9cbiAgcmVhZG9ubHkgc2V0dXA/OiBib29sZWFuO1xufVxuXG4vKipcbiAqIENpcmNsZWNpIENsYXNzIHRvIG1hbmFnZSBgLmNpcmNsZWNpL2NvbmZpZy55bWxgLlxuICogQ2hlY2sgcHJvamVuJ3MgZG9jcyBmb3IgbW9yZSBpbmZvcm1hdGlvbi5cbiAqXG4gKiBAc2VlIGh0dHBzOi8vY2lyY2xlY2kuY29tL2RvY3MvMi4wL2NvbmZpZ3VyYXRpb24tcmVmZXJlbmNlL1xuICovXG5leHBvcnQgY2xhc3MgQ2lyY2xlY2kgZXh0ZW5kcyBDb21wb25lbnQge1xuICAvKipcbiAgICogVGhlIHlhbWwgZmlsZSBmb3IgdGhlIENpcmNsZWNpIHBpcGVsaW5lXG4gICAqICovXG4gIHB1YmxpYyByZWFkb25seSBmaWxlOiBZYW1sRmlsZTtcbiAgLyoqXG4gICAqIGludGVybmFsIGNvcHkgb2Ygb3B0aW9ucyB0byBzaGFyZSBvcHRpb25zIGJldHdlZW4gZnVuY3Rpb25zXG4gICAqIEBwcml2YXRlXG4gICAqL1xuICBwcml2YXRlIG9wdGlvbnM6IENpcmNsZUNpUHJvcHM7XG4gIC8qKlxuICAgKiBpbnRlcm5hbCBtYXAgb2Ygb3Jic1xuICAgKiBAcHJpdmF0ZVxuICAgKi9cbiAgcHJpdmF0ZSByZWFkb25seSBvcmJzOiBSZWNvcmQ8c3RyaW5nLCBzdHJpbmc+O1xuICAvKipcbiAgICogaW50ZXJuYWwgbGlzdCBvZiB3b3JrZmxvd3NcbiAgICogQHByaXZhdGVcbiAgICovXG4gIHByaXZhdGUgd29ya2Zsb3dzOiBXb3JrZmxvd1tdO1xuICAvKipcbiAgICogaW50ZXJuYWwgbGlzdCBvZiBqb2JzXG4gICAqIEBwcml2YXRlXG4gICAqL1xuICBwcml2YXRlIHJlYWRvbmx5IGpvYnM6IEpvYltdO1xuXG4gIGNvbnN0cnVjdG9yKHByb2plY3Q6IFByb2plY3QsIG9wdGlvbnM6IENpcmNsZUNpUHJvcHMgPSB7fSkge1xuICAgIHN1cGVyKHByb2plY3QpO1xuICAgIHRoaXMub3B0aW9ucyA9IG9wdGlvbnM7XG4gICAgdGhpcy5vcmJzID0gb3B0aW9ucy5vcmJzID8/IHt9O1xuICAgIHRoaXMud29ya2Zsb3dzID0gb3B0aW9ucy53b3JrZmxvd3MgPz8gW107XG4gICAgdGhpcy5qb2JzID0gb3B0aW9ucy5qb2JzID8/IFtdO1xuICAgIHRoaXMuZmlsZSA9IG5ldyBZYW1sRmlsZShwcm9qZWN0LCBcIi5jaXJjbGVjaS9jb25maWcueW1sXCIsIHtcbiAgICAgIG9iajogKCkgPT4gdGhpcy5yZW5kZXJDaXJjbGVDaSgpLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIGZ1bmN0aW9uIHRvIHByZXBhcmUgdGhlIHJlbmRlcmluZyBvZiB0aGUgeWFtbCBmaWxlLlxuICAgKiBPYmplY3RzIHdpdGggZHluYW1pYyBrZXlzIGxpa2Ugd29ya2Zsb3dzIGFuZCBqb2JzIHJlcXVpcmVkIHRoZSBmaWVsZCBgaWRlbnRpZmllcmAgdG8gYmUgc2V0LlxuICAgKiBUaG9zZSBvYmplY3Qgd2lsbCBiZSByZWR1Y2VkIGJ5IGBpZGVudGlmaWVyYCBmaWVsZCBiZWZvcmUgcmVuZGVyaW5nXG4gICAqIEBwcml2YXRlXG4gICAqL1xuICBwcml2YXRlIHJlbmRlckNpcmNsZUNpKCkge1xuICAgIC8vIHJlbmRlciBkeW5hbWljIGtleXMgZm9yIHdvcmtmbG93XG4gICAgY29uc3Qgd29ya2Zsb3dSZWNvcmRzOiBSZWNvcmQ8c3RyaW5nLCBhbnk+ID0ge307XG4gICAgZm9yIChjb25zdCB3b3JrZmxvdyBvZiB0aGlzLndvcmtmbG93cykge1xuICAgICAgY29uc3QgeyBpZGVudGlmaWVyLCAuLi5yZWR1Y2VkIH0gPSB3b3JrZmxvdztcbiAgICAgIHJlZHVjZWQuam9icyA9IHRoaXMucmVuZGVySm9icyh3b3JrZmxvdy5qb2JzKTtcbiAgICAgIHdvcmtmbG93UmVjb3Jkc1tpZGVudGlmaWVyXSA9IHRoaXMuc25ha2VDYXNlKHJlZHVjZWQpO1xuICAgIH1cblxuICAgIC8vIHJlbmRlciBkeW5hbWljIGtleXMgZm9yIGpvYnNcbiAgICBjb25zdCBqb2JSZWNvcmRzOiBSZWNvcmQ8c3RyaW5nLCBhbnk+ID0ge307XG4gICAgZm9yIChjb25zdCBqb2Igb2YgdGhpcy5qb2JzKSB7XG4gICAgICBjb25zdCB7IGlkZW50aWZpZXIsIC4uLnJlZHVjZWQgfSA9IGpvYjtcbiAgICAgIGpvYlJlY29yZHNbaWRlbnRpZmllcl0gPSB0aGlzLnNuYWtlQ2FzZShyZWR1Y2VkKTtcbiAgICB9XG5cbiAgICByZXR1cm4ge1xuICAgICAgdmVyc2lvbjogdGhpcy5vcHRpb25zLnZlcnNpb24gfHwgMi4xLFxuICAgICAgc2V0dXA6IHRoaXMub3B0aW9ucy5zZXR1cCxcbiAgICAgIG9yYnM6IHRoaXMub3JicyxcbiAgICAgIGpvYnM6IE9iamVjdC5rZXlzKGpvYlJlY29yZHMpLmxlbmd0aCA+IDAgPyBqb2JSZWNvcmRzIDogdW5kZWZpbmVkLFxuICAgICAgd29ya2Zsb3dzOiB3b3JrZmxvd1JlY29yZHMsXG4gICAgfTtcbiAgfVxuXG4gIC8qKlxuICAgKiBhZGQgbmV3IHdvcmtmbG93IHRvIGV4aXN0aW5nIHBpcGVsaW5lXG4gICAqIEBwYXJhbSB3b3JrZmxvd1xuICAgKi9cbiAgcHVibGljIGFkZFdvcmtmbG93KHdvcmtmbG93OiBXb3JrZmxvdykge1xuICAgIHRoaXMud29ya2Zsb3dzID0gWy4uLnRoaXMud29ya2Zsb3dzLCB3b3JrZmxvd107XG4gIH1cblxuICAvKipcbiAgICogcmVkdWNlIG9iamVjdHMgd2l0aCBgaWRlbnRpZmllcmAgZmllbGQgZm9yIFdvcmtmbG93Sm9icy5cbiAgICogQSB3b3JrZmxvdyBqb2IgY2FuIGNvbnRhaW4gYG9yYlBhcmFtZXRlcmAgd2hpY2ggYXJlIHBhc3NlZCB0byBvcmJzLlxuICAgKiBUaGlzIG1hcCBpcyBkaXJlY3RseSBhZGRlZCBhcyBgUmVjb3JkPHN0cmluZyxhbnk+YCB0byBqb2IuXG4gICAqIFNvIHdlIGdvbm5hIGFkZCB0aG9zZSBhZnRlciB0aGUgZGVmYXVsdCBmaWVsZCBvZiBXb3JrZmxvd0pvYi5cbiAgICogQHNlZSBodHRwczovL2NpcmNsZWNpLmNvbS9kZXZlbG9wZXIvb3Jicy9vcmIvY2lyY2xlY2kvbm9kZSN1c2FnZS1pbnN0YWxsX25vZGVqc1xuICAgKiBAcGFyYW0gam9ic1xuICAgKi9cbiAgcHJpdmF0ZSByZW5kZXJKb2JzID0gKGpvYnM6IFdvcmtmbG93Sm9iW10gPSBbXSk6IGFueSA9PiB7XG4gICAgbGV0IHJlc3VsdDogYW55ID0gW107XG4gICAgZm9yIChjb25zdCBqb2Igb2Ygam9icyA/PyBbXSkge1xuICAgICAgY29uc3QgeyBpZGVudGlmaWVyLCBvcmJQYXJhbWV0ZXJzLCAuLi5yZWR1Y2VkIH0gPSBqb2I7XG4gICAgICBpZiAoaXNPYmplY3RDb250YWluaW5nRmllbGRFeGFjdGx5KGpvYiwgXCJpZGVudGlmaWVyXCIpKSB7XG4gICAgICAgIHJlc3VsdCA9IFsuLi5yZXN1bHQsIGlkZW50aWZpZXJdO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgcmVzdWx0ID0gW1xuICAgICAgICAgIC4uLnJlc3VsdCxcbiAgICAgICAgICB7IFtpZGVudGlmaWVyXTogeyAuLi5yZWR1Y2VkLCAuLi5vcmJQYXJhbWV0ZXJzIH0gfSxcbiAgICAgICAgXTtcbiAgICAgIH1cbiAgICB9XG4gICAgcmV0dXJuIHJlc3VsdDtcbiAgfTtcblxuICAvKipcbiAgICogQWRkIGEgQ2lyY2xlY2kgT3JiIHRvIHBpcGVsaW5lLiBXaWxsIHRocm93IGVycm9yIGlmIHRoZSBvcmIgYWxyZWFkeSBleGlzdHNcbiAgICogQHBhcmFtIG5hbWVcbiAgICogQHBhcmFtIG9yYlxuICAgKi9cbiAgcHVibGljIGFkZE9yYihuYW1lOiBzdHJpbmcsIG9yYjogc3RyaW5nKSB7XG4gICAgaWYgKHRoaXMub3Jic1tuYW1lXSAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYENpcmNsZWNpIENvbmZpZyBhbHJlYWR5IGNvbnRhaW5zIGFuIG9yYiBuYW1lZCAke25hbWV9LmApO1xuICAgIH1cbiAgICB0aGlzLm9yYnNbbmFtZV0gPSBvcmI7XG4gIH1cblxuICAvKipcbiAgICogU25ha2UgY2FzZSBmb3IgbGlzdGVkIGtleXMuIFRoZXJlIGFyZSB0b28gbWFueSBleGNlcHRpb25zIG5lZWRlZCB0byBkbyBpdCByZWN1cnNpdmUgd2l0aG91dCBhIHdoaXRlbGlzdC5cbiAgICogVGhpcyBsaXN0IG5lZWRzIHRvIGJlIHVwZGF0ZWQgb25jZSB3ZSBhZGQgZmllbGQgd2l0aCBzbmFrZSBjYXNlIGxpa2UgYGF3c19hdXRoYC5cbiAgICogQHBhcmFtIGlucHV0XG4gICAqL1xuICBwcml2YXRlIHNuYWtlQ2FzZSA9IChpbnB1dDogYW55KTogYW55ID0+IHtcbiAgICBjb25zdCBzbmFrZUNhc2VLZXlXb3JkcyA9IFtcbiAgICAgIFwiYXdzQXV0aFwiLFxuICAgICAgXCJ3b3JraW5nRGlyZWN0b3J5XCIsXG4gICAgICBcInJlc291cmNlQ2xhc3NcIixcbiAgICAgIFwiZG9ja2VyTGF5ZXJDYWNoaW5nXCIsXG4gICAgICBcIm5vT3V0cHV0VGltZW91dFwiLFxuICAgIF07XG4gICAgcmV0dXJuIHNuYWtlQ2FzZUtleXMoaW5wdXQsIHRydWUsIHNuYWtlQ2FzZUtleVdvcmRzKTtcbiAgfTtcbn1cblxuZXhwb3J0IGNvbnN0IGlzT2JqZWN0Q29udGFpbmluZ0ZpZWxkRXhhY3RseSA9IChcbiAgb2JqOiBhbnksXG4gIGZpZWxkOiBzdHJpbmdcbik6IGJvb2xlYW4gPT4ge1xuICByZXR1cm4gT2JqZWN0LmtleXMob2JqKS5sZW5ndGggPT0gMSAmJiBPYmplY3Qua2V5cyhvYmopLmluY2x1ZGVzKGZpZWxkKTtcbn07XG4iXX0=