"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FILE_MANIFEST = void 0;
exports.cleanup = cleanup;
const fs = require("fs");
const path = require("path");
const glob = require("glob");
const common_1 = require("./common");
const logging = require("./logging");
exports.FILE_MANIFEST = `${common_1.PROJEN_DIR}/files.json`;
function cleanup(dir, newFiles, exclude) {
    try {
        const manifestFiles = getFilesFromManifest(dir);
        if (manifestFiles.length > 0) {
            // Use `FILE_MANIFEST` to remove files that are no longer managed by projen
            removeFiles(findOrphanedFiles(dir, manifestFiles, newFiles));
        }
        else {
            // Remove all files managed by projen with legacy logic
            removeFiles(findGeneratedFiles(dir, exclude));
        }
    }
    catch (e) {
        logging.warn(`warning: failed to clean up generated files: ${e.stack}`);
    }
}
function removeFiles(files) {
    for (const file of files) {
        fs.rmSync(file, { force: true, recursive: true });
    }
}
function findGeneratedFiles(dir, exclude) {
    const ignore = [
        ...readGitIgnore(dir),
        "node_modules/**",
        ...exclude,
        ".git/**",
    ];
    const files = glob.sync("**", {
        ignore,
        cwd: dir,
        dot: true,
        nodir: true,
        absolute: true,
    });
    const generated = new Array();
    for (const file of files) {
        const contents = fs.readFileSync(file, "utf-8");
        if (contents.includes(common_1.PROJEN_MARKER)) {
            generated.push(file);
        }
    }
    return generated;
}
function findOrphanedFiles(dir, oldFiles, newFiles) {
    return oldFiles
        .filter((old) => !newFiles.includes(old))
        .map((f) => path.resolve(dir, f));
}
function getFilesFromManifest(dir) {
    try {
        const fileManifestPath = path.resolve(dir, exports.FILE_MANIFEST);
        if (fs.existsSync(fileManifestPath)) {
            const fileManifest = JSON.parse(fs.readFileSync(fileManifestPath, "utf-8"));
            if (fileManifest.files) {
                return fileManifest.files;
            }
        }
    }
    catch (e) {
        logging.warn(`warning: unable to get files to clean from file manifest: ${e.stack}`);
    }
    return [];
}
function readGitIgnore(dir) {
    const filepath = path.join(dir, ".gitignore");
    if (!fs.existsSync(filepath)) {
        return [];
    }
    return fs
        .readFileSync(filepath, "utf-8")
        .split("\n")
        .filter((x) => x?.trim() !== "")
        .filter((x) => !x.startsWith("#") && !x.startsWith("!"))
        .map((x) => x.replace(/^\//, "")) // remove "/" prefix
        .map((x) => `${x}\n${x}/**`)
        .join("\n")
        .split("\n");
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2xlYW51cC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uL3NyYy9jbGVhbnVwLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQVFBLDBCQWFDO0FBckJELHlCQUF5QjtBQUN6Qiw2QkFBNkI7QUFDN0IsNkJBQTZCO0FBQzdCLHFDQUFxRDtBQUNyRCxxQ0FBcUM7QUFFeEIsUUFBQSxhQUFhLEdBQUcsR0FBRyxtQkFBVSxhQUFhLENBQUM7QUFFeEQsU0FBZ0IsT0FBTyxDQUFDLEdBQVcsRUFBRSxRQUFrQixFQUFFLE9BQWlCO0lBQ3hFLElBQUksQ0FBQztRQUNILE1BQU0sYUFBYSxHQUFHLG9CQUFvQixDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQ2hELElBQUksYUFBYSxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUUsQ0FBQztZQUM3QiwyRUFBMkU7WUFDM0UsV0FBVyxDQUFDLGlCQUFpQixDQUFDLEdBQUcsRUFBRSxhQUFhLEVBQUUsUUFBUSxDQUFDLENBQUMsQ0FBQztRQUMvRCxDQUFDO2FBQU0sQ0FBQztZQUNOLHVEQUF1RDtZQUN2RCxXQUFXLENBQUMsa0JBQWtCLENBQUMsR0FBRyxFQUFFLE9BQU8sQ0FBQyxDQUFDLENBQUM7UUFDaEQsQ0FBQztJQUNILENBQUM7SUFBQyxPQUFPLENBQU0sRUFBRSxDQUFDO1FBQ2hCLE9BQU8sQ0FBQyxJQUFJLENBQUMsZ0RBQWdELENBQUMsQ0FBQyxLQUFLLEVBQUUsQ0FBQyxDQUFDO0lBQzFFLENBQUM7QUFDSCxDQUFDO0FBRUQsU0FBUyxXQUFXLENBQUMsS0FBZTtJQUNsQyxLQUFLLE1BQU0sSUFBSSxJQUFJLEtBQUssRUFBRSxDQUFDO1FBQ3pCLEVBQUUsQ0FBQyxNQUFNLENBQUMsSUFBSSxFQUFFLEVBQUUsS0FBSyxFQUFFLElBQUksRUFBRSxTQUFTLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztJQUNwRCxDQUFDO0FBQ0gsQ0FBQztBQUVELFNBQVMsa0JBQWtCLENBQUMsR0FBVyxFQUFFLE9BQWlCO0lBQ3hELE1BQU0sTUFBTSxHQUFHO1FBQ2IsR0FBRyxhQUFhLENBQUMsR0FBRyxDQUFDO1FBQ3JCLGlCQUFpQjtRQUNqQixHQUFHLE9BQU87UUFDVixTQUFTO0tBQ1YsQ0FBQztJQUVGLE1BQU0sS0FBSyxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFO1FBQzVCLE1BQU07UUFDTixHQUFHLEVBQUUsR0FBRztRQUNSLEdBQUcsRUFBRSxJQUFJO1FBQ1QsS0FBSyxFQUFFLElBQUk7UUFDWCxRQUFRLEVBQUUsSUFBSTtLQUNmLENBQUMsQ0FBQztJQUVILE1BQU0sU0FBUyxHQUFHLElBQUksS0FBSyxFQUFVLENBQUM7SUFFdEMsS0FBSyxNQUFNLElBQUksSUFBSSxLQUFLLEVBQUUsQ0FBQztRQUN6QixNQUFNLFFBQVEsR0FBRyxFQUFFLENBQUMsWUFBWSxDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsQ0FBQztRQUVoRCxJQUFJLFFBQVEsQ0FBQyxRQUFRLENBQUMsc0JBQWEsQ0FBQyxFQUFFLENBQUM7WUFDckMsU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUN2QixDQUFDO0lBQ0gsQ0FBQztJQUVELE9BQU8sU0FBUyxDQUFDO0FBQ25CLENBQUM7QUFFRCxTQUFTLGlCQUFpQixDQUN4QixHQUFXLEVBQ1gsUUFBa0IsRUFDbEIsUUFBa0I7SUFFbEIsT0FBTyxRQUFRO1NBQ1osTUFBTSxDQUFDLENBQUMsR0FBRyxFQUFFLEVBQUUsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLENBQUM7U0FDeEMsR0FBRyxDQUFDLENBQUMsQ0FBUyxFQUFFLEVBQUUsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDO0FBQzlDLENBQUM7QUFFRCxTQUFTLG9CQUFvQixDQUFDLEdBQVc7SUFDdkMsSUFBSSxDQUFDO1FBQ0gsTUFBTSxnQkFBZ0IsR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxxQkFBYSxDQUFDLENBQUM7UUFDMUQsSUFBSSxFQUFFLENBQUMsVUFBVSxDQUFDLGdCQUFnQixDQUFDLEVBQUUsQ0FBQztZQUNwQyxNQUFNLFlBQVksR0FBRyxJQUFJLENBQUMsS0FBSyxDQUM3QixFQUFFLENBQUMsWUFBWSxDQUFDLGdCQUFnQixFQUFFLE9BQU8sQ0FBQyxDQUMzQyxDQUFDO1lBQ0YsSUFBSSxZQUFZLENBQUMsS0FBSyxFQUFFLENBQUM7Z0JBQ3ZCLE9BQU8sWUFBWSxDQUFDLEtBQUssQ0FBQztZQUM1QixDQUFDO1FBQ0gsQ0FBQztJQUNILENBQUM7SUFBQyxPQUFPLENBQU0sRUFBRSxDQUFDO1FBQ2hCLE9BQU8sQ0FBQyxJQUFJLENBQ1YsNkRBQTZELENBQUMsQ0FBQyxLQUFLLEVBQUUsQ0FDdkUsQ0FBQztJQUNKLENBQUM7SUFFRCxPQUFPLEVBQUUsQ0FBQztBQUNaLENBQUM7QUFFRCxTQUFTLGFBQWEsQ0FBQyxHQUFXO0lBQ2hDLE1BQU0sUUFBUSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRyxFQUFFLFlBQVksQ0FBQyxDQUFDO0lBQzlDLElBQUksQ0FBQyxFQUFFLENBQUMsVUFBVSxDQUFDLFFBQVEsQ0FBQyxFQUFFLENBQUM7UUFDN0IsT0FBTyxFQUFFLENBQUM7SUFDWixDQUFDO0lBRUQsT0FBTyxFQUFFO1NBQ04sWUFBWSxDQUFDLFFBQVEsRUFBRSxPQUFPLENBQUM7U0FDL0IsS0FBSyxDQUFDLElBQUksQ0FBQztTQUNYLE1BQU0sQ0FBQyxDQUFDLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQyxFQUFFLElBQUksRUFBRSxLQUFLLEVBQUUsQ0FBQztTQUMvQixNQUFNLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLFVBQVUsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxVQUFVLENBQUMsR0FBRyxDQUFDLENBQUM7U0FDdkQsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDLG9CQUFvQjtTQUNyRCxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDO1NBQzNCLElBQUksQ0FBQyxJQUFJLENBQUM7U0FDVixLQUFLLENBQUMsSUFBSSxDQUFDLENBQUM7QUFDakIsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGZzIGZyb20gXCJmc1wiO1xuaW1wb3J0ICogYXMgcGF0aCBmcm9tIFwicGF0aFwiO1xuaW1wb3J0ICogYXMgZ2xvYiBmcm9tIFwiZ2xvYlwiO1xuaW1wb3J0IHsgUFJPSkVOX0RJUiwgUFJPSkVOX01BUktFUiB9IGZyb20gXCIuL2NvbW1vblwiO1xuaW1wb3J0ICogYXMgbG9nZ2luZyBmcm9tIFwiLi9sb2dnaW5nXCI7XG5cbmV4cG9ydCBjb25zdCBGSUxFX01BTklGRVNUID0gYCR7UFJPSkVOX0RJUn0vZmlsZXMuanNvbmA7XG5cbmV4cG9ydCBmdW5jdGlvbiBjbGVhbnVwKGRpcjogc3RyaW5nLCBuZXdGaWxlczogc3RyaW5nW10sIGV4Y2x1ZGU6IHN0cmluZ1tdKSB7XG4gIHRyeSB7XG4gICAgY29uc3QgbWFuaWZlc3RGaWxlcyA9IGdldEZpbGVzRnJvbU1hbmlmZXN0KGRpcik7XG4gICAgaWYgKG1hbmlmZXN0RmlsZXMubGVuZ3RoID4gMCkge1xuICAgICAgLy8gVXNlIGBGSUxFX01BTklGRVNUYCB0byByZW1vdmUgZmlsZXMgdGhhdCBhcmUgbm8gbG9uZ2VyIG1hbmFnZWQgYnkgcHJvamVuXG4gICAgICByZW1vdmVGaWxlcyhmaW5kT3JwaGFuZWRGaWxlcyhkaXIsIG1hbmlmZXN0RmlsZXMsIG5ld0ZpbGVzKSk7XG4gICAgfSBlbHNlIHtcbiAgICAgIC8vIFJlbW92ZSBhbGwgZmlsZXMgbWFuYWdlZCBieSBwcm9qZW4gd2l0aCBsZWdhY3kgbG9naWNcbiAgICAgIHJlbW92ZUZpbGVzKGZpbmRHZW5lcmF0ZWRGaWxlcyhkaXIsIGV4Y2x1ZGUpKTtcbiAgICB9XG4gIH0gY2F0Y2ggKGU6IGFueSkge1xuICAgIGxvZ2dpbmcud2Fybihgd2FybmluZzogZmFpbGVkIHRvIGNsZWFuIHVwIGdlbmVyYXRlZCBmaWxlczogJHtlLnN0YWNrfWApO1xuICB9XG59XG5cbmZ1bmN0aW9uIHJlbW92ZUZpbGVzKGZpbGVzOiBzdHJpbmdbXSkge1xuICBmb3IgKGNvbnN0IGZpbGUgb2YgZmlsZXMpIHtcbiAgICBmcy5ybVN5bmMoZmlsZSwgeyBmb3JjZTogdHJ1ZSwgcmVjdXJzaXZlOiB0cnVlIH0pO1xuICB9XG59XG5cbmZ1bmN0aW9uIGZpbmRHZW5lcmF0ZWRGaWxlcyhkaXI6IHN0cmluZywgZXhjbHVkZTogc3RyaW5nW10pIHtcbiAgY29uc3QgaWdub3JlID0gW1xuICAgIC4uLnJlYWRHaXRJZ25vcmUoZGlyKSxcbiAgICBcIm5vZGVfbW9kdWxlcy8qKlwiLFxuICAgIC4uLmV4Y2x1ZGUsXG4gICAgXCIuZ2l0LyoqXCIsXG4gIF07XG5cbiAgY29uc3QgZmlsZXMgPSBnbG9iLnN5bmMoXCIqKlwiLCB7XG4gICAgaWdub3JlLFxuICAgIGN3ZDogZGlyLFxuICAgIGRvdDogdHJ1ZSxcbiAgICBub2RpcjogdHJ1ZSxcbiAgICBhYnNvbHV0ZTogdHJ1ZSxcbiAgfSk7XG5cbiAgY29uc3QgZ2VuZXJhdGVkID0gbmV3IEFycmF5PHN0cmluZz4oKTtcblxuICBmb3IgKGNvbnN0IGZpbGUgb2YgZmlsZXMpIHtcbiAgICBjb25zdCBjb250ZW50cyA9IGZzLnJlYWRGaWxlU3luYyhmaWxlLCBcInV0Zi04XCIpO1xuXG4gICAgaWYgKGNvbnRlbnRzLmluY2x1ZGVzKFBST0pFTl9NQVJLRVIpKSB7XG4gICAgICBnZW5lcmF0ZWQucHVzaChmaWxlKTtcbiAgICB9XG4gIH1cblxuICByZXR1cm4gZ2VuZXJhdGVkO1xufVxuXG5mdW5jdGlvbiBmaW5kT3JwaGFuZWRGaWxlcyhcbiAgZGlyOiBzdHJpbmcsXG4gIG9sZEZpbGVzOiBzdHJpbmdbXSxcbiAgbmV3RmlsZXM6IHN0cmluZ1tdXG4pIHtcbiAgcmV0dXJuIG9sZEZpbGVzXG4gICAgLmZpbHRlcigob2xkKSA9PiAhbmV3RmlsZXMuaW5jbHVkZXMob2xkKSlcbiAgICAubWFwKChmOiBzdHJpbmcpID0+IHBhdGgucmVzb2x2ZShkaXIsIGYpKTtcbn1cblxuZnVuY3Rpb24gZ2V0RmlsZXNGcm9tTWFuaWZlc3QoZGlyOiBzdHJpbmcpOiBzdHJpbmdbXSB7XG4gIHRyeSB7XG4gICAgY29uc3QgZmlsZU1hbmlmZXN0UGF0aCA9IHBhdGgucmVzb2x2ZShkaXIsIEZJTEVfTUFOSUZFU1QpO1xuICAgIGlmIChmcy5leGlzdHNTeW5jKGZpbGVNYW5pZmVzdFBhdGgpKSB7XG4gICAgICBjb25zdCBmaWxlTWFuaWZlc3QgPSBKU09OLnBhcnNlKFxuICAgICAgICBmcy5yZWFkRmlsZVN5bmMoZmlsZU1hbmlmZXN0UGF0aCwgXCJ1dGYtOFwiKVxuICAgICAgKTtcbiAgICAgIGlmIChmaWxlTWFuaWZlc3QuZmlsZXMpIHtcbiAgICAgICAgcmV0dXJuIGZpbGVNYW5pZmVzdC5maWxlcztcbiAgICAgIH1cbiAgICB9XG4gIH0gY2F0Y2ggKGU6IGFueSkge1xuICAgIGxvZ2dpbmcud2FybihcbiAgICAgIGB3YXJuaW5nOiB1bmFibGUgdG8gZ2V0IGZpbGVzIHRvIGNsZWFuIGZyb20gZmlsZSBtYW5pZmVzdDogJHtlLnN0YWNrfWBcbiAgICApO1xuICB9XG5cbiAgcmV0dXJuIFtdO1xufVxuXG5mdW5jdGlvbiByZWFkR2l0SWdub3JlKGRpcjogc3RyaW5nKSB7XG4gIGNvbnN0IGZpbGVwYXRoID0gcGF0aC5qb2luKGRpciwgXCIuZ2l0aWdub3JlXCIpO1xuICBpZiAoIWZzLmV4aXN0c1N5bmMoZmlsZXBhdGgpKSB7XG4gICAgcmV0dXJuIFtdO1xuICB9XG5cbiAgcmV0dXJuIGZzXG4gICAgLnJlYWRGaWxlU3luYyhmaWxlcGF0aCwgXCJ1dGYtOFwiKVxuICAgIC5zcGxpdChcIlxcblwiKVxuICAgIC5maWx0ZXIoKHgpID0+IHg/LnRyaW0oKSAhPT0gXCJcIilcbiAgICAuZmlsdGVyKCh4KSA9PiAheC5zdGFydHNXaXRoKFwiI1wiKSAmJiAheC5zdGFydHNXaXRoKFwiIVwiKSlcbiAgICAubWFwKCh4KSA9PiB4LnJlcGxhY2UoL15cXC8vLCBcIlwiKSkgLy8gcmVtb3ZlIFwiL1wiIHByZWZpeFxuICAgIC5tYXAoKHgpID0+IGAke3h9XFxuJHt4fS8qKmApXG4gICAgLmpvaW4oXCJcXG5cIilcbiAgICAuc3BsaXQoXCJcXG5cIik7XG59XG4iXX0=