"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.discoverTaskCommands = discoverTaskCommands;
const chalk = require("chalk");
const helpers_1 = require("yargs/helpers");
const logging = require("../logging");
/**
 * Reads .projen/tasks.json and adds CLI commands for all tasks.
 * @param ya yargs
 */
function discoverTaskCommands(runtime, ya) {
    const tasks = runtime.manifest.tasks ?? {};
    for (const task of Object.values(tasks)) {
        ya.command(task.name, task.description ?? "", taskCommandHandler(task));
    }
    function taskCommandHandler(task) {
        return (args) => {
            const taskReceivesArgs = task.steps?.reduce((receiveArgs, step) => receiveArgs || (step.receiveArgs ?? false), false);
            let taskArgs = [];
            if (taskReceivesArgs) {
                args.help(false);
                args.strict(false);
                args.strictCommands(false);
                taskArgs = (0, helpers_1.hideBin)(process.argv).slice(1);
            }
            args.option("inspect", {
                alias: "i",
                desc: "show all steps in this task",
            });
            const argv = args.argv;
            if (argv.inspect) {
                return inspectTask(task.name);
            }
            else {
                try {
                    runtime.runTask(task.name, [], taskArgs);
                }
                catch (e) {
                    logging.error(e.message);
                    process.exit(1);
                }
            }
        };
    }
    function inspectTask(name, indent = 0) {
        const writeln = (s) => console.log(" ".repeat(indent) + s);
        const task = runtime.tryFindTask(name);
        if (!task) {
            throw new Error(`${name}: unable to resolve subtask with name "${name}"`);
        }
        if (task.description) {
            writeln(`${chalk.underline("description")}: ${task.description}`);
        }
        for (const [k, v] of Object.entries(task.env ?? {})) {
            writeln(`${chalk.underline("env")}: ${k}=${v}`);
        }
        for (const step of task.steps ?? []) {
            if (step.spawn) {
                writeln(`- ${chalk.bold(step.spawn)}`);
                inspectTask(step.spawn, indent + 2);
            }
            else if (step.exec) {
                writeln(`- exec: ${step.exec}`);
            }
            else if (step.builtin) {
                writeln(`- builtin: ${step.builtin}`);
            }
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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