"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitAttributesFile = exports.EndOfLine = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const file_1 = require("./file");
/**
 * The end of line characters supported by git.
 */
var EndOfLine;
(function (EndOfLine) {
    /**
     * Maintain existing (mixed values within one file are normalised by looking
     * at what's used after the first line)
     */
    EndOfLine["AUTO"] = "auto";
    /**
     * Carriage Return + Line Feed characters (\r\n), common on Windows
     */
    EndOfLine["CRLF"] = "crlf";
    /**
     * Line Feed only (\n), common on Linux and macOS as well as inside git repos
     */
    EndOfLine["LF"] = "lf";
    /**
     * Disable and do not configure the end of line character
     */
    EndOfLine["NONE"] = "none";
})(EndOfLine || (exports.EndOfLine = EndOfLine = {}));
/**
 * Assign attributes to file names in a git repository.
 *
 * @see https://git-scm.com/docs/gitattributes
 */
class GitAttributesFile extends file_1.FileBase {
    constructor(scope, options) {
        super(scope, ".gitattributes", {
            editGitignore: false,
        });
        this.attributes = new Map();
        this.endOfLine = options?.endOfLine ?? EndOfLine.LF;
        if (this.endOfLine != EndOfLine.NONE) {
            let endOfLineAttributes = [`text=auto`];
            if (this.endOfLine != EndOfLine.AUTO) {
                endOfLineAttributes.push(`eol=${this.endOfLine}`);
            }
            // Setting a default end of line for all text files in the repository
            // This line should be the first one in order to use it as a default for text files and allow for overriding in later lines
            this.addAttributes("*", ...endOfLineAttributes);
        }
    }
    /**
     * Maps a set of attributes to a set of files.
     * @param glob Glob pattern to match files in the repo
     * @param attributes Attributes to assign to these files.
     */
    addAttributes(glob, ...attributes) {
        if (!this.attributes.has(glob)) {
            this.attributes.set(glob, new Set());
        }
        const set = this.attributes.get(glob);
        for (const attribute of attributes) {
            set.add(attribute);
        }
    }
    /**
     * Add attributes necessary to mark these files as stored in LFS
     */
    addLfsPattern(glob) {
        this.addAttributes(glob, "filter=lfs", "diff=lfs", "merge=lfs", "-text");
    }
    /**
     * Whether the current gitattributes file has any LFS patterns
     */
    get hasLfsPatterns() {
        return Array.from(this.attributes.values()).some((attrs) => attrs.has("filter=lfs"));
    }
    preSynthesize() {
        this.project.addPackageIgnore("/.gitattributes");
        super.preSynthesize();
    }
    synthesizeContent(_) {
        // We can assume the file map is never empty.
        const entries = Array.from(this.attributes.entries()).sort(([l], [r]) => l.localeCompare(r));
        if (entries.length === 0) {
            return undefined;
        }
        return [
            ...(this.marker ? [`# ${this.marker}`] : []),
            "",
            ...entries.map(([name, attributes]) => `${name} ${Array.from(attributes).join(" ")}`),
        ].join("\n");
    }
}
exports.GitAttributesFile = GitAttributesFile;
_a = JSII_RTTI_SYMBOL_1;
GitAttributesFile[_a] = { fqn: "projen.GitAttributesFile", version: "0.91.13" };
//# sourceMappingURL=data:application/json;base64,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