"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AutoQueue = exports.MergeMethod = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const component_1 = require("../component");
const gh = require("../github");
/**
 * The merge method used to add the PR to the merge queue
 *
 * Behavior can be further configured in repository settings.
 */
var MergeMethod;
(function (MergeMethod) {
    MergeMethod["SQUASH"] = "squash";
    MergeMethod["MERGE"] = "merge";
    MergeMethod["REBASE"] = "rebase";
})(MergeMethod || (exports.MergeMethod = MergeMethod = {}));
/**
 * Automatically add pull requests to the merge queue
 * PRs will be merged once they pass required checks.
 */
class AutoQueue extends component_1.Component {
    constructor(scope, options = {}) {
        super(scope);
        const workflowEngine = gh.GitHub.of(this.project);
        if (!workflowEngine) {
            throw new Error(`Cannot add ${new.target.name} to project without GitHub enabled. Please enable GitHub for this project.`);
        }
        const labels = options.labels ?? [];
        const usernames = options.allowedUsernames ?? [];
        const conditions = [];
        if (labels.length > 0) {
            conditions.push("(" +
                labels
                    .map((l) => `contains(github.event.pull_request.labels.*.name, '${l}')`)
                    .join(" || ") +
                ")");
        }
        if (usernames.length > 0) {
            conditions.push("(" +
                usernames
                    .map((u) => `github.event.pull_request.user.login == '${u}'`)
                    .join(" || ") +
                ")");
        }
        const credentials = options.projenCredentials ?? workflowEngine.projenCredentials;
        const mergeMethod = options.mergeMethod ?? MergeMethod.SQUASH;
        const autoQueueJob = {
            name: "Set AutoQueue on PR #${{ github.event.number }}",
            runsOn: options.runsOn ?? ["ubuntu-latest"],
            permissions: {
                pullRequests: gh.workflows.JobPermission.WRITE,
                contents: gh.workflows.JobPermission.WRITE,
            },
            if: conditions.length ? conditions.join(" && ") : undefined,
            steps: [
                ...credentials.setupSteps,
                {
                    uses: "peter-evans/enable-pull-request-automerge@v3",
                    with: {
                        token: credentials.tokenRef,
                        "pull-request-number": "${{ github.event.number }}",
                        "merge-method": mergeMethod,
                    },
                },
            ],
        };
        const workflow = workflowEngine.addWorkflow("auto-queue");
        workflow.on({
            // The 'pull request' event gives the workflow 'read-only' permissions on some
            // pull requests (such as the ones from dependabot) when using the `GITHUB_TOKEN`
            // security token. This prevents the workflow from approving these pull requests.
            // Github has placed this guard so as to prevent security attacks by simply opening
            // a pull request and triggering a workflow on a commit that was not vetted to make
            // unintended changes to the repository.
            //
            // Instead use the 'pull request target' event here that gives the Github workflow
            // 'read-write' permissions. This is safe because, this event, unlike the 'pull request'
            // event references the BASE commit of the pull request and not the HEAD commit.
            //
            // We only enable auto-queue when a PR is opened, reopened or moving from Draft to Ready.
            // That way a user can always disable auto-queue if they want to and the workflow will
            // not automatically re-enable it, unless one of the events occurs.
            pullRequestTarget: {
                types: ["opened", "reopened", "ready_for_review"],
            },
        });
        workflow.addJobs({ enableAutoQueue: autoQueueJob });
    }
}
exports.AutoQueue = AutoQueue;
_a = JSII_RTTI_SYMBOL_1;
AutoQueue[_a] = { fqn: "projen.github.AutoQueue", version: "0.91.13" };
//# sourceMappingURL=data:application/json;base64,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