"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Dependabot = exports.VersioningStrategy = exports.DependabotScheduleInterval = exports.DependabotRegistryType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const component_1 = require("../component");
const util_1 = require("../util");
const yaml_1 = require("../yaml");
/**
 * Each configuration type requires you to provide particular settings.
 * Some types allow more than one way to connect
 * @see https://docs.github.com/en/code-security/supply-chain-security/keeping-your-dependencies-updated-automatically/configuration-options-for-dependency-updates#configuration-options-for-private-registries
 */
var DependabotRegistryType;
(function (DependabotRegistryType) {
    /**
     * The composer-repository type supports username and password.
     * @see https://docs.github.com/en/code-security/supply-chain-security/keeping-your-dependencies-updated-automatically/configuration-options-for-dependency-updates#composer-repository
     */
    DependabotRegistryType["COMPOSER_REGISTRY"] = "composer-registry";
    /**
     * The docker-registry type supports username and password.
     * The docker-registry type can also be used to pull from Amazon ECR using static AWS credentials
     * @see https://docs.github.com/en/code-security/supply-chain-security/keeping-your-dependencies-updated-automatically/configuration-options-for-dependency-updates#docker-registry
     */
    DependabotRegistryType["DOCKER_REGISTRY"] = "docker-registry";
    /**
     * The git type supports username and password
     * @see https://docs.github.com/en/code-security/supply-chain-security/keeping-your-dependencies-updated-automatically/configuration-options-for-dependency-updates#git
     */
    DependabotRegistryType["GIT"] = "git";
    /**
     * The hex-organization type supports organization and key
     * @see https://docs.github.com/en/code-security/supply-chain-security/keeping-your-dependencies-updated-automatically/configuration-options-for-dependency-updates#hex-organization
     */
    DependabotRegistryType["HEX_ORGANIZATION"] = "hex-organization";
    /**
     * The maven-repository type supports username and password, or token
     * @see https://docs.github.com/en/code-security/supply-chain-security/keeping-your-dependencies-updated-automatically/configuration-options-for-dependency-updates#maven-repository
     */
    DependabotRegistryType["MAVEN_REPOSITORY"] = "maven-repository";
    /**
     * The npm-registry type supports username and password, or token
     * @see https://docs.github.com/en/code-security/supply-chain-security/keeping-your-dependencies-updated-automatically/configuration-options-for-dependency-updates#npm-registry
     */
    DependabotRegistryType["NPM_REGISTRY"] = "npm-registry";
    /**
     * The nuget-feed type supports username and password, or token
     * @see https://docs.github.com/en/code-security/supply-chain-security/keeping-your-dependencies-updated-automatically/configuration-options-for-dependency-updates#nuget-feed
     */
    DependabotRegistryType["NUGET_FEED"] = "nuget-feed";
    /**
     * The python-index type supports username and password, or token
     * @see https://docs.github.com/en/code-security/supply-chain-security/keeping-your-dependencies-updated-automatically/configuration-options-for-dependency-updates#python-index
     */
    DependabotRegistryType["PYTHON_INDEX"] = "python-index";
    /**
     * The rubygems-server type supports username and password, or token
     * @see https://docs.github.com/en/code-security/supply-chain-security/keeping-your-dependencies-updated-automatically/configuration-options-for-dependency-updates#rubygems-server
     */
    DependabotRegistryType["RUBYGEMS_SERVER"] = "rubygems-server";
    /**
     * The terraform-registry type supports a token
     * @see https://docs.github.com/en/code-security/supply-chain-security/keeping-your-dependencies-updated-automatically/configuration-options-for-dependency-updates#terraform-registry
     */
    DependabotRegistryType["TERRAFORM_REGISTRY"] = "terraform-registry";
})(DependabotRegistryType || (exports.DependabotRegistryType = DependabotRegistryType = {}));
/**
 * How often to check for new versions and raise pull requests for version
 * updates.
 */
var DependabotScheduleInterval;
(function (DependabotScheduleInterval) {
    /**
     * Runs on every weekday, Monday to Friday.
     */
    DependabotScheduleInterval["DAILY"] = "daily";
    /**
     * Runs once each week. By default, this is on Monday.
     */
    DependabotScheduleInterval["WEEKLY"] = "weekly";
    /**
     * Runs once each month. This is on the first day of the month.
     */
    DependabotScheduleInterval["MONTHLY"] = "monthly";
})(DependabotScheduleInterval || (exports.DependabotScheduleInterval = DependabotScheduleInterval = {}));
/**
 * The strategy to use when edits manifest and lock files.
 */
var VersioningStrategy;
(function (VersioningStrategy) {
    /**
     * Only create pull requests to update lockfiles updates. Ignore any new
     * versions that would require package manifest changes.
     */
    VersioningStrategy["LOCKFILE_ONLY"] = "lockfile-only";
    /**
     * - For apps, the version requirements are increased.
     * - For libraries, the range of versions is widened.
     */
    VersioningStrategy["AUTO"] = "auto";
    /**
     * Relax the version requirement to include both the new and old version, when
     * possible.
     */
    VersioningStrategy["WIDEN"] = "widen";
    /**
     * Always increase the version requirement to match the new version.
     */
    VersioningStrategy["INCREASE"] = "increase";
    /**
     * Increase the version requirement only when required by the new version.
     */
    VersioningStrategy["INCREASE_IF_NECESSARY"] = "increase-if-necessary";
})(VersioningStrategy || (exports.VersioningStrategy = VersioningStrategy = {}));
/**
 * Defines dependabot configuration for node projects.
 *
 * Since module versions are managed in projen, the versioning strategy will be
 * configured to "lockfile-only" which means that only updates that can be done
 * on the lockfile itself will be proposed.
 */
class Dependabot extends component_1.Component {
    constructor(github, options = {}) {
        super(github.project);
        const project = github.project;
        this.allow = [];
        this.ignore = [];
        this.ignoresProjen = options.ignoreProjen ?? true;
        const registries = options.registries
            ? (0, util_1.kebabCaseKeys)(options.registries)
            : undefined;
        const groups = options.groups ? (0, util_1.kebabCaseKeys)(options.groups) : undefined;
        this.config = {
            version: 2,
            registries,
            updates: [
                {
                    "package-ecosystem": "npm",
                    "versioning-strategy": "lockfile-only",
                    directory: "/",
                    schedule: {
                        interval: options.scheduleInterval ?? DependabotScheduleInterval.DAILY,
                    },
                    allow: () => (this.allow.length > 0 ? this.allow : undefined),
                    ignore: () => (this.ignore.length > 0 ? this.ignore : undefined),
                    labels: options.labels ? options.labels : undefined,
                    registries: registries ? Object.keys(registries) : undefined,
                    groups: groups ? groups : undefined,
                    assignees: options.assignees && options.assignees.length > 0
                        ? options.assignees
                        : undefined,
                    reviewers: options.reviewers && options.reviewers.length > 0
                        ? options.reviewers
                        : undefined,
                    "open-pull-requests-limit": options.openPullRequestsLimit !== undefined
                        ? options.openPullRequestsLimit
                        : undefined,
                    "target-branch": options.targetBranch,
                },
            ],
        };
        new yaml_1.YamlFile(project, ".github/dependabot.yml", {
            obj: this.config,
            committed: true,
        });
        for (const i of options.allow ?? []) {
            this.addAllow(i.dependencyName);
        }
        for (const i of options.ignore ?? []) {
            this.addIgnore(i.dependencyName, ...(i.versions ?? []));
        }
        if (this.ignoresProjen) {
            this.addIgnore("projen");
        }
    }
    /**
     * Allows a dependency from automatic updates.
     *
     * @param dependencyName Use to allow updates for dependencies with matching
     * names, optionally using `*` to match zero or more characters.
     */
    addAllow(dependencyName) {
        this.allow.push({
            "dependency-name": dependencyName,
        });
    }
    /**
     * Ignores a dependency from automatic updates.
     *
     * @param dependencyName Use to ignore updates for dependencies with matching
     * names, optionally using `*` to match zero or more characters.
     * @param versions Use to ignore specific versions or ranges of versions. If
     * you want to define a range, use the standard pattern for the package
     * manager (for example: `^1.0.0` for npm, or `~> 2.0` for Bundler).
     */
    addIgnore(dependencyName, ...versions) {
        this.ignore.push({
            "dependency-name": dependencyName,
            versions: () => (versions.length > 0 ? versions : undefined),
        });
    }
}
exports.Dependabot = Dependabot;
_a = JSII_RTTI_SYMBOL_1;
Dependabot[_a] = { fqn: "projen.github.Dependabot", version: "0.91.13" };
//# sourceMappingURL=data:application/json;base64,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