import { JobStep, AppPermissions } from "./workflows-model";
/**
 * Options for `GithubCredentials.fromPersonalAccessToken`
 */
export interface GithubCredentialsPersonalAccessTokenOptions {
    readonly secret?: string;
}
/**
 * Options for `GithubCredentials.fromApp`
 */
export interface GithubCredentialsAppOptions {
    readonly appIdSecret?: string;
    readonly privateKeySecret?: string;
    /**
     * The permissions granted to the token.
     *
     * @default - all permissions granted to the app
     */
    readonly permissions?: AppPermissions;
}
/**
 * Represents a method of providing GitHub API access for projen workflows.
 */
export declare class GithubCredentials {
    private readonly options;
    /**
     * Provide API access through a GitHub personal access token.
     *
     * The token must be added as a secret to the GitHub repo, and the name of the
     * secret can be specified here.
     *
     * @see https://docs.github.com/en/authentication/keeping-your-account-and-data-secure/creating-a-personal-access-token
     * @default - a secret named "PROJEN_GITHUB_TOKEN"
     */
    static fromPersonalAccessToken(options?: GithubCredentialsPersonalAccessTokenOptions): GithubCredentials;
    /**
     * Provide API access through a GitHub App.
     *
     * The GitHub App must be installed on the GitHub repo, its App ID and a
     * private key must be added as secrets to the repo. The name of the secrets
     * can be specified here.
     *
     * @see https://docs.github.com/en/developers/apps/building-github-apps/creating-a-github-app
     * @see https://projen.io/docs/integrations/github/#github-app
     * @default - app id stored in "PROJEN_APP_ID" and private key stored in "PROJEN_APP_PRIVATE_KEY" with all permissions attached to the app
     */
    static fromApp(options?: GithubCredentialsAppOptions): GithubCredentials;
    private constructor();
    /**
     * Setup steps to obtain GitHub credentials.
     */
    get setupSteps(): JobStep[];
    /**
     * The value to use in a workflow when a GitHub token is expected. This
     * typically looks like "${{ some.path.to.a.value }}".
     */
    get tokenRef(): string;
}
