"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GithubCredentials = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const util_1 = require("../util");
/**
 * Represents a method of providing GitHub API access for projen workflows.
 */
class GithubCredentials {
    /**
     * Provide API access through a GitHub personal access token.
     *
     * The token must be added as a secret to the GitHub repo, and the name of the
     * secret can be specified here.
     *
     * @see https://docs.github.com/en/authentication/keeping-your-account-and-data-secure/creating-a-personal-access-token
     * @default - a secret named "PROJEN_GITHUB_TOKEN"
     */
    static fromPersonalAccessToken(options = {}) {
        return new GithubCredentials({
            setupSteps: [],
            tokenRef: `\${{ secrets.${options.secret ?? "PROJEN_GITHUB_TOKEN"} }}`,
        });
    }
    /**
     * Provide API access through a GitHub App.
     *
     * The GitHub App must be installed on the GitHub repo, its App ID and a
     * private key must be added as secrets to the repo. The name of the secrets
     * can be specified here.
     *
     * @see https://docs.github.com/en/developers/apps/building-github-apps/creating-a-github-app
     * @see https://projen.io/docs/integrations/github/#github-app
     * @default - app id stored in "PROJEN_APP_ID" and private key stored in "PROJEN_APP_PRIVATE_KEY" with all permissions attached to the app
     */
    static fromApp(options = {}) {
        const appIdSecret = options.appIdSecret ?? "PROJEN_APP_ID";
        const privateKeySecret = options.privateKeySecret ?? "PROJEN_APP_PRIVATE_KEY";
        return new GithubCredentials({
            setupSteps: [
                {
                    name: "Generate token",
                    id: "generate_token",
                    uses: "tibdex/github-app-token@3beb63f4bd073e61482598c45c71c1019b59b73a",
                    with: {
                        app_id: `\${{ secrets.${appIdSecret} }}`,
                        private_key: `\${{ secrets.${privateKeySecret} }}`,
                        permissions: options.permissions
                            ? JSON.stringify((0, util_1.snakeCaseKeys)(options.permissions))
                            : undefined,
                    },
                },
            ],
            tokenRef: "${{ steps.generate_token.outputs.token }}",
        });
    }
    constructor(options) {
        this.options = options;
    }
    /**
     * Setup steps to obtain GitHub credentials.
     */
    get setupSteps() {
        return [...this.options.setupSteps];
    }
    /**
     * The value to use in a workflow when a GitHub token is expected. This
     * typically looks like "${{ some.path.to.a.value }}".
     */
    get tokenRef() {
        return this.options.tokenRef;
    }
}
exports.GithubCredentials = GithubCredentials;
_a = JSII_RTTI_SYMBOL_1;
GithubCredentials[_a] = { fqn: "projen.github.GithubCredentials", version: "0.91.13" };
//# sourceMappingURL=data:application/json;base64,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