"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitHub = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const actions_provider_1 = require("./actions-provider");
const dependabot_1 = require("./dependabot");
const github_credentials_1 = require("./github-credentials");
const merge_queue_1 = require("./merge-queue");
const mergify_1 = require("./mergify");
const pr_template_1 = require("./pr-template");
const pull_request_backport_1 = require("./pull-request-backport");
const pull_request_lint_1 = require("./pull-request-lint");
const workflows_1 = require("./workflows");
const component_1 = require("../component");
class GitHub extends component_1.Component {
    /**
     * Returns the `GitHub` component of a project or `undefined` if the project
     * does not have a GitHub component.
     */
    static of(project) {
        const isGitHub = (c) => c instanceof GitHub;
        return project.components.find(isGitHub);
    }
    constructor(project, options = {}) {
        super(project);
        this.actions = new actions_provider_1.GitHubActionsProvider();
        this.workflowsEnabled = options.workflows ?? true;
        this._downloadLfs = options.downloadLfs;
        if (options.projenCredentials && options.projenTokenSecret) {
            throw new Error("projenTokenSecret is deprecated, please use projenCredentials instead");
        }
        if (options.projenTokenSecret) {
            this.projenCredentials = github_credentials_1.GithubCredentials.fromPersonalAccessToken({
                secret: options.projenTokenSecret,
            });
        }
        else if (options.projenCredentials) {
            this.projenCredentials = options.projenCredentials;
        }
        else {
            this.projenCredentials = github_credentials_1.GithubCredentials.fromPersonalAccessToken({
                secret: "PROJEN_GITHUB_TOKEN",
            });
        }
        if (options.mergify ?? true) {
            this.mergify = new mergify_1.Mergify(this, options.mergifyOptions);
        }
        if (options.mergeQueue ?? false) {
            this.mergeQueue = new merge_queue_1.MergeQueue(this, options.mergeQueueOptions);
        }
        if (options.pullRequestLint ?? true) {
            new pull_request_lint_1.PullRequestLint(this, options.pullRequestLintOptions);
        }
        if (options.pullRequestBackport ?? false) {
            if (options.mergify) {
                this.project.logger.warn("pullRequestBackport should not be used with mergify as mergify provides its own backport functionality. Please disable pullRequestBackport or mergify.");
            }
            new pull_request_backport_1.PullRequestBackport(this, options.pullRequestBackportOptions);
        }
    }
    /**
     * All workflows.
     */
    get workflows() {
        const isWorkflow = (c) => c instanceof workflows_1.GithubWorkflow;
        return this.project.components
            .filter(isWorkflow)
            .sort((w1, w2) => w1.name.localeCompare(w2.name));
    }
    /**
     * Adds a workflow to the project.
     * @param name Name of the workflow
     * @returns a GithubWorkflow instance
     */
    addWorkflow(name) {
        const workflow = new workflows_1.GithubWorkflow(this, name);
        return workflow;
    }
    addPullRequestTemplate(...content) {
        return new pr_template_1.PullRequestTemplate(this, { lines: content });
    }
    addDependabot(options) {
        return new dependabot_1.Dependabot(this, options);
    }
    /**
     * Finds a GitHub workflow by name. Returns `undefined` if the workflow cannot be found.
     * @param name The name of the GitHub workflow
     */
    tryFindWorkflow(name) {
        return this.workflows.find((w) => w.name === name);
    }
    /**
     * Whether downloading from LFS is enabled for this GitHub project
     */
    get downloadLfs() {
        return this._downloadLfs ?? this.project.gitattributes.hasLfsPatterns;
    }
}
exports.GitHub = GitHub;
_a = JSII_RTTI_SYMBOL_1;
GitHub[_a] = { fqn: "projen.github.GitHub", version: "0.91.13" };
//# sourceMappingURL=data:application/json;base64,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