"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PullRequestLint = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const _1 = require(".");
const workflows_model_1 = require("./workflows-model");
const component_1 = require("../component");
const runner_options_1 = require("../runner-options");
/**
 * Configure validations to run on GitHub pull requests.
 * Only generates a file if at least one linter is configured.
 */
class PullRequestLint extends component_1.Component {
    constructor(github, options = {}) {
        super(github.project);
        this.github = github;
        this.options = options;
        const checkSemanticTitle = options.semanticTitle ?? true;
        const checkContributorStatement = Boolean(options.contributorStatement);
        // should only create a workflow if one or more linters are enabled
        if (!checkSemanticTitle && !checkContributorStatement) {
            return;
        }
        const workflow = github.addWorkflow("pull-request-lint");
        workflow.on({
            pullRequestTarget: {
                types: [
                    "labeled",
                    "opened",
                    "synchronize",
                    "reopened",
                    "ready_for_review",
                    "edited",
                ],
            },
            // run on merge group, but use a condition later to always succeed
            // needed so the workflow can be a required status check
            mergeGroup: {},
        });
        // All checks are run against the PR and can only be evaluated within a PR context
        // Needed so jobs can be set as required and will run successfully on merge group checks.
        const prCheck = "(github.event_name == 'pull_request' || github.event_name == 'pull_request_target')";
        if (checkSemanticTitle) {
            const opts = options.semanticTitleOptions ?? {};
            const types = opts.types ?? ["feat", "fix", "chore"];
            const validateJob = {
                name: "Validate PR title",
                if: prCheck,
                ...(0, runner_options_1.filteredRunsOnOptions)(options.runsOn, options.runsOnGroup),
                permissions: {
                    pullRequests: workflows_model_1.JobPermission.WRITE,
                },
                steps: [
                    {
                        uses: "amannn/action-semantic-pull-request@v5.4.0",
                        env: {
                            GITHUB_TOKEN: "${{ secrets.GITHUB_TOKEN }}",
                        },
                        with: {
                            types: types.join("\n"),
                            requireScope: opts.requireScope ?? false,
                        },
                    },
                ],
            };
            workflow.addJobs({ validate: validateJob });
        }
        if (options.contributorStatement) {
            const opts = options.contributorStatementOptions ?? {};
            const users = opts.exemptUsers ?? [];
            const labels = opts.exemptLabels ?? [];
            const conditions = [prCheck];
            const exclusions = [
                ...labels.map((l) => `contains(github.event.pull_request.labels.*.name, '${l}')`),
                ...users.map((u) => `github.event.pull_request.user.login == '${u}'`),
            ];
            if (exclusions.length) {
                conditions.push(`!(${exclusions.join(" || ")})`);
            }
            const script = (core) => {
                const actual = process.env.PR_BODY.replace(/\r?\n/g, "\n");
                const expected = process.env.EXPECTED.replace(/\r?\n/g, "\n");
                if (!actual.includes(expected)) {
                    console.log("%j", actual);
                    console.log("%j", expected);
                    core.setFailed(`${process.env.HELP}: ${expected}`);
                }
            };
            const helpMessage = "Contributor statement missing from PR description. Please include the following text in the PR description";
            const contributorStatement = {
                name: "Require Contributor Statement",
                runsOn: options.runsOn ?? ["ubuntu-latest"],
                permissions: {
                    pullRequests: workflows_model_1.JobPermission.READ,
                },
                if: conditions.join(" && "),
                env: {
                    PR_BODY: "${{ github.event.pull_request.body }}",
                    EXPECTED: options.contributorStatement,
                    HELP: helpMessage,
                },
                steps: [
                    {
                        uses: "actions/github-script@v6",
                        with: {
                            script: fnBody(script),
                        },
                    },
                ],
            };
            workflow.addJobs({ contributorStatement });
        }
    }
    preSynthesize() {
        if (this.options.contributorStatement) {
            // Append to PR template in preSynthesize so it's always at the end of the file
            const prTemplate = _1.PullRequestTemplate.of(this.project) ??
                this.github.addPullRequestTemplate();
            prTemplate?.addLine("");
            prTemplate?.addLine("---");
            prTemplate?.addLine(this.options.contributorStatement);
            prTemplate?.addLine("");
        }
    }
}
exports.PullRequestLint = PullRequestLint;
_a = JSII_RTTI_SYMBOL_1;
PullRequestLint[_a] = { fqn: "projen.github.PullRequestLint", version: "0.91.13" };
/**
 * Helper to generate a JS script as string from a function object
 * @returns A prettified string of the function's body
 */
function fnBody(fn) {
    const def = fn.toString().replace(/\r?\n/g, "\n");
    const body = def
        .substring(def.indexOf("{") + 1, def.lastIndexOf("}"))
        .split("\n");
    const minIndentation = Math.min(...body
        .filter((l) => l.trim()) // ignore empty lines
        .map((l) => l.search(/\S|$/)));
    return body
        .map((l) => l.replace(" ".repeat(minIndentation), ""))
        .join("\n")
        .trim();
}
//# sourceMappingURL=data:application/json;base64,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