"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.WorkflowActions = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const constants_1 = require("./constants");
const util_1 = require("./private/util");
const workflow_steps_1 = require("./workflow-steps");
const REPO = (0, util_1.context)("github.repository");
const RUN_ID = (0, util_1.context)("github.run_id");
const SERVER_URL = (0, util_1.context)("github.server_url");
const RUN_URL = `${SERVER_URL}/${REPO}/actions/runs/${RUN_ID}`;
const GIT_PATCH_FILE_DEFAULT = "repo.patch";
const RUNNER_TEMP = "${{ runner.temp }}";
/**
 * A set of utility functions for creating GitHub actions in workflows.
 */
class WorkflowActions {
    /**
     * Creates a .patch file from the current git diff and uploads it as an
     * artifact. Use `checkoutWithPatch` to download and apply in another job.
     *
     * If a patch was uploaded, the action can optionally fail the job.
     *
     * @param options Options
     * @returns Job steps
     */
    static uploadGitPatch(options) {
        const MUTATIONS_FOUND = `steps.${options.stepId}.outputs.${options.outputName}`;
        const GIT_PATCH_FILE = options.patchFile ?? GIT_PATCH_FILE_DEFAULT;
        const steps = [
            {
                id: options.stepId,
                name: options.stepName ?? "Find mutations",
                run: [
                    "git add .",
                    `git diff --staged --patch --exit-code > ${GIT_PATCH_FILE} || echo "${options.outputName}=true" >> $GITHUB_OUTPUT`,
                ].join("\n"),
                // always run from root of repository
                // overrides default working directory which is set by some workflows using this function
                workingDirectory: "./",
            },
            workflow_steps_1.WorkflowSteps.uploadArtifact({
                if: MUTATIONS_FOUND,
                name: "Upload patch",
                with: {
                    name: GIT_PATCH_FILE,
                    path: GIT_PATCH_FILE,
                    includeHiddenFiles: (0, util_1.isHiddenPath)(GIT_PATCH_FILE) ? true : undefined,
                },
            }),
        ];
        if (options.mutationError) {
            steps.push({
                name: "Fail build on mutation",
                if: MUTATIONS_FOUND,
                run: [
                    `echo "::error::${options.mutationError}"`,
                    `cat ${GIT_PATCH_FILE}`,
                    "exit 1",
                ].join("\n"),
            });
        }
        return steps;
    }
    /**
     * Checks out a repository and applies a git patch that was created using
     * `uploadGitPatch`.
     *
     * @param options Options
     * @returns Job steps
     */
    static checkoutWithPatch(options = {}) {
        const { patchFile, ...restOfOptions } = options;
        const GIT_PATCH_FILE = options.patchFile ?? GIT_PATCH_FILE_DEFAULT;
        return [
            workflow_steps_1.WorkflowSteps.checkout({ with: restOfOptions }),
            {
                name: "Download patch",
                uses: "actions/download-artifact@v4",
                with: { name: GIT_PATCH_FILE, path: RUNNER_TEMP },
            },
            {
                name: "Apply patch",
                run: `[ -s ${RUNNER_TEMP}/${GIT_PATCH_FILE} ] && git apply ${RUNNER_TEMP}/${GIT_PATCH_FILE} || echo "Empty patch. Skipping."`,
            },
        ];
    }
    /**
     * A step that creates a pull request based on the current repo state.
     *
     * @param options Options
     * @returns Job steps
     */
    static createPullRequest(options) {
        const workflowName = options.workflowName;
        const branchName = options.branchName ?? `github-actions/${workflowName}`;
        const stepId = options.stepId ?? "create-pr";
        const stepName = options.stepName ?? "Create Pull Request";
        const gitIdentity = options.gitIdentity ?? constants_1.DEFAULT_GITHUB_ACTIONS_USER;
        const committer = `${gitIdentity.name} <${gitIdentity.email}>`;
        const pullRequestDescription = options.pullRequestDescription
            .trimEnd()
            .endsWith(".")
            ? options.pullRequestDescription.trimEnd()
            : `${options.pullRequestDescription.trimEnd()}.`;
        const title = options.pullRequestTitle;
        const description = [
            `${pullRequestDescription} See details in [workflow run].`,
            "",
            `[Workflow Run]: ${RUN_URL}`,
            "",
            "------",
            "",
            `*Automatically created by projen via the "${workflowName}" workflow*`,
        ].join("\n");
        return [
            {
                name: stepName,
                id: stepId,
                uses: "peter-evans/create-pull-request@v6",
                with: {
                    token: options.credentials?.tokenRef,
                    "commit-message": `${title}\n\n${description}`,
                    branch: branchName,
                    base: options.baseBranch,
                    title: title,
                    labels: options.labels?.join(",") || undefined,
                    assignees: options.assignees?.join(",") || undefined,
                    body: description,
                    author: committer,
                    committer: committer,
                    signoff: options.signoff ?? true,
                },
            },
        ];
    }
    /**
     * Configures the git identity (user name and email).
     * @param id The identity to use
     * @returns Job steps
     *
     * @deprecated use `WorkflowSteps.setupGitIdentity` instead
     */
    static setupGitIdentity(id) {
        return [workflow_steps_1.WorkflowSteps.setupGitIdentity({ gitIdentity: id })];
    }
}
exports.WorkflowActions = WorkflowActions;
_a = JSII_RTTI_SYMBOL_1;
WorkflowActions[_a] = { fqn: "projen.github.WorkflowActions", version: "0.91.13" };
//# sourceMappingURL=data:application/json;base64,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