"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.WorkflowSteps = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const object_1 = require("../util/object");
/**
 * A collection of very commonly used, individual, GitHub Workflow Job steps.
 */
class WorkflowSteps {
    /**
     * Checks out a repository.
     *
     * @param options Options to configure the `checkout` JobStep
     * @returns A JobStep that checks out a repository
     */
    static checkout(options = {}) {
        const checkoutWith = (0, object_1.removeNullOrUndefinedProperties)({
            "fetch-depth": options?.with?.fetchDepth,
            token: options?.with?.token,
            ref: options?.with?.ref,
            repository: options?.with?.repository,
            path: options?.with?.path,
            ...(options?.with?.lfs ? { lfs: true } : {}),
        });
        return {
            ...this.buildJobStepConfig({
                ...options,
                name: options.name ?? "Checkout",
            }),
            uses: "actions/checkout@v4",
            with: Object.keys(checkoutWith).length > 0 ? checkoutWith : undefined,
        };
    }
    /**
     * Configures the git identity (user name and email).
     *
     * @param options Options to configure the git identity JobStep
     * @returns Job step that configures the provided git identity
     */
    static setupGitIdentity(options) {
        return {
            ...this.buildJobStepConfig({
                ...options,
                name: options.name ?? "Set git identity",
            }),
            run: [
                `git config user.name "${options.gitIdentity.name}"`,
                `git config user.email "${options.gitIdentity.email}"`,
            ].join("\n"),
        };
    }
    /**
     * Checks if a tag exists.
     *
     * Requires a checkout step to have been run before this step with "fetch-depth" set to "0".
     *
     * Outputs:
     * - `exists`: A string value of 'true' or 'false' indicating if the tag exists.
     *
     * @param tag The tag to check. You may use valid bash code instead of a literal string in this field.
     * @param options Options to configure the `tag-exists` JobStep
     * @returns Job step that checks if the provided tag exists
     */
    static tagExists(tag, options) {
        const checkTag = (remoteTag) => `git ls-remote -q --exit-code --tags origin ${remoteTag}`;
        const varIsSet = (variable) => `[ ! -z "$${variable}" ]`;
        const setOutput = (value) => `(echo "exists=${value ? "true" : "false"}" >> $GITHUB_OUTPUT)`;
        return {
            ...this.buildJobStepConfig({
                ...options,
                name: options.name ?? "Check if tag exists",
                id: options.id ?? "check-tag",
            }),
            run: [
                `TAG=${tag}`,
                `(${varIsSet("TAG")} && ${checkTag("$TAG")} && ${setOutput(true)}) || ${setOutput(false)}`,
                "cat $GITHUB_OUTPUT",
            ].join("\n"),
        };
    }
    /**
     * Uploads an artifact.
     *
     * @param options Options to configure the `upload-artifact` JobStep
     * @returns A JobStep that uploads an artifact
     */
    static uploadArtifact(options) {
        const uploadArtifactWith = (0, object_1.removeNullOrUndefinedProperties)({
            name: options?.with?.name,
            path: options?.with?.path,
            overwrite: options?.with?.overwrite ?? true,
            "if-no-files-found": options?.with?.ifNoFilesFound,
            "retention-days": options?.with?.retentionDays,
            "compression-level": options?.with?.compressionLevel,
            "include-hidden-files": options?.with?.includeHiddenFiles,
        });
        return {
            ...this.buildJobStepConfig({
                ...options,
                name: options.name ?? "Upload artifact",
            }),
            uses: "actions/upload-artifact@v4.4.0",
            with: uploadArtifactWith,
        };
    }
    /**
     * Downloads an artifact.
     *
     * @param options Options to configure the `download-artifact` JobStep
     * @returns A JobStep that downloads an artifact
     */
    static downloadArtifact(options) {
        const downloadArtifactWith = options?.with
            ? (0, object_1.removeNullOrUndefinedProperties)({
                name: options?.with?.name,
                path: options?.with?.path,
                pattern: options?.with?.pattern,
                repository: options?.with?.repository,
                "merge-multiple": options?.with?.mergeMultiple,
                "github-token": options?.with?.token,
                "run-id": options?.with?.runId,
            })
            : undefined;
        return {
            ...this.buildJobStepConfig({
                ...options,
                name: options?.name ?? "Download artifact",
            }),
            uses: "actions/download-artifact@v4",
            with: downloadArtifactWith,
        };
    }
    /**
     * Simple adapter to ensure we only include the necessary fields for a JobStepConfiguration.
     */
    static buildJobStepConfig(options) {
        return {
            continueOnError: options?.continueOnError,
            env: options?.env,
            id: options?.id,
            if: options?.if,
            name: options?.name,
            timeoutMinutes: options?.timeoutMinutes,
            workingDirectory: options?.workingDirectory,
        };
    }
}
exports.WorkflowSteps = WorkflowSteps;
_a = JSII_RTTI_SYMBOL_1;
WorkflowSteps[_a] = { fqn: "projen.github.WorkflowSteps", version: "0.91.13" };
//# sourceMappingURL=data:application/json;base64,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