"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GithubWorkflow = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const node_path_1 = require("node:path");
const case_1 = require("case");
const _resolve_1 = require("../_resolve");
const component_1 = require("../component");
const util_1 = require("../util");
const yaml_1 = require("../yaml");
/**
 * Workflow for GitHub.
 *
 * A workflow is a configurable automated process made up of one or more jobs.
 *
 * @see https://docs.github.com/en/actions/reference/workflow-syntax-for-github-actions
 */
class GithubWorkflow extends component_1.Component {
    /**
     * @param github The GitHub component of the project this workflow belongs to.
     * @param name The name of the workflow, displayed under the repository's "Actions" tab.
     * @param options Additional options to configure the workflow.
     */
    constructor(github, name, options = {}) {
        super(github.project, `${new.target.name}#${name}`);
        this.events = {};
        this.jobs = {};
        const defaultConcurrency = {
            cancelInProgress: false,
            group: "${{ github.workflow }}",
        };
        this.name = name;
        this.concurrency = options.limitConcurrency
            ? (0, util_1.deepMerge)([
                defaultConcurrency,
                options.concurrencyOptions,
            ])
            : undefined;
        this.projenCredentials = github.projenCredentials;
        this.actions = github.actions;
        const workflowsEnabled = github.workflowsEnabled || options.force;
        if (workflowsEnabled) {
            const fileName = options.fileName ?? `${name.toLocaleLowerCase()}.yml`;
            const extension = (0, node_path_1.extname)(fileName).toLowerCase();
            if (![".yml", ".yaml"].includes(extension)) {
                throw new Error(`GitHub Workflow files must have either a .yml or .yaml file extension, got: ${fileName}`);
            }
            this.file = new yaml_1.YamlFile(this.project, `.github/workflows/${fileName}`, {
                obj: () => this.renderWorkflow(),
                // GitHub needs to read the file from the repository in order to work.
                committed: true,
            });
        }
    }
    /**
     * Add events to triggers the workflow.
     *
     * @param events The event(s) to trigger the workflow.
     */
    on(events) {
        this.events = {
            ...this.events,
            ...events,
        };
    }
    /**
     * Adds a single job to the workflow.
     * @param id The job name (unique within the workflow)
     * @param job The job specification
     */
    addJob(id, job) {
        this.addJobs({ [id]: job });
    }
    /**
     * Add jobs to the workflow.
     *
     * @param jobs Jobs to add.
     */
    addJobs(jobs) {
        verifyJobConstraints(jobs);
        this.jobs = {
            ...this.jobs,
            ...jobs,
        };
    }
    /**
     * Get a single job from the workflow.
     * @param id The job name (unique within the workflow)
     */
    getJob(id) {
        return this.jobs[id];
    }
    /**
     * Updates a single job to the workflow.
     * @param id The job name (unique within the workflow)
     */
    updateJob(id, job) {
        this.updateJobs({ [id]: job });
    }
    /**
     * Updates jobs for this worklow
     * Does a complete replace, it does not try to merge the jobs
     *
     * @param jobs Jobs to update.
     */
    updateJobs(jobs) {
        verifyJobConstraints(jobs);
        const newJobIds = Object.keys(jobs);
        const updatedJobs = Object.entries(this.jobs).map(([jobId, job]) => {
            if (newJobIds.includes(jobId)) {
                return [jobId, jobs[jobId]];
            }
            return [jobId, job];
        });
        this.jobs = {
            ...Object.fromEntries(updatedJobs),
        };
    }
    /**
     * Removes a single job to the workflow.
     * @param id The job name (unique within the workflow)
     */
    removeJob(id) {
        const updatedJobs = Object.entries(this.jobs).filter(([jobId]) => jobId !== id);
        this.jobs = {
            ...Object.fromEntries(updatedJobs),
        };
    }
    renderWorkflow() {
        return {
            name: this.name,
            "run-name": this.runName,
            on: snakeCaseKeys(this.events),
            concurrency: this.concurrency
                ? {
                    group: this.concurrency?.group,
                    "cancel-in-progress": this.concurrency.cancelInProgress,
                }
                : undefined,
            jobs: renderJobs(this.jobs, this.actions),
        };
    }
}
exports.GithubWorkflow = GithubWorkflow;
_a = JSII_RTTI_SYMBOL_1;
GithubWorkflow[_a] = { fqn: "projen.github.GithubWorkflow", version: "0.91.13" };
function snakeCaseKeys(obj) {
    if (typeof obj !== "object" || obj == null) {
        return obj;
    }
    if (Array.isArray(obj)) {
        return obj.map(snakeCaseKeys);
    }
    const result = {};
    for (let [k, v] of Object.entries(obj)) {
        if (typeof v === "object" && v != null) {
            v = snakeCaseKeys(v);
        }
        result[(0, case_1.snake)(k)] = v;
    }
    return result;
}
function renderJobs(jobs, actions) {
    const result = {};
    for (const [name, job] of Object.entries(jobs)) {
        result[name] = renderJob(job);
    }
    return result;
    /** @see https://docs.github.com/en/actions/reference/workflow-syntax-for-github-actions */
    function renderJob(job) {
        const steps = new Array();
        // https://docs.github.com/en/actions/using-workflows/reusing-workflows#supported-keywords-for-jobs-that-call-a-reusable-workflow
        if ("uses" in job) {
            return {
                name: job.name,
                needs: arrayOrScalar(job.needs),
                if: job.if,
                permissions: (0, util_1.kebabCaseKeys)(job.permissions),
                concurrency: job.concurrency,
                uses: job.uses,
                with: job.with,
                secrets: job.secrets,
                strategy: renderJobStrategy(job.strategy),
            };
        }
        if (job.tools) {
            steps.push(...setupTools(job.tools));
        }
        const userDefinedSteps = (0, util_1.kebabCaseKeys)((0, _resolve_1.resolve)(job.steps), false);
        steps.push(...userDefinedSteps);
        return {
            name: job.name,
            needs: arrayOrScalar(job.needs),
            "runs-on": arrayOrScalar(job.runsOnGroup) ?? arrayOrScalar(job.runsOn),
            permissions: (0, util_1.kebabCaseKeys)(job.permissions),
            environment: job.environment,
            concurrency: job.concurrency,
            outputs: renderJobOutputs(job.outputs),
            env: job.env,
            defaults: (0, util_1.kebabCaseKeys)(job.defaults),
            if: job.if,
            steps: steps.map(renderStep),
            "timeout-minutes": job.timeoutMinutes,
            strategy: renderJobStrategy(job.strategy),
            "continue-on-error": job.continueOnError,
            container: job.container,
            services: job.services,
        };
    }
    function renderJobOutputs(output) {
        if (output == null) {
            return undefined;
        }
        const rendered = {};
        for (const [name, { stepId, outputName }] of Object.entries(output)) {
            rendered[name] = `\${{ steps.${stepId}.outputs.${outputName} }}`;
        }
        return rendered;
    }
    function renderJobStrategy(strategy) {
        if (strategy == null) {
            return undefined;
        }
        const rendered = {
            "max-parallel": strategy.maxParallel,
            "fail-fast": strategy.failFast,
        };
        if (strategy.matrix) {
            const matrix = {
                include: strategy.matrix.include,
                exclude: strategy.matrix.exclude,
            };
            for (const [key, values] of Object.entries(strategy.matrix.domain ?? {})) {
                if (key in matrix) {
                    // A domain key was set to `include`, or `exclude`:
                    throw new Error(`Illegal job strategy matrix key: ${key}`);
                }
                matrix[key] = values;
            }
            rendered.matrix = matrix;
        }
        return rendered;
    }
    function renderStep(step) {
        return {
            name: step.name,
            id: step.id,
            if: step.if,
            uses: step.uses && actions.get(step.uses),
            env: step.env,
            run: step.run,
            with: step.with,
            "continue-on-error": step.continueOnError,
            "timeout-minutes": step.timeoutMinutes,
            "working-directory": step.workingDirectory,
        };
    }
}
function arrayOrScalar(arr) {
    if (!Array.isArray(arr)) {
        return arr;
    }
    if (arr == null || arr.length === 0) {
        return arr;
    }
    if (arr.length === 1) {
        return arr[0];
    }
    return arr;
}
function setupTools(tools) {
    const steps = [];
    if (tools.java) {
        steps.push({
            uses: "actions/setup-java@v4",
            with: { distribution: "corretto", "java-version": tools.java.version },
        });
    }
    if (tools.node) {
        steps.push({
            uses: "actions/setup-node@v4",
            with: { "node-version": tools.node.version },
        });
    }
    if (tools.python) {
        steps.push({
            uses: "actions/setup-python@v5",
            with: { "python-version": tools.python.version },
        });
    }
    if (tools.go) {
        steps.push({
            uses: "actions/setup-go@v5",
            with: { "go-version": tools.go.version },
        });
    }
    if (tools.dotnet) {
        steps.push({
            uses: "actions/setup-dotnet@v4",
            with: { "dotnet-version": tools.dotnet.version },
        });
    }
    return steps;
}
function verifyJobConstraints(jobs) {
    // verify that job has a "permissions" statement to ensure workflow can
    // operate in repos with default tokens set to readonly
    for (const [id, job] of Object.entries(jobs)) {
        if (!job.permissions) {
            throw new Error(`${id}: all workflow jobs must have a "permissions" clause to ensure workflow can operate in restricted repositories`);
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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