"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CiConfiguration = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const case_1 = require("case");
const component_1 = require("../component");
const yaml_1 = require("../yaml");
/**
 * CI for GitLab.
 * A CI is a configurable automated process made up of one or more stages/jobs.
 * @see https://docs.gitlab.com/ee/ci/yaml/
 */
class CiConfiguration extends component_1.Component {
    get defaultCache() {
        return this._defaultCache;
    }
    constructor(project, name, options) {
        super(project);
        /**
         * Defines default scripts that should run *after* all jobs. Can be overriden by the job level `afterScript`.
         */
        this.defaultAfterScript = [];
        /**
         * Defines default scripts that should run *before* all jobs. Can be overriden by the job level `afterScript`.
         */
        this.defaultBeforeScript = [];
        /**
         * A default list of additional Docker images to run scripts in. The service image is linked to the image specified in the  image parameter.
         */
        this.defaultServices = [];
        /**
         * Used to select a specific runner from the list of all runners that are available for the project.
         */
        this.defaultTags = [];
        /**
         * Can be `Include` or `Include[]`. Each `Include` will be a string, or an
         * object with properties for the method if including external YAML file. The external
         * content will be fetched, included and evaluated along the `.gitlab-ci.yml`.
         */
        this.include = [];
        /**
         * Groups jobs into stages. All jobs in one stage must complete before next stage is
         * executed. Defaults to ['build', 'test', 'deploy'].
         */
        this.stages = [];
        /**
         * Global variables that are passed to jobs.
         * If the job already has that variable defined, the job-level variable takes precedence.
         */
        this.variables = {};
        /**
         * The jobs in the CI configuration.
         */
        this.jobs = {};
        this.name = path.parse(name).name;
        const derivedPath = this.name === "gitlab-ci"
            ? ".gitlab-ci.yml"
            : `.gitlab/ci-templates/${name.toLocaleLowerCase()}.yml`;
        this.path = options?.path ?? derivedPath;
        this.file = new yaml_1.YamlFile(this.project, this.path, {
            obj: () => this.renderCI(),
            // GitLab needs to read the file from the repository in order to work.
            committed: true,
        });
        const defaults = options?.default;
        if (defaults) {
            this.defaultAfterScript.push(...(defaults.afterScript ?? []));
            this.defaultArtifacts = defaults.artifacts;
            defaults.beforeScript &&
                this.defaultBeforeScript.push(...defaults.beforeScript);
            defaults.cache && this.addDefaultCaches(defaults.cache);
            this.defaultIdTokens = defaults.idTokens;
            this.defaultImage = defaults.image;
            this.defaultInterruptible = defaults.interruptible;
            this.defaultRetry = defaults.retry;
            defaults.services && this.addServices(...defaults.services);
            defaults.tags && this.defaultTags.push(...defaults.tags);
            this.defaultTimeout = defaults.timeout;
        }
        this.pages = options?.pages;
        this.workflow = options?.workflow;
        if (options?.stages) {
            this.addStages(...options.stages);
        }
        if (options?.variables) {
            this.addGlobalVariables(options.variables);
        }
        if (options?.jobs) {
            this.addJobs(options.jobs);
        }
    }
    /**
     * Add additional yml/yaml files to the CI includes
     * @param includes The includes to add.
     */
    addIncludes(...includes) {
        for (const additional of includes) {
            this.assertIsValidInclude(additional);
            for (const existing of this.include) {
                if (this.areEqualIncludes(existing, additional)) {
                    throw new Error(`${this.name}: GitLab CI ${existing} already contains one or more templates specified in ${additional}.`);
                }
            }
            this.include.push(additional);
        }
    }
    /**
     * Throw an error if the provided Include is invalid.
     * @see https://gitlab.com/gitlab-org/gitlab/-/blob/master/lib/gitlab/ci/config/external/mapper.rb
     * @param include the Include to validate.
     */
    assertIsValidInclude(include) {
        const combos = [
            include.local,
            include.file && include.project,
            include.remote,
            include.template,
        ];
        const len = combos.filter((x) => Boolean(x)).length;
        if (len !== 1) {
            throw new Error(`${this.name}: GitLab CI include ${include} contains ${len} property combination(s).
        A valid include configuration specifies *one* of the following property combinations.
        * local
        * file, project
        * remote
        * template  
        `);
        }
    }
    /**
     * Check if the equality of Includes.
     * @see https://gitlab.com/gitlab-org/gitlab/-/blob/master/lib/gitlab/ci/config/external/mapper.rb
     * @param x First include to compare.
     * @param y Second include to compare.
     * @returns Whether the includes are equal.
     */
    areEqualIncludes(x, y) {
        if (x.local === y.local && x.local !== undefined) {
            return true;
        }
        else if (x.template === y.template && x.template !== undefined) {
            return true;
        }
        else if (x.remote === y.remote && x.remote !== undefined) {
            return true;
        }
        else if (x.project === y.project && x.ref === y.ref) {
            const xFiles = x.file ? x.file : [];
            const yFiles = y.file ? y.file : [];
            const allFiles = xFiles.concat(yFiles);
            return new Set(allFiles).size !== allFiles.length;
        }
        return false;
    }
    /**
     * Add additional services.
     * @param services The services to add.
     */
    addServices(...services) {
        for (const additional of services) {
            for (const existing of this.defaultServices) {
                if (additional.name === existing.name &&
                    additional.alias === existing.alias) {
                    throw new Error(`${this.name}: GitLab CI already contains service ${additional}.`);
                }
            }
            this.defaultServices.push(additional);
        }
    }
    /**
     * Add a globally defined variable to the CI configuration.
     * @param variables The variables to add.
     */
    addGlobalVariables(variables) {
        for (const [key, value] of Object.entries(variables)) {
            if (this.variables[key] !== undefined) {
                throw new Error(`${this.name}: GitLab CI already contains variable ${key}.`);
            }
            this.variables[key] = value;
        }
    }
    /**
     * Add stages to the CI configuration if not already present.
     * @param stages stages to add.
     */
    addStages(...stages) {
        for (const stage of stages) {
            if (!this.stages.includes(stage)) {
                this.stages.push(stage);
            }
        }
    }
    /**
     * Add jobs and their stages to the CI configuration.
     * @param jobs Jobs to add.
     */
    addJobs(jobs) {
        for (const [key, value] of Object.entries(jobs)) {
            if (this.jobs[key] !== undefined) {
                throw new Error(`${this.name}: GitLab CI already contains job ${key}.`);
            }
            this.jobs[key] = value;
            if (value.stage) {
                this.addStages(value.stage);
            }
            if (value.cache) {
                this.assertIsValidCacheSetup(value.cache);
            }
        }
    }
    isValidCacheSetup(caches) {
        const MAX_CONFIGURABLE_CACHES = 4;
        return caches.length <= MAX_CONFIGURABLE_CACHES;
    }
    assertIsValidCacheSetup(caches) {
        if (!this.isValidCacheSetup(caches)) {
            throw new Error(`${this.name}: GitLab CI can only define up to 4 caches, got: ${caches.length}`);
        }
    }
    /**
     * Adds up to 4 default caches configuration to the CI configuration.
     * @param caches Caches to add.
     */
    addDefaultCaches(caches) {
        this.assertIsValidCacheSetup(caches);
        this._defaultCache = caches;
    }
    renderCI() {
        return {
            default: this.renderDefault(),
            include: this.include.length > 0 ? snakeCaseKeys(this.include) : undefined,
            pages: snakeCaseKeys(this.pages),
            services: this.defaultServices.length > 0
                ? snakeCaseKeys(this.defaultServices)
                : undefined,
            variables: Object.entries(this.variables).length > 0 ? this.variables : undefined,
            workflow: snakeCaseKeys(this.workflow),
            stages: this.stages.length > 0 ? this.stages : undefined,
            // we do not want to change job names
            // as they can be hidden (https://docs.gitlab.com/ee/ci/jobs/index.html#hide-jobs)
            // or referenced in extends
            ...snakeCaseKeys(this.jobs, true),
        };
    }
    renderDefault() {
        const defaults = {
            afterScript: this.defaultAfterScript.length > 0
                ? this.defaultAfterScript
                : undefined,
            artifacts: this.defaultArtifacts,
            beforeScript: this.defaultBeforeScript.length > 0
                ? this.defaultBeforeScript
                : undefined,
            cache: this.defaultCache,
            idTokens: this.defaultIdTokens,
            image: this.defaultImage,
            interruptible: this.defaultInterruptible,
            retry: this.defaultRetry,
            services: this.defaultServices.length > 0 ? this.defaultServices : undefined,
            tags: this.defaultTags.length > 0 ? this.defaultTags : undefined,
            timeout: this.defaultTimeout,
        };
        return Object.values(defaults).filter((x) => x).length
            ? snakeCaseKeys(defaults)
            : undefined;
    }
}
exports.CiConfiguration = CiConfiguration;
_a = JSII_RTTI_SYMBOL_1;
CiConfiguration[_a] = { fqn: "projen.gitlab.CiConfiguration", version: "0.91.13" };
function snakeCaseKeys(obj, skipTopLevel = false) {
    if (typeof obj !== "object" || obj == null) {
        return obj;
    }
    if (Array.isArray(obj)) {
        return obj.map((o) => snakeCaseKeys(o));
    }
    const result = {};
    for (let [k, v] of Object.entries(obj)) {
        if (typeof v === "object" &&
            v != null &&
            k !== "variables" &&
            k !== "idTokens") {
            v = snakeCaseKeys(v);
        }
        result[skipTopLevel ? k : (0, case_1.snake)(k)] = v;
    }
    return result;
}
//# sourceMappingURL=data:application/json;base64,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