"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NpmAccess = exports.NodePackageManager = exports.NodePackage = exports.CodeArtifactAuthProvider = void 0;
exports.defaultNpmToken = defaultNpmToken;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const path_1 = require("path");
const semver = require("semver");
const util_1 = require("./util");
const yarnrc_1 = require("./yarnrc");
const _resolve_1 = require("../_resolve");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
const json_1 = require("../json");
const release_1 = require("../release");
const task_runtime_1 = require("../task-runtime");
const util_2 = require("../util");
const UNLICENSED = "UNLICENSED";
const DEFAULT_NPM_REGISTRY_URL = "https://registry.npmjs.org/";
const GITHUB_PACKAGES_REGISTRY = "npm.pkg.github.com";
const DEFAULT_NPM_TOKEN_SECRET = "NPM_TOKEN";
const DEFAULT_GITHUB_TOKEN_SECRET = "GITHUB_TOKEN";
/**
 * Options for authorizing requests to a AWS CodeArtifact npm repository.
 */
var CodeArtifactAuthProvider;
(function (CodeArtifactAuthProvider) {
    /**
     * Fixed credentials provided via Github secrets.
     */
    CodeArtifactAuthProvider["ACCESS_AND_SECRET_KEY_PAIR"] = "ACCESS_AND_SECRET_KEY_PAIR";
    /**
     * Ephemeral credentials provided via Github's OIDC integration with an IAM role.
     * See:
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_create_oidc.html
     * https://docs.github.com/en/actions/deployment/security-hardening-your-deployments/configuring-openid-connect-in-amazon-web-services
     */
    CodeArtifactAuthProvider["GITHUB_OIDC"] = "GITHUB_OIDC";
})(CodeArtifactAuthProvider || (exports.CodeArtifactAuthProvider = CodeArtifactAuthProvider = {}));
/**
 * Represents the npm `package.json` file.
 */
class NodePackage extends component_1.Component {
    /**
     * Returns the `NodePackage` instance associated with a project or `undefined` if
     * there is no NodePackage.
     * @param project The project
     * @returns A NodePackage, or undefined
     */
    static of(project) {
        const isIt = (o) => o instanceof NodePackage;
        return project.components.find(isIt);
    }
    constructor(project, options = {}) {
        super(project);
        this.scripts = {};
        this.scriptsToBeRemoved = new Set();
        this.keywords = new Set();
        this.bin = {};
        this.engines = {};
        this.packageName = options.packageName ?? project.name;
        this.peerDependencyOptions = {
            pinnedDevDependency: true,
            ...options.peerDependencyOptions,
        };
        this.allowLibraryDependencies = options.allowLibraryDependencies ?? true;
        this.packageManager =
            options.packageManager ?? NodePackageManager.YARN_CLASSIC;
        this.entrypoint = options.entrypoint ?? "lib/index.js";
        this.lockFile = determineLockfile(this.packageManager);
        this.project.annotateGenerated(`/${this.lockFile}`);
        const { npmAccess, npmRegistry, npmRegistryUrl, npmTokenSecret, codeArtifactOptions, scopedPackagesOptions, npmProvenance, } = this.parseNpmOptions(options);
        this.npmAccess = npmAccess;
        this.npmRegistry = npmRegistry;
        this.npmRegistryUrl = npmRegistryUrl;
        this.npmTokenSecret = npmTokenSecret;
        this.codeArtifactOptions = codeArtifactOptions;
        this.scopedPackagesOptions = scopedPackagesOptions;
        this.npmProvenance = npmProvenance;
        this.processDeps(options);
        this._prev = this.readPackageJson();
        // empty objects are here to preserve order for backwards compatibility
        this.manifest = {
            name: this.packageName,
            description: options.description,
            repository: !options.repository
                ? undefined
                : {
                    type: "git",
                    url: options.repository,
                    directory: options.repositoryDirectory,
                },
            bin: () => this.renderBin(),
            scripts: () => this.renderScripts(),
            author: this.renderAuthor(options),
            devDependencies: {},
            peerDependencies: {},
            dependencies: {},
            bundledDependencies: [],
            ...this.renderPackageResolutions(),
            keywords: () => this.renderKeywords(),
            engines: () => this.renderEngines(),
            main: this.entrypoint !== "" ? this.entrypoint : undefined,
            license: () => this.license ?? UNLICENSED,
            homepage: options.homepage,
            publishConfig: () => this.renderPublishConfig(),
            typesVersions: this._prev?.typesVersions,
            // in release CI builds we bump the version before we run "build" so we want
            // to preserve the version number. otherwise, we always set it to 0.0.0
            version: this.determineVersion(this._prev?.version),
            bugs: options.bugsEmail || options.bugsUrl
                ? {
                    email: options.bugsEmail,
                    url: options.bugsUrl,
                }
                : undefined,
        };
        // Configure Yarn Berry if using
        if (this.packageManager === NodePackageManager.YARN_BERRY ||
            this.packageManager === NodePackageManager.YARN2) {
            this.configureYarnBerry(project, options);
        }
        // add tasks for scripts from options (if specified)
        // @deprecated
        for (const [cmdname, shell] of Object.entries(options.scripts ?? {})) {
            project.addTask(cmdname, { exec: shell });
        }
        this.file = new json_1.JsonFile(this, "package.json", {
            obj: this.manifest,
            readonly: false, // we want "yarn add" to work and we have anti-tamper
            newline: true, // all package managers prefer a newline, see https://github.com/projen/projen/issues/2076
            committed: true, // needs to be committed so users can install the dependencies
        });
        this.addKeywords(...(options.keywords ?? []));
        this.addBin(options.bin ?? {});
        // automatically add all executable files under "bin"
        if (options.autoDetectBin ?? true) {
            this.autoDiscoverBinaries();
        }
        // node version
        this.minNodeVersion = options.minNodeVersion;
        this.maxNodeVersion = options.maxNodeVersion;
        this.pnpmVersion = options.pnpmVersion ?? "9";
        this.bunVersion = options.bunVersion ?? "latest";
        this.addNodeEngine();
        this.addCodeArtifactLoginScript();
        // license
        if (options.licensed ?? true) {
            this.license = options.license ?? "Apache-2.0";
        }
        this.installTask = project.addTask("install", {
            description: "Install project dependencies and update lockfile (non-frozen)",
            exec: this.installAndUpdateLockfileCommand,
        });
        this.installCiTask = project.addTask("install:ci", {
            description: "Install project dependencies using frozen lockfile",
            exec: this.installCommand,
        });
    }
    /**
     * Defines normal dependencies.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addDeps(...deps) {
        for (const dep of deps) {
            this.project.deps.addDependency(dep, dependencies_1.DependencyType.RUNTIME);
        }
    }
    /**
     * Defines development/test dependencies.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addDevDeps(...deps) {
        for (const dep of deps) {
            this.project.deps.addDependency(dep, dependencies_1.DependencyType.BUILD);
        }
    }
    /**
     * Defines peer dependencies.
     *
     * When adding peer dependencies, a devDependency will also be added on the
     * pinned version of the declared peer. This will ensure that you are testing
     * your code against the minimum version required from your consumers.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addPeerDeps(...deps) {
        if (Object.keys(deps).length && !this.allowLibraryDependencies) {
            throw new Error(`cannot add peer dependencies to an APP project: ${Object.keys(deps).join(",")}`);
        }
        for (const dep of deps) {
            this.project.deps.addDependency(dep, dependencies_1.DependencyType.PEER);
        }
    }
    /**
     * Defines bundled dependencies.
     *
     * Bundled dependencies will be added as normal dependencies as well as to the
     * `bundledDependencies` section of your `package.json`.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addBundledDeps(...deps) {
        if (deps.length && !this.allowLibraryDependencies) {
            throw new Error(`cannot add bundled dependencies to an APP project: ${deps.join(",")}`);
        }
        for (const dep of deps) {
            this.project.deps.addDependency(dep, dependencies_1.DependencyType.BUNDLED);
        }
    }
    /**
     * Adds an `engines` requirement to your package.
     * @param engine The engine (e.g. `node`)
     * @param version The semantic version requirement (e.g. `^10`)
     */
    addEngine(engine, version) {
        this.engines[engine] = version;
    }
    /**
     * Adds keywords to package.json (deduplicated)
     * @param keywords The keywords to add
     */
    addKeywords(...keywords) {
        for (const k of keywords) {
            this.keywords.add(k);
        }
    }
    addBin(bins) {
        for (const [k, v] of Object.entries(bins)) {
            this.bin[k] = v;
        }
    }
    /**
     * Add a npm package.json script.
     *
     * @param name The script name
     * @param command The command to execute
     */
    setScript(name, command) {
        this.scripts[name] = command;
    }
    /**
     * Removes an npm script (always successful).
     *
     * @param name The name of the script.
     */
    removeScript(name) {
        // need to keep track in case there's a task of the same name
        this.scriptsToBeRemoved.add(name);
        delete this.scripts[name];
    }
    /**
     * Indicates if a script by the given name is defined.
     * @param name The name of the script
     * @deprecated Use `project.tasks.tryFind(name)`
     */
    hasScript(name) {
        return this.project.tasks.tryFind(name) !== undefined;
    }
    /**
     * Directly set fields in `package.json`.
     * @escape
     * @param name field name
     * @param value field value
     */
    addField(name, value) {
        this.manifest[name] = value;
    }
    /**
     * Sets the package version.
     * @param version Package version.
     */
    addVersion(version) {
        this.manifest.version = version;
    }
    /**
     * Defines resolutions for dependencies to change the normally resolved
     * version of a dependency to something else.
     *
     * @param resolutions Names resolutions to be added. Specify a version or
     * range with this syntax:
     * `module@^7`
     */
    addPackageResolutions(...resolutions) {
        for (const resolution of resolutions) {
            this.project.deps.addDependency(resolution, dependencies_1.DependencyType.OVERRIDE);
        }
    }
    /**
     * Returns the command to execute in order to install all dependencies (always frozen).
     */
    get installCommand() {
        return this.renderInstallCommand(true);
    }
    /**
     * Renders `yarn install` or `npm install` with lockfile update (not frozen)
     */
    get installAndUpdateLockfileCommand() {
        return this.renderInstallCommand(false);
    }
    /**
     * Attempt to resolve the currently installed version for a given dependency.
     *
     * @remarks
     * This method will first look through the current project's dependencies.
     * If found and semantically valid (not '*'), that will be used.
     * Otherwise, it will fall back to locating a `package.json` manifest for the dependency
     * through node's internal resolution reading the version from there.
     *
     * @param dependencyName Dependency to resolve for.
     */
    tryResolveDependencyVersion(dependencyName) {
        try {
            const fromDeps = this.project.deps.tryGetDependency(dependencyName);
            const version = semver.coerce(fromDeps?.version, { loose: true });
            if (version) {
                return version.format();
            }
        }
        catch { }
        return (0, util_1.tryResolveDependencyVersion)(dependencyName, {
            paths: [this.project.outdir],
        });
    }
    // ---------------------------------------------------------------------------------------
    synthesize() {
        this._renderedDeps = this.renderDependencies();
        super.synthesize();
    }
    postSynthesize() {
        super.postSynthesize();
        // only run "install" if package.json has changed or if we don't have a
        // `node_modules` directory.
        if (this.file.changed ||
            !(0, fs_1.existsSync)((0, path_1.join)(this.project.outdir, "node_modules"))) {
            this.installDependencies();
        }
        // resolve "*" deps in package.json and update it. if it was changed,
        // install deps again so that lockfile is updated.
        if (this.resolveDepsAndWritePackageJson()) {
            this.installDependencies();
        }
    }
    /**
     * The command which executes "projen".
     * @deprecated use `project.projenCommand` instead.
     */
    get projenCommand() {
        return this.project.projenCommand;
    }
    /**
     * Returns `true` if we are running within a CI build.
     */
    get isAutomatedBuild() {
        return (0, util_2.isTruthy)(process.env.CI);
    }
    determineVersion(currVersion) {
        if (!this.isReleaseBuild) {
            return "0.0.0";
        }
        return currVersion ?? "0.0.0";
    }
    /**
     * Returns `true` if this is a CI release build.
     */
    get isReleaseBuild() {
        return (0, util_2.isTruthy)(process.env.RELEASE);
    }
    // -------------------------------------------------------------------------------------------
    parseNpmOptions(options) {
        let npmRegistryUrl = options.npmRegistryUrl;
        if (options.npmRegistry) {
            if (npmRegistryUrl) {
                throw new Error('cannot use the deprecated "npmRegistry" together with "npmRegistryUrl". please use "npmRegistryUrl" instead.');
            }
            npmRegistryUrl = `https://${options.npmRegistry}`;
        }
        const npmr = new URL(npmRegistryUrl ?? DEFAULT_NPM_REGISTRY_URL);
        if (!npmr || !npmr.hostname || !npmr.href) {
            throw new Error(`unable to determine npm registry host from url ${npmRegistryUrl}. Is this really a URL?`);
        }
        const npmAccess = options.npmAccess ?? defaultNpmAccess(this.packageName);
        if (!isScoped(this.packageName) && npmAccess === NpmAccess.RESTRICTED) {
            throw new Error(`"npmAccess" cannot be RESTRICTED for non-scoped npm package "${this.packageName}"`);
        }
        const npmProvenance = options.npmProvenance ?? npmAccess === NpmAccess.PUBLIC;
        if (npmProvenance && npmAccess !== NpmAccess.PUBLIC) {
            throw new Error(`"npmProvenance" can only be enabled for public packages`);
        }
        const isAwsCodeArtifact = (0, release_1.isAwsCodeArtifactRegistry)(npmRegistryUrl);
        const hasScopedPackage = options.scopedPackagesOptions &&
            options.scopedPackagesOptions.length !== 0;
        if (isAwsCodeArtifact) {
            if (options.npmTokenSecret) {
                throw new Error('"npmTokenSecret" must not be specified when publishing AWS CodeArtifact.');
            }
            else if (options.codeArtifactOptions?.authProvider ===
                CodeArtifactAuthProvider.GITHUB_OIDC) {
                if (options.codeArtifactOptions.accessKeyIdSecret ||
                    options.codeArtifactOptions.secretAccessKeySecret) {
                    throw new Error("access and secret key pair should not be provided when using GITHUB_OIDC auth provider for AWS CodeArtifact");
                }
                else if (!options.codeArtifactOptions.roleToAssume) {
                    throw new Error('"roleToAssume" property is required when using GITHUB_OIDC for AWS CodeArtifact options');
                }
            }
        }
        else {
            if ((options.codeArtifactOptions?.accessKeyIdSecret ||
                options.codeArtifactOptions?.secretAccessKeySecret ||
                options.codeArtifactOptions?.roleToAssume) &&
                !hasScopedPackage) {
                throw new Error("codeArtifactOptions must only be specified when publishing AWS CodeArtifact or used in scoped packages.");
            }
        }
        // apply defaults for AWS CodeArtifact
        let codeArtifactOptions;
        if (isAwsCodeArtifact || hasScopedPackage) {
            const authProvider = options.codeArtifactOptions?.authProvider ??
                CodeArtifactAuthProvider.ACCESS_AND_SECRET_KEY_PAIR;
            const isAccessSecretKeyPairAuth = authProvider === CodeArtifactAuthProvider.ACCESS_AND_SECRET_KEY_PAIR;
            codeArtifactOptions = {
                authProvider,
                accessKeyIdSecret: options.codeArtifactOptions?.accessKeyIdSecret ??
                    (isAccessSecretKeyPairAuth ? "AWS_ACCESS_KEY_ID" : undefined),
                secretAccessKeySecret: options.codeArtifactOptions?.secretAccessKeySecret ??
                    (isAccessSecretKeyPairAuth ? "AWS_SECRET_ACCESS_KEY" : undefined),
                roleToAssume: options.codeArtifactOptions?.roleToAssume,
            };
        }
        return {
            npmAccess,
            npmRegistry: npmr.hostname + this.renderNpmRegistryPath(npmr.pathname),
            npmRegistryUrl: npmr.href,
            npmTokenSecret: defaultNpmToken(options.npmTokenSecret, npmr.hostname),
            codeArtifactOptions,
            scopedPackagesOptions: this.parseScopedPackagesOptions(options.scopedPackagesOptions),
            npmProvenance,
        };
    }
    parseScopedPackagesOptions(scopedPackagesOptions) {
        if (!scopedPackagesOptions) {
            return undefined;
        }
        return scopedPackagesOptions.map((option) => {
            if (!isScoped(option.scope)) {
                throw new Error(`Scope must start with "@" in options, found ${option.scope}`);
            }
            if (!(0, release_1.isAwsCodeArtifactRegistry)(option.registryUrl)) {
                throw new Error(`Only AWS Code artifact scoped registry is supported for now, found ${option.registryUrl}`);
            }
            const result = {
                registryUrl: option.registryUrl,
                scope: option.scope,
            };
            return result;
        });
    }
    addCodeArtifactLoginScript() {
        if (!this.scopedPackagesOptions ||
            this.scopedPackagesOptions.length === 0) {
            return;
        }
        this.project.addTask("ca:login", {
            requiredEnv: ["AWS_ACCESS_KEY_ID", "AWS_SECRET_ACCESS_KEY"],
            steps: [
                { exec: "which aws" }, // check that AWS CLI is installed
                ...this.scopedPackagesOptions.map((scopedPackagesOption) => {
                    const { registryUrl, scope } = scopedPackagesOption;
                    const { domain, region, accountId, registry } = (0, util_1.extractCodeArtifactDetails)(registryUrl);
                    // reference: https://docs.aws.amazon.com/codeartifact/latest/ug/npm-auth.html
                    const commands = [
                        `npm config set ${scope}:registry ${registryUrl}`,
                        `CODEARTIFACT_AUTH_TOKEN=$(aws codeartifact get-authorization-token --domain ${domain} --region ${region} --domain-owner ${accountId} --query authorizationToken --output text)`,
                        `npm config set //${registry}:_authToken=$CODEARTIFACT_AUTH_TOKEN`,
                    ];
                    if (!this.minNodeVersion || semver.major(this.minNodeVersion) <= 16)
                        commands.push(`npm config set //${registry}:always-auth=true`);
                    return {
                        exec: commands.join("; "),
                    };
                }),
            ],
        });
    }
    addNodeEngine() {
        if (!this.minNodeVersion && !this.maxNodeVersion) {
            return;
        }
        let nodeVersion = "";
        if (this.minNodeVersion) {
            nodeVersion += `>= ${this.minNodeVersion}`;
        }
        if (this.maxNodeVersion) {
            nodeVersion += ` <= ${this.maxNodeVersion}`;
        }
        this.addEngine("node", nodeVersion);
    }
    renderNpmRegistryPath(path) {
        if (!path || path == "/") {
            return "";
        }
        else {
            return path;
        }
    }
    renderInstallCommand(frozen) {
        switch (this.packageManager) {
            case NodePackageManager.YARN:
            case NodePackageManager.YARN_CLASSIC:
                return [
                    "yarn install",
                    "--check-files", // ensure all modules exist (especially projen which was just removed).
                    ...(frozen ? ["--frozen-lockfile"] : []),
                ].join(" ");
            case NodePackageManager.YARN2:
            case NodePackageManager.YARN_BERRY:
                return ["yarn install", ...(frozen ? ["--immutable"] : [])].join(" ");
            case NodePackageManager.NPM:
                return frozen ? "npm ci" : "npm install";
            case NodePackageManager.PNPM:
                return frozen
                    ? "pnpm i --frozen-lockfile"
                    : "pnpm i --no-frozen-lockfile";
            case NodePackageManager.BUN:
                return ["bun install", ...(frozen ? ["--frozen-lockfile"] : [])].join(" ");
            default:
                throw new Error(`unexpected package manager ${this.packageManager}`);
        }
    }
    processDeps(options) {
        this.addDeps(...(options.deps ?? []));
        this.addDevDeps(...(options.devDeps ?? []));
        this.addPeerDeps(...(options.peerDeps ?? []));
        this.addBundledDeps(...(options.bundledDeps ?? []));
    }
    renderDependencies() {
        const devDependencies = {};
        const peerDependencies = {};
        const dependencies = {};
        const bundledDependencies = new Array();
        // synthetic dependencies: add a pinned build dependency to ensure we are
        // testing against the minimum requirement of the peer.
        if (this.peerDependencyOptions.pinnedDevDependency) {
            for (const dep of this.project.deps.all.filter((d) => d.type === dependencies_1.DependencyType.PEER)) {
                let req = dep.name;
                // Skip if we already have a runtime dependency on this peer and no build dependency yet.
                // If there is a build dep already, we need to override its version.
                if (this.project.deps.tryGetDependency(dep.name, dependencies_1.DependencyType.RUNTIME) &&
                    !this.project.deps.tryGetDependency(dep.name, dependencies_1.DependencyType.BUILD)) {
                    continue;
                }
                if (dep.version) {
                    const ver = (0, util_1.minVersion)(dep.version);
                    if (!ver) {
                        throw new Error(`unable to determine minimum semver for peer dependency ${dep.name}@${dep.version}`);
                    }
                    req += "@" + ver;
                }
                this.addDevDeps(req);
            }
        }
        for (const dep of this.project.deps.all) {
            let version = dep.version ?? "*";
            let name = dep.name;
            if (name.startsWith("file:")) {
                const localDependencyPath = name.substring(5);
                const depPackageJson = (0, path_1.resolve)(this.project.outdir, localDependencyPath, "package.json");
                const pkgFile = (0, fs_1.readFileSync)(depPackageJson, "utf8");
                const pkg = JSON.parse(pkgFile);
                version = localDependencyPath;
                name = pkg.name;
            }
            switch (dep.type) {
                case dependencies_1.DependencyType.BUNDLED:
                    bundledDependencies.push(name);
                    const depDecls = this.project.deps.all.filter((d) => d.name === name);
                    if (depDecls.some((d) => d.type === dependencies_1.DependencyType.PEER)) {
                        throw new Error(`unable to bundle "${name}": it cannot appear as a peer dependency (bundled would always take precedence over peer)`);
                    }
                    // I've observed that at least npm 10.8.2 will silently fail to bundle
                    // a dependency if it is [also] part of `devDependencies`. It must exist in
                    // `dependencies` and `dependencies` only.
                    if (depDecls.some((d) => d.type === dependencies_1.DependencyType.BUILD)) {
                        throw new Error(`unable to bundle "${name}": it cannot appear as a devDependency (only prod dependencies are bundled, and any dependency appearing as a devDependency is considered to be not a prod dependency)`);
                    }
                    // also add as a runtime dependency
                    dependencies[name] = version;
                    break;
                case dependencies_1.DependencyType.PEER:
                    peerDependencies[name] = version;
                    break;
                case dependencies_1.DependencyType.RUNTIME:
                    dependencies[name] = version;
                    break;
                case dependencies_1.DependencyType.TEST:
                case dependencies_1.DependencyType.DEVENV:
                case dependencies_1.DependencyType.BUILD:
                    devDependencies[name] = version;
                    break;
            }
        }
        // returns a lazy value to normalize during synthesis
        const normalize = (obj) => () => (0, util_2.sorted)(obj);
        // update the manifest we are about to save into `package.json`
        this.manifest.devDependencies = normalize(devDependencies);
        this.manifest.peerDependencies = normalize(peerDependencies);
        this.manifest.dependencies = normalize(dependencies);
        this.manifest.bundledDependencies = (0, util_2.sorted)(bundledDependencies);
        // nothing further to do if package.json file does not exist
        if (!this._prev) {
            return { devDependencies, peerDependencies, dependencies };
        }
        const readDeps = (user, current = {}) => {
            for (const [name, userVersion] of Object.entries(user)) {
                const currentVersion = current[name];
                // respect user version if it's not '*' or if current version is undefined
                if (userVersion !== "*" || !currentVersion || currentVersion === "*") {
                    continue;
                }
                // memoize current version in memory so it is preserved when saving
                user[name] = currentVersion;
            }
            // report removals
            for (const name of Object.keys(current ?? {})) {
                if (!user[name]) {
                    this.project.logger.verbose(`${name}: removed`);
                }
            }
        };
        readDeps(devDependencies, this._prev.devDependencies);
        readDeps(dependencies, this._prev.dependencies);
        readDeps(peerDependencies, this._prev.peerDependencies);
        return { devDependencies, dependencies, peerDependencies };
    }
    /**
     * Resolves any deps that do not have a specified version (e.g. `*`) and
     * update `package.json` if needed.
     *
     * @returns `true` if package.json was updated or `false` if not.
     */
    resolveDepsAndWritePackageJson() {
        const outdir = this.project.outdir;
        const rootPackageJson = (0, path_1.join)(outdir, "package.json");
        const original = (0, fs_1.readFileSync)(rootPackageJson, "utf8");
        const pkg = JSON.parse(original);
        const resolveDeps = (current, user) => {
            const result = {};
            current = current ?? {};
            user = user ?? {};
            for (const [name, currentDefinition] of Object.entries(user)) {
                // find actual version from node_modules
                let desiredVersion = currentDefinition;
                if (currentDefinition === "*") {
                    // we already know we don't have the version in project `deps`,
                    // so skip straight to checking manifest.
                    const resolvedVersion = (0, util_1.tryResolveDependencyVersion)(name, {
                        paths: [this.project.outdir],
                    });
                    if (!resolvedVersion) {
                        this.project.logger.warn(`unable to resolve version for ${name} from installed modules`);
                        continue;
                    }
                    desiredVersion = `^${resolvedVersion}`;
                }
                if (currentDefinition !== desiredVersion) {
                    this.project.logger.verbose(`${name}: ${currentDefinition} => ${desiredVersion}`);
                }
                result[name] = desiredVersion;
            }
            // print removed packages
            for (const name of Object.keys(current)) {
                if (!result[name]) {
                    this.project.logger.verbose(`${name} removed`);
                }
            }
            return result;
        };
        const rendered = this._renderedDeps;
        if (!rendered) {
            throw new Error("assertion failed");
        }
        const deps = resolveDeps(pkg.dependencies, rendered.dependencies);
        const devDeps = resolveDeps(pkg.devDependencies, rendered.devDependencies);
        const peerDeps = resolveDeps(pkg.peerDependencies, rendered.peerDependencies);
        if (this.peerDependencyOptions.pinnedDevDependency) {
            for (const [name, version] of Object.entries(peerDeps)) {
                // Skip if we already have a runtime dependency on this peer
                // or if devDependency version is already set.
                // Relies on the "*" devDependency added in the presynth step
                if (deps[name] || rendered.devDependencies[name] !== "*") {
                    continue;
                }
                // Take version and pin as dev dependency
                const ver = (0, util_1.minVersion)(version);
                if (!ver) {
                    throw new Error(`unable to determine minimum semver for peer dependency ${name}@${version}`);
                }
                devDeps[name] = ver;
            }
        }
        pkg.dependencies = (0, util_2.sorted)(deps);
        pkg.devDependencies = (0, util_2.sorted)(devDeps);
        pkg.peerDependencies = (0, util_2.sorted)(peerDeps);
        const updated = JSON.stringify(pkg, undefined, 2) + "\n";
        if (original === updated) {
            return false;
        }
        (0, util_2.writeFile)(rootPackageJson, updated);
        return true;
    }
    renderPackageResolutions() {
        const render = () => {
            const overridingDependencies = this.project.deps.all.filter((dep) => dep.type === dependencies_1.DependencyType.OVERRIDE);
            if (!overridingDependencies.length) {
                return undefined;
            }
            return Object.fromEntries(overridingDependencies.map(({ name, version = "*" }) => [name, version]));
        };
        switch (this.packageManager) {
            case NodePackageManager.NPM:
                return { overrides: render };
            case NodePackageManager.PNPM:
                return this.project.parent
                    ? undefined
                    : { pnpm: { overrides: render } };
            case NodePackageManager.YARN:
            case NodePackageManager.YARN2:
            case NodePackageManager.YARN_CLASSIC:
            case NodePackageManager.YARN_BERRY:
            case NodePackageManager.BUN:
            default:
                return { resolutions: render };
        }
    }
    renderPublishConfig() {
        // When npm provenance is enabled, we need to always render the public access
        // But when npmAccess is the set to the default, we prefer to omit it
        const shouldOmitAccess = !this.npmProvenance &&
            this.npmAccess === defaultNpmAccess(this.packageName);
        // omit values if they are the same as the npm defaults
        return (0, _resolve_1.resolve)({
            registry: this.npmRegistryUrl !== DEFAULT_NPM_REGISTRY_URL
                ? this.npmRegistryUrl
                : undefined,
            access: shouldOmitAccess ? undefined : this.npmAccess,
        }, { omitEmpty: true });
    }
    renderKeywords() {
        const kwds = Array.from(this.keywords);
        return (0, util_2.sorted)(kwds.sort());
    }
    renderEngines() {
        return (0, util_2.sorted)(this.engines);
    }
    autoDiscoverBinaries() {
        const binrel = "bin";
        const bindir = (0, path_1.join)(this.project.outdir, binrel);
        if ((0, fs_1.existsSync)(bindir)) {
            for (const file of (0, fs_1.readdirSync)(bindir)) {
                try {
                    (0, fs_1.accessSync)((0, path_1.join)(bindir, file), fs_1.constants.X_OK);
                    const binPath = (0, path_1.join)(binrel, file);
                    const normalizedPath = (0, util_2.normalizePersistedPath)(binPath);
                    this.bin[file] = normalizedPath;
                }
                catch (e) {
                    // not executable, skip
                }
            }
        }
    }
    renderAuthor(options) {
        let author;
        if (options.authorName) {
            author = {
                name: options.authorName,
                email: options.authorEmail,
                url: options.authorUrl,
                organization: options.authorOrganization ?? false,
            };
        }
        else {
            if (options.authorEmail ||
                options.authorUrl ||
                options.authorOrganization !== undefined) {
                throw new Error('"authorName" is required if specifying "authorEmail" or "authorUrl"');
            }
        }
        return author;
    }
    renderBin() {
        return (0, util_2.sorted)(this.bin);
    }
    renderScripts() {
        const result = {};
        const tasks = this.project.tasks.all
            .filter((t) => 
        // Must remove to prevent overriding built-in npm command (which would loop)
        t.name !== this.installTask.name && t.name !== this.installCiTask.name)
            .sort((x, y) => x.name.localeCompare(y.name));
        for (const task of tasks) {
            if (this.scriptsToBeRemoved.has(task.name)) {
                continue;
            }
            result[task.name] = this.npmScriptForTask(task);
        }
        return {
            ...result,
            ...this.scripts,
        };
    }
    npmScriptForTask(task) {
        return `${this.projenCommand} ${task.name}`;
    }
    readPackageJson() {
        const file = (0, path_1.join)(this.project.outdir, "package.json");
        if (!(0, fs_1.existsSync)(file)) {
            return undefined;
        }
        return JSON.parse((0, fs_1.readFileSync)(file, "utf-8"));
    }
    installDependencies() {
        this.project.logger.info("Installing dependencies...");
        const runtime = new task_runtime_1.TaskRuntime(this.project.outdir);
        const taskToRun = this.isAutomatedBuild
            ? this.installCiTask
            : this.installTask;
        runtime.runTask(taskToRun.name);
    }
    configureYarnBerry(project, options) {
        const { version = "4.0.1", yarnRcOptions = {}, zeroInstalls = false, } = options.yarnBerryOptions || {};
        this.checkForConflictingYarnOptions(yarnRcOptions);
        // Set the `packageManager` field in `package.json` to the version specified. This tells `corepack` which version
        // of `yarn` to use.
        this.addField("packageManager", `yarn@${version}`);
        this.configureYarnBerryGitignore(zeroInstalls);
        new yarnrc_1.Yarnrc(project, version, yarnRcOptions);
    }
    checkForConflictingYarnOptions(yarnRcOptions) {
        if (this.npmAccess &&
            yarnRcOptions.npmPublishAccess &&
            this.npmAccess.toString() !== yarnRcOptions.npmPublishAccess.toString()) {
            throw new Error(`Cannot set npmAccess (${this.npmAccess}) and yarnRcOptions.npmPublishAccess (${yarnRcOptions.npmPublishAccess}) to different values.`);
        }
        if (this.npmRegistryUrl &&
            yarnRcOptions.npmRegistryServer &&
            this.npmRegistryUrl !== yarnRcOptions.npmRegistryServer) {
            throw new Error(`Cannot set npmRegistryUrl (${this.npmRegistryUrl}) and yarnRcOptions.npmRegistryServer (${yarnRcOptions.npmRegistryServer}) to different values.`);
        }
    }
    /** See https://yarnpkg.com/getting-started/qa#which-files-should-be-gitignored */
    configureYarnBerryGitignore(zeroInstalls) {
        const { gitignore } = this.project;
        // These patterns are the same whether or not you're using zero-installs
        gitignore.exclude(".yarn/*");
        gitignore.include(".yarn/patches", ".yarn/plugins", ".yarn/releases", ".yarn/sdks", ".yarn/versions");
        if (zeroInstalls) {
            gitignore.include("!.yarn/cache");
        }
        else {
            gitignore.exclude(".pnp.*");
        }
    }
}
exports.NodePackage = NodePackage;
_a = JSII_RTTI_SYMBOL_1;
NodePackage[_a] = { fqn: "projen.javascript.NodePackage", version: "0.91.13" };
/**
 * The node package manager to use.
 */
var NodePackageManager;
(function (NodePackageManager) {
    /**
     * Use `yarn` as the package manager.
     *
     * @deprecated For `yarn` 1.x use `YARN_CLASSIC` for `yarn` >= 2 use `YARN_BERRY`. Currently, `NodePackageManager.YARN` means `YARN_CLASSIC`. In the future, we might repurpose it to mean `YARN_BERRY`.
     */
    NodePackageManager["YARN"] = "yarn";
    /**
     * Use `yarn` versions >= 2 as the package manager.
     *
     * @deprecated use YARN_BERRY instead
     */
    NodePackageManager["YARN2"] = "yarn2";
    /**
     * Use `yarn` 1.x as the package manager.
     */
    NodePackageManager["YARN_CLASSIC"] = "yarn_classic";
    /**
     * Use `yarn` versions >= 2 as the package manager.
     */
    NodePackageManager["YARN_BERRY"] = "yarn_berry";
    /**
     * Use `npm` as the package manager.
     */
    NodePackageManager["NPM"] = "npm";
    /**
     * Use `pnpm` as the package manager.
     */
    NodePackageManager["PNPM"] = "pnpm";
    /**
     * Use `bun` as the package manager
     */
    NodePackageManager["BUN"] = "bun";
})(NodePackageManager || (exports.NodePackageManager = NodePackageManager = {}));
/**
 * Npm package access level
 */
var NpmAccess;
(function (NpmAccess) {
    /**
     * Package is public.
     */
    NpmAccess["PUBLIC"] = "public";
    /**
     * Package can only be accessed with credentials.
     */
    NpmAccess["RESTRICTED"] = "restricted";
})(NpmAccess || (exports.NpmAccess = NpmAccess = {}));
/**
 * Determines if an npm package is "scoped" (i.e. it starts with "xxx@").
 */
function isScoped(packageName) {
    return packageName.includes("@");
}
function defaultNpmAccess(packageName) {
    return isScoped(packageName) ? NpmAccess.RESTRICTED : NpmAccess.PUBLIC;
}
function defaultNpmToken(npmToken, registry) {
    // if we are publishing to AWS CdodeArtifact, no NPM_TOKEN used (will be requested using AWS CLI later).
    if ((0, release_1.isAwsCodeArtifactRegistry)(registry)) {
        return undefined;
    }
    // if we are publishing to GitHub Packages, default to GITHUB_TOKEN.
    const isGitHubPackages = registry === GITHUB_PACKAGES_REGISTRY;
    return (npmToken ??
        (isGitHubPackages ? DEFAULT_GITHUB_TOKEN_SECRET : DEFAULT_NPM_TOKEN_SECRET));
}
function determineLockfile(packageManager) {
    if (packageManager === NodePackageManager.YARN ||
        packageManager === NodePackageManager.YARN2 ||
        packageManager === NodePackageManager.YARN_CLASSIC ||
        packageManager === NodePackageManager.YARN_BERRY) {
        return "yarn.lock";
    }
    else if (packageManager === NodePackageManager.NPM) {
        return "package-lock.json";
    }
    else if (packageManager === NodePackageManager.PNPM) {
        return "pnpm-lock.yaml";
    }
    else if (packageManager === NodePackageManager.BUN) {
        return "bun.lockb";
    }
    throw new Error(`unsupported package manager ${packageManager}`);
}
//# sourceMappingURL=data:application/json;base64,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