"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NodeProject = exports.AutoRelease = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path_1 = require("path");
const bundler_1 = require("./bundler");
const jest_1 = require("./jest");
const license_checker_1 = require("./license-checker");
const node_package_1 = require("./node-package");
const projenrc_1 = require("./projenrc");
const build_1 = require("../build");
const consts_1 = require("../build/private/consts");
const common_1 = require("../common");
const dependencies_1 = require("../dependencies");
const github_1 = require("../github");
const constants_1 = require("../github/constants");
const util_1 = require("../github/private/util");
const workflows_model_1 = require("../github/workflows-model");
const ignore_file_1 = require("../ignore-file");
const javascript_1 = require("../javascript");
const license_1 = require("../license");
const projenrc_json_1 = require("../projenrc-json");
const release_1 = require("../release");
const runner_options_1 = require("../runner-options");
const util_2 = require("../util");
const path_2 = require("../util/path");
const PROJEN_SCRIPT = "projen";
/**
 * Automatic bump modes
 */
var AutoRelease;
(function (AutoRelease) {
    /**
     * Automatically bump & release a new version for every commit to "main"
     */
    AutoRelease[AutoRelease["EVERY_COMMIT"] = 0] = "EVERY_COMMIT";
    /**
     * Automatically bump & release a new version on a daily basis.
     */
    AutoRelease[AutoRelease["DAILY"] = 1] = "DAILY";
})(AutoRelease || (exports.AutoRelease = AutoRelease = {}));
/**
 * Node.js project.
 *
 * @pjid node
 */
class NodeProject extends github_1.GitHubProject {
    /**
     * The .npmrc file
     */
    get npmrc() {
        if (!this._npmrc) {
            this._npmrc = new javascript_1.NpmConfig(this, { omitEmpty: true });
        }
        return this._npmrc;
    }
    /**
     * @deprecated use `package.allowLibraryDependencies`
     */
    get allowLibraryDependencies() {
        return this.package.allowLibraryDependencies;
    }
    /**
     * @deprecated use `package.entrypoint`
     */
    get entrypoint() {
        return this.package.entrypoint;
    }
    /**
     * The minimum node version required by this package to function.
     *
     * This value indicates the package is incompatible with older versions.
     */
    get minNodeVersion() {
        return this.package.minNodeVersion;
    }
    /**
     * Maximum node version supported by this package.
     *
     * The value indicates the package is incompatible with newer versions.
     */
    get maxNodeVersion() {
        return this.package.maxNodeVersion;
    }
    /**
     * The package manager to use.
     *
     * @deprecated use `package.packageManager`
     */
    get packageManager() {
        return this.package.packageManager;
    }
    /**
     * @deprecated use `package.addField(x, y)`
     */
    get manifest() {
        return this.package.manifest;
    }
    constructor(options) {
        super({
            ...options,
            // Node projects have the specific projen version locked via lockfile, so we can skip the @<VERSION> part of the top-level project
            projenCommand: options.projenCommand ?? "npx projen",
        });
        this.package = new node_package_1.NodePackage(this, options);
        this.workflowBootstrapSteps = options.workflowBootstrapSteps ?? [];
        this.workflowGitIdentity =
            options.workflowGitIdentity ?? constants_1.DEFAULT_GITHUB_ACTIONS_USER;
        this.workflowPackageCache = options.workflowPackageCache ?? false;
        this.artifactsDirectory =
            options.artifactsDirectory ?? consts_1.DEFAULT_ARTIFACTS_DIRECTORY;
        (0, util_1.ensureNotHiddenPath)(this.artifactsDirectory, "artifactsDirectory");
        const normalizedArtifactsDirectory = (0, util_2.normalizePersistedPath)(this.artifactsDirectory);
        this.artifactsJavascriptDirectory = path_1.posix.join(normalizedArtifactsDirectory, "js");
        this.runScriptCommand = (() => {
            switch (this.packageManager) {
                case node_package_1.NodePackageManager.NPM:
                    return "npm run";
                case node_package_1.NodePackageManager.YARN:
                case node_package_1.NodePackageManager.YARN2:
                case node_package_1.NodePackageManager.YARN_CLASSIC:
                case node_package_1.NodePackageManager.YARN_BERRY:
                    return "yarn run";
                case node_package_1.NodePackageManager.PNPM:
                    return "pnpm run";
                case node_package_1.NodePackageManager.BUN:
                    return "bun run";
                default:
                    throw new Error(`unexpected package manager ${this.packageManager}`);
            }
        })();
        const envCommand = (() => {
            switch (this.packageManager) {
                case node_package_1.NodePackageManager.PNPM:
                    return '$(pnpm -c exec "node --print process.env.PATH")';
                case node_package_1.NodePackageManager.BUN:
                    return '$(bun --eval "console.log(process.env.PATH)")';
                default:
                    return '$(npx -c "node --print process.env.PATH")';
            }
        })();
        this.nodeVersion =
            options.workflowNodeVersion ?? this.package.minNodeVersion;
        // add PATH for all tasks which includes the project's npm .bin list
        this.tasks.addEnvironment("PATH", envCommand);
        this.addLicense(options);
        if (options.npmignoreEnabled ?? true) {
            this.npmignore = new ignore_file_1.IgnoreFile(this, ".npmignore", options.npmIgnoreOptions);
        }
        this.addDefaultGitIgnore();
        if (options.gitignore?.length) {
            for (const i of options.gitignore) {
                this.gitignore.exclude(i);
            }
        }
        if (options.npmignore?.length) {
            if (!this.npmignore) {
                throw new Error('.npmignore is not defined for an APP project type. Add "npmIgnore: true" to override this');
            }
            for (const i of options.npmignore) {
                this.npmignore.exclude(i);
            }
        }
        if (!this.ejected) {
            this.setScript(PROJEN_SCRIPT, this.package.projenCommand);
        }
        this.npmignore?.exclude(`/${common_1.PROJEN_DIR}/`);
        const projen = options.projenDevDependency ?? (this.parent ? false : true);
        if (projen && !this.ejected) {
            const postfix = options.projenVersion ? `@${options.projenVersion}` : "";
            this.addDevDeps(`projen${postfix}`);
            if (!this.deps.isDependencySatisfied("constructs", dependencies_1.DependencyType.BUILD, "^10.0.0")) {
                this.addDevDeps(`constructs@^10.0.0`);
            }
        }
        if (!options.defaultReleaseBranch) {
            throw new Error('"defaultReleaseBranch" is temporarily a required option while we migrate its default value from "master" to "main"');
        }
        const buildEnabled = options.buildWorkflow ?? (this.parent ? false : true);
        // configure jest if enabled
        // must be before the build/release workflows
        if (options.jest ?? true) {
            this.jest = new jest_1.Jest(this, options.jestOptions);
        }
        const requiresIdTokenPermission = (options.scopedPackagesOptions ?? []).length > 0 &&
            options.codeArtifactOptions?.authProvider ===
                release_1.CodeArtifactAuthProvider.GITHUB_OIDC;
        const workflowPermissions = {
            idToken: requiresIdTokenPermission ? workflows_model_1.JobPermission.WRITE : undefined,
        };
        const buildWorkflowOptions = options.buildWorkflowOptions ?? {};
        if (buildEnabled && (this.github || github_1.GitHub.of(this.root))) {
            this.buildWorkflow = new build_1.BuildWorkflow(this, {
                buildTask: this.buildTask,
                artifactsDirectory: this.artifactsDirectory,
                containerImage: options.workflowContainerImage,
                gitIdentity: this.workflowGitIdentity,
                mutableBuild: options.mutableBuild,
                workflowTriggers: options.buildWorkflowTriggers,
                permissions: workflowPermissions,
                ...buildWorkflowOptions,
                preBuildSteps: this.renderWorkflowSetup({
                    installStepConfiguration: {
                        workingDirectory: this.determineInstallWorkingDirectory(),
                    },
                    mutable: buildWorkflowOptions.mutableBuild ?? options.mutableBuild ?? true,
                }).concat(buildWorkflowOptions.preBuildSteps ?? []),
                postBuildSteps: [...(options.postBuildSteps ?? [])],
                ...(0, runner_options_1.filteredRunsOnOptions)(options.workflowRunsOn, options.workflowRunsOnGroup),
            });
            this.buildWorkflow.addPostBuildSteps(...this.renderUploadCoverageJobStep(options));
        }
        const release = options.release ??
            options.releaseWorkflow ??
            (this.parent ? false : true);
        if (release) {
            this.release = new release_1.Release(this, {
                versionFile: "package.json", // this is where "version" is set after bump
                task: this.buildTask,
                branch: options.defaultReleaseBranch ?? "main",
                ...options,
                artifactsDirectory: this.artifactsDirectory,
                releaseWorkflowSetupSteps: [
                    ...this.renderWorkflowSetup({
                        installStepConfiguration: {
                            workingDirectory: this.determineInstallWorkingDirectory(),
                        },
                        mutable: false,
                    }),
                    ...(options.releaseWorkflowSetupSteps ?? []),
                ],
                postBuildSteps: [
                    ...(options.postBuildSteps ?? []),
                    ...this.renderUploadCoverageJobStep(options),
                ],
                workflowNodeVersion: this.nodeVersion,
                workflowPermissions,
            });
            this.publisher = this.release.publisher;
            const nodePackageToReleaseCodeArtifactAuthProviderMapping = {
                [node_package_1.CodeArtifactAuthProvider.ACCESS_AND_SECRET_KEY_PAIR]: release_1.CodeArtifactAuthProvider.ACCESS_AND_SECRET_KEY_PAIR,
                [node_package_1.CodeArtifactAuthProvider.GITHUB_OIDC]: release_1.CodeArtifactAuthProvider.GITHUB_OIDC,
            };
            if (options.releaseToNpm ?? false) {
                const codeArtifactOptions = (0, release_1.isAwsCodeArtifactRegistry)(this.package.npmRegistry)
                    ? {
                        accessKeyIdSecret: options.codeArtifactOptions?.accessKeyIdSecret,
                        secretAccessKeySecret: options.codeArtifactOptions?.secretAccessKeySecret,
                        roleToAssume: options.codeArtifactOptions?.roleToAssume,
                        authProvider: options.codeArtifactOptions?.authProvider
                            ? nodePackageToReleaseCodeArtifactAuthProviderMapping[options.codeArtifactOptions.authProvider]
                            : release_1.CodeArtifactAuthProvider.ACCESS_AND_SECRET_KEY_PAIR,
                    }
                    : {};
                this.release.publisher.publishToNpm({
                    registry: this.package.npmRegistry,
                    npmTokenSecret: this.package.npmTokenSecret,
                    npmProvenance: this.package.npmProvenance,
                    codeArtifactOptions,
                });
            }
        }
        else {
            // validate that no release options are selected if the release workflow is disabled.
            if (options.releaseToNpm) {
                throw new Error('"releaseToNpm" is not supported if "release" is not set');
            }
            if (options.releaseEveryCommit) {
                throw new Error('"releaseEveryCommit" is not supported if "release" is not set');
            }
            if (options.releaseSchedule) {
                throw new Error('"releaseSchedule" is not supported if "release" is not set');
            }
        }
        if ((options.autoMerge ?? true) &&
            this.github?.mergify &&
            this.buildWorkflow?.buildJobIds) {
            this.autoMerge = new github_1.AutoMerge(this.github, options.autoMergeOptions);
            this.autoMerge.addConditionsLater({
                render: () => this.buildWorkflow?.buildJobIds.map((id) => `status-success=${id}`) ??
                    [],
            });
        }
        const dependabot = options.dependabot ?? false;
        const depsUpgrade = options.depsUpgrade ?? !dependabot;
        if (dependabot && depsUpgrade) {
            throw new Error("'dependabot' cannot be configured together with 'depsUpgrade'");
        }
        const depsAutoApprove = options.autoApproveUpgrades ?? false;
        if (depsAutoApprove && !this.autoApprove && this.github) {
            throw new Error("Automatic approval of dependencies upgrades requires configuring `autoApproveOptions`");
        }
        const autoApproveLabel = (condition) => condition && this.autoApprove?.label
            ? [this.autoApprove.label]
            : undefined;
        if (dependabot) {
            const defaultOptions = {
                labels: autoApproveLabel(depsAutoApprove),
            };
            this.github?.addDependabot((0, util_2.deepMerge)([defaultOptions, options.dependabotOptions ?? {}]));
        }
        if (depsUpgrade) {
            const defaultOptions = {
                workflowOptions: {
                    container: options.workflowContainerImage
                        ? {
                            image: options.workflowContainerImage,
                        }
                        : undefined,
                    labels: autoApproveLabel(depsAutoApprove),
                    gitIdentity: this.workflowGitIdentity,
                    permissions: workflowPermissions,
                },
            };
            this.upgradeWorkflow = new javascript_1.UpgradeDependencies(this, (0, util_2.deepMerge)([defaultOptions, options.depsUpgradeOptions ?? {}]));
        }
        if (options.pullRequestTemplate ?? true) {
            this.github?.addPullRequestTemplate(...(options.pullRequestTemplateContents ?? []));
        }
        const projenrcJs = options.projenrcJs ?? !options.projenrcJson;
        if (!this.parent && projenrcJs) {
            const projenrcJsFile = new projenrc_1.Projenrc(this, options.projenrcJsOptions);
            this.npmignore?.exclude(`/${projenrcJsFile.filePath}`);
        }
        else if (options.projenrcJson) {
            const projenrcJsonFile = projenrc_json_1.ProjenrcJson.of(this);
            if (projenrcJsonFile) {
                this.npmignore?.exclude(`/${projenrcJsonFile.filePath}`);
            }
        }
        // add a bundler component - this enables things like Lambda bundling and in the future web bundling.
        this.bundler = new bundler_1.Bundler(this, options.bundlerOptions);
        const shouldPackage = options.package ?? true;
        if (release && !shouldPackage) {
            this.logger.warn("When `release` is enabled, `package` must also be enabled as it is required by release. Force enabling `package`.");
        }
        if (release || shouldPackage) {
            this.packageTask.exec(`mkdir -p ${this.artifactsJavascriptDirectory}`);
            const pkgMgr = this.package.packageManager === node_package_1.NodePackageManager.PNPM
                ? "pnpm"
                : "npm";
            this.packageTask.exec(`${pkgMgr} pack --pack-destination ${this.artifactsJavascriptDirectory}`);
        }
        if (options.prettier ?? false) {
            this.prettier = new javascript_1.Prettier(this, { ...options.prettierOptions });
        }
        // For PNPM, the default resolution mode is "lowest", which leads to any non-versioned (ie '*') dependencies being
        // resolved to the lowest available version, which is unlikely to be expected behaviour for users. We set resolution
        // mode to "highest" to match the behaviour of yarn and npm.
        if (this.package.packageManager === node_package_1.NodePackageManager.PNPM) {
            this.npmrc.addConfig("resolution-mode", "highest");
        }
        if (options.checkLicenses) {
            new license_checker_1.LicenseChecker(this, options.checkLicenses);
        }
    }
    determineInstallWorkingDirectory() {
        if (this.parent) {
            return (0, path_2.ensureRelativePathStartsWithDot)((0, path_1.relative)(".", this.root.outdir));
        }
        return;
    }
    renderUploadCoverageJobStep(options) {
        // run codecov if enabled or a secret token name is passed in
        // AND jest must be configured
        if ((options.codeCov || options.codeCovTokenSecret) && this.jest?.config) {
            return [
                {
                    name: "Upload coverage to Codecov",
                    uses: "codecov/codecov-action@v4",
                    with: options.codeCovTokenSecret
                        ? {
                            token: `\${{ secrets.${options.codeCovTokenSecret} }}`,
                            directory: this.jest.config.coverageDirectory,
                        }
                        : {
                            directory: this.jest.config.coverageDirectory,
                        },
                },
            ];
        }
        else {
            return [];
        }
    }
    addBins(bins) {
        this.package.addBin(bins);
    }
    /**
     * Replaces the contents of an npm package.json script.
     *
     * @param name The script name
     * @param command The command to execute
     */
    setScript(name, command) {
        this.package.setScript(name, command);
    }
    /**
     * Replaces the contents of multiple npm package.json scripts.
     * @param scripts The scripts to set
     */
    addScripts(scripts) {
        for (const [name, command] of Object.entries(scripts)) {
            this.package.setScript(name, command);
        }
    }
    /**
     * Removes the npm script (always successful).
     * @param name The name of the script.
     */
    removeScript(name) {
        this.package.removeScript(name);
    }
    /**
     * Indicates if a script by the name name is defined.
     * @param name The name of the script
     * @deprecated Use `project.tasks.tryFind(name)`
     */
    hasScript(name) {
        return this.package.hasScript(name);
    }
    /**
     * DEPRECATED
     * @deprecated use `project.compileTask.exec()`
     */
    addCompileCommand(...commands) {
        for (const c of commands) {
            this.compileTask.exec(c);
        }
    }
    /**
     * DEPRECATED
     * @deprecated use `project.testTask.exec()`
     */
    addTestCommand(...commands) {
        for (const c of commands) {
            this.testTask.exec(c);
        }
    }
    /**
     * Directly set fields in `package.json`.
     * @param fields The fields to set
     */
    addFields(fields) {
        for (const [name, value] of Object.entries(fields)) {
            this.package.addField(name, value);
        }
    }
    /**
     * Adds keywords to package.json (deduplicated)
     * @param keywords The keywords to add
     */
    addKeywords(...keywords) {
        this.package.addKeywords(...keywords);
    }
    /**
     * Get steps for scoped package access
     *
     * @param codeArtifactOptions Details of logging in to AWS
     * @returns array of job steps required for each private scoped packages
     */
    getScopedPackageSteps(codeArtifactOptions) {
        const parsedCodeArtifactOptions = {
            accessKeyIdSecret: codeArtifactOptions?.accessKeyIdSecret ?? "AWS_ACCESS_KEY_ID",
            secretAccessKeySecret: codeArtifactOptions?.secretAccessKeySecret ?? "AWS_SECRET_ACCESS_KEY",
            roleToAssume: codeArtifactOptions?.roleToAssume,
            authProvider: codeArtifactOptions?.authProvider,
        };
        if (parsedCodeArtifactOptions.authProvider ===
            node_package_1.CodeArtifactAuthProvider.GITHUB_OIDC) {
            return [
                {
                    name: "Configure AWS Credentials",
                    uses: "aws-actions/configure-aws-credentials@v4",
                    with: {
                        "aws-region": "us-east-2",
                        "role-to-assume": parsedCodeArtifactOptions.roleToAssume,
                        "role-duration-seconds": 900,
                    },
                },
                {
                    name: "AWS CodeArtifact Login",
                    run: `${this.runScriptCommand} ca:login`,
                },
            ];
        }
        if (parsedCodeArtifactOptions.roleToAssume) {
            return [
                {
                    name: "Configure AWS Credentials",
                    uses: "aws-actions/configure-aws-credentials@v4",
                    with: {
                        "aws-access-key-id": (0, util_1.secretToString)(parsedCodeArtifactOptions.accessKeyIdSecret),
                        "aws-secret-access-key": (0, util_1.secretToString)(parsedCodeArtifactOptions.secretAccessKeySecret),
                        "aws-region": "us-east-2",
                        "role-to-assume": parsedCodeArtifactOptions.roleToAssume,
                        "role-duration-seconds": 900,
                    },
                },
                {
                    name: "AWS CodeArtifact Login",
                    run: `${this.runScriptCommand} ca:login`,
                },
            ];
        }
        return [
            {
                name: "AWS CodeArtifact Login",
                run: `${this.runScriptCommand} ca:login`,
                env: {
                    AWS_ACCESS_KEY_ID: (0, util_1.secretToString)(parsedCodeArtifactOptions.accessKeyIdSecret),
                    AWS_SECRET_ACCESS_KEY: (0, util_1.secretToString)(parsedCodeArtifactOptions.secretAccessKeySecret),
                },
            },
        ];
    }
    /**
     * Returns the set of workflow steps which should be executed to bootstrap a
     * workflow.
     *
     * @param options Options.
     * @returns Job steps
     */
    renderWorkflowSetup(options = {}) {
        const install = new Array();
        // first run the workflow bootstrap steps
        install.push(...this.workflowBootstrapSteps);
        if (this.package.packageManager === node_package_1.NodePackageManager.PNPM) {
            install.push({
                name: "Setup pnpm",
                uses: "pnpm/action-setup@v3",
                with: { version: this.package.pnpmVersion },
            });
        }
        else if (this.package.packageManager === node_package_1.NodePackageManager.BUN) {
            install.push({
                name: "Setup bun",
                uses: "oven-sh/setup-bun@v2",
                with: { "bun-version": this.package.bunVersion },
            });
        }
        if (this.package.packageManager !== node_package_1.NodePackageManager.BUN) {
            if (this.nodeVersion || this.workflowPackageCache) {
                const cache = this.package.packageManager === node_package_1.NodePackageManager.YARN
                    ? "yarn"
                    : this.package.packageManager === node_package_1.NodePackageManager.YARN2
                        ? "yarn"
                        : this.package.packageManager === node_package_1.NodePackageManager.YARN_CLASSIC
                            ? "yarn"
                            : this.package.packageManager === node_package_1.NodePackageManager.YARN_BERRY
                                ? "yarn"
                                : this.packageManager === node_package_1.NodePackageManager.BUN
                                    ? "bun"
                                    : this.package.packageManager === node_package_1.NodePackageManager.PNPM
                                        ? "pnpm"
                                        : "npm";
                install.push({
                    name: "Setup Node.js",
                    uses: "actions/setup-node@v4",
                    with: {
                        ...(this.nodeVersion && {
                            "node-version": this.nodeVersion,
                        }),
                        ...(this.workflowPackageCache && {
                            cache,
                        }),
                    },
                });
            }
        }
        const mutable = options.mutable ?? false;
        if (this.package.scopedPackagesOptions) {
            install.push(...this.getScopedPackageSteps(this.package.codeArtifactOptions));
        }
        install.push({
            name: "Install dependencies",
            run: mutable
                ? this.package.installAndUpdateLockfileCommand
                : this.package.installCommand,
            ...(options.installStepConfiguration ?? {}),
        });
        return install;
    }
    /**
     * Defines normal dependencies.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addDeps(...deps) {
        return this.package.addDeps(...deps);
    }
    /**
     * Defines development/test dependencies.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addDevDeps(...deps) {
        return this.package.addDevDeps(...deps);
    }
    /**
     * Defines peer dependencies.
     *
     * When adding peer dependencies, a devDependency will also be added on the
     * pinned version of the declared peer. This will ensure that you are testing
     * your code against the minimum version required from your consumers.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addPeerDeps(...deps) {
        return this.package.addPeerDeps(...deps);
    }
    /**
     * Defines bundled dependencies.
     *
     * Bundled dependencies will be added as normal dependencies as well as to the
     * `bundledDependencies` section of your `package.json`.
     *
     * @param deps Names modules to install. By default, the the dependency will
     * be installed in the next `npx projen` run and the version will be recorded
     * in your `package.json` file. You can upgrade manually or using `yarn
     * add/upgrade`. If you wish to specify a version range use this syntax:
     * `module@^7`.
     */
    addBundledDeps(...deps) {
        return this.package.addBundledDeps(...deps);
    }
    /**
     * Adds patterns to be ignored by npm.
     *
     * @param pattern The pattern to ignore.
     *
     * @remarks
     * If you are having trouble getting an ignore to populate, try using your construct or component's preSynthesize method to properly delay calling this method.
     */
    addPackageIgnore(pattern) {
        this.npmignore?.addPatterns(pattern);
    }
    addLicense(options) {
        if (this.package.license) {
            new license_1.License(this, {
                spdx: this.package.license,
                copyrightOwner: options.copyrightOwner ?? options.authorName,
                copyrightPeriod: options.copyrightPeriod,
            });
        }
    }
    addDefaultGitIgnore() {
        this.gitignore.exclude("# Logs", "logs", "*.log", "npm-debug.log*", "yarn-debug.log*", "yarn-error.log*", "lerna-debug.log*", "# Diagnostic reports (https://nodejs.org/api/report.html)", "report.[0-9]*.[0-9]*.[0-9]*.[0-9]*.json", "# Runtime data", "pids", "*.pid", "*.seed", "*.pid.lock", "# Directory for instrumented libs generated by jscoverage/JSCover", "lib-cov", "# Coverage directory used by tools like istanbul", "coverage", "*.lcov", "# nyc test coverage", ".nyc_output", "# Compiled binary addons (https://nodejs.org/api/addons.html)", "build/Release", "# Dependency directories", "node_modules/", "jspm_packages/", "# TypeScript cache", "*.tsbuildinfo", "# Optional eslint cache", ".eslintcache", "# Output of 'npm pack'", "*.tgz", "# Yarn Integrity file", ".yarn-integrity", "# parcel-bundler cache (https://parceljs.org/)", ".cache");
    }
    /**
     * Returns the shell command to execute in order to run a task. This will
     * typically be `npx projen TASK`.
     *
     * @param task The task for which the command is required
     */
    runTaskCommand(task) {
        return `${this.package.projenCommand} ${task.name}`;
    }
    /**
     * The job ID of the build workflow.
     */
    get buildWorkflowJobId() {
        return this.buildWorkflow?.buildJobIds[0];
    }
}
exports.NodeProject = NodeProject;
_a = JSII_RTTI_SYMBOL_1;
NodeProject[_a] = { fqn: "projen.javascript.NodeProject", version: "0.91.13" };
//# sourceMappingURL=data:application/json;base64,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