"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.UpgradeDependenciesSchedule = exports.UpgradeDependencies = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const component_1 = require("../component");
const dependencies_1 = require("../dependencies");
const github_1 = require("../github");
const constants_1 = require("../github/constants");
const workflow_actions_1 = require("../github/workflow-actions");
const workflows_model_1 = require("../github/workflows-model");
const javascript_1 = require("../javascript");
const release_1 = require("../release");
const runner_options_1 = require("../runner-options");
const CREATE_PATCH_STEP_ID = "create_patch";
const PATCH_CREATED_OUTPUT = "patch_created";
/**
 * Upgrade node project dependencies.
 */
class UpgradeDependencies extends component_1.Component {
    constructor(project, options = {}) {
        super(project);
        /**
         * The workflows that execute the upgrades. One workflow per branch.
         */
        this.workflows = [];
        this._project = project;
        this.options = options;
        this.depTypes = this.options.types ?? [
            dependencies_1.DependencyType.BUILD,
            dependencies_1.DependencyType.BUNDLED,
            dependencies_1.DependencyType.DEVENV,
            dependencies_1.DependencyType.PEER,
            dependencies_1.DependencyType.RUNTIME,
            dependencies_1.DependencyType.TEST,
            dependencies_1.DependencyType.OPTIONAL,
        ];
        this.upgradeTarget = this.options.target ?? "minor";
        this.satisfyPeerDependencies = this.options.satisfyPeerDependencies ?? true;
        this.includeDeprecatedVersions =
            this.options.includeDeprecatedVersions ?? false;
        this.pullRequestTitle = options.pullRequestTitle ?? "upgrade dependencies";
        this.gitIdentity =
            options.workflowOptions?.gitIdentity ?? constants_1.DEFAULT_GITHUB_ACTIONS_USER;
        this.permissions = {
            contents: workflows_model_1.JobPermission.READ,
            ...options.workflowOptions?.permissions,
        };
        this.postBuildSteps = [];
        this.containerOptions = options.workflowOptions?.container;
        this.postUpgradeTask =
            project.tasks.tryFind("post-upgrade") ??
                project.tasks.addTask("post-upgrade", {
                    description: "Runs after upgrading dependencies",
                });
        this.upgradeTask = project.addTask(options.taskName ?? "upgrade", {
            // this task should not run in CI mode because its designed to
            // update package.json and lock files.
            env: { CI: "0" },
            description: this.pullRequestTitle,
            steps: { toJSON: () => this.renderTaskSteps() },
        });
        this.upgradeTask.lock(); // this task is a lazy value, so make it readonly
        if (this.upgradeTask && project.github && (options.workflow ?? true)) {
            if (options.workflowOptions?.branches) {
                for (const branch of options.workflowOptions.branches) {
                    this.workflows.push(this.createWorkflow(this.upgradeTask, project.github, branch));
                }
            }
            else if (release_1.Release.of(project)) {
                const release = release_1.Release.of(project);
                release._forEachBranch((branch) => {
                    this.workflows.push(this.createWorkflow(this.upgradeTask, project.github, branch));
                });
            }
            else {
                // represents the default repository branch.
                // just like not specifying anything.
                const defaultBranch = undefined;
                this.workflows.push(this.createWorkflow(this.upgradeTask, project.github, defaultBranch));
            }
        }
    }
    /**
     * Add steps to execute a successful build.
     * @param steps workflow steps
     */
    addPostBuildSteps(...steps) {
        this.postBuildSteps.push(...steps);
    }
    renderTaskSteps() {
        function executeCommand(packageManager) {
            switch (packageManager) {
                case javascript_1.NodePackageManager.NPM:
                case javascript_1.NodePackageManager.YARN:
                case javascript_1.NodePackageManager.YARN_CLASSIC:
                    return "npx";
                case javascript_1.NodePackageManager.PNPM:
                    return "pnpm dlx";
                case javascript_1.NodePackageManager.YARN2:
                case javascript_1.NodePackageManager.YARN_BERRY:
                    return "yarn dlx";
                case javascript_1.NodePackageManager.BUN:
                    return "bunx";
            }
        }
        const steps = new Array();
        // Package Manager upgrade should always include all deps
        const includeForPackageManagerUpgrade = this.buildDependencyList(true);
        if (includeForPackageManagerUpgrade.length === 0) {
            return [{ exec: "echo No dependencies to upgrade." }];
        }
        // Removing `npm-check-updates` from our dependency tree because it depends on a package
        // that uses an npm-specific feature that causes an invalid dependency tree when using Yarn 1.
        // See https://github.com/projen/projen/pull/3136 for more details.
        const includeForNcu = this.buildDependencyList(false);
        const ncuCommand = [
            `${executeCommand(this._project.package.packageManager)} npm-check-updates@16`,
            "--upgrade",
            `--target=${this.upgradeTarget}`,
            `--${this.satisfyPeerDependencies ? "peer" : "no-peer"}`,
            `--${this.includeDeprecatedVersions ? "deprecated" : "no-deprecated"}`,
            `--dep=${this.renderNcuDependencyTypes(this.depTypes)}`,
            `--filter=${includeForNcu.join(",")}`,
        ];
        // bump versions in package.json
        if (includeForNcu.length) {
            steps.push({ exec: ncuCommand.join(" ") });
        }
        // run "yarn/npm install" to update the lockfile and install any deps (such as projen)
        steps.push({ exec: this._project.package.installAndUpdateLockfileCommand });
        // run upgrade command to upgrade transitive deps as well
        steps.push({
            exec: this.renderUpgradePackagesCommand(includeForPackageManagerUpgrade),
        });
        // run "projen" to give projen a chance to update dependencies (it will also run "yarn install")
        steps.push({ exec: this._project.projenCommand });
        steps.push({ spawn: this.postUpgradeTask.name });
        return steps;
    }
    /**
     * Render projen dependencies types to a list of ncu compatible types
     */
    renderNcuDependencyTypes(types) {
        return Array.from(new Set(types
            .map((type) => {
            switch (type) {
                case dependencies_1.DependencyType.PEER:
                    return "peer";
                case dependencies_1.DependencyType.RUNTIME:
                    return "prod";
                case dependencies_1.DependencyType.OPTIONAL:
                    return "optional";
                case dependencies_1.DependencyType.TEST:
                case dependencies_1.DependencyType.DEVENV:
                case dependencies_1.DependencyType.BUILD:
                    return "dev";
                case dependencies_1.DependencyType.BUNDLED:
                default:
                    return false;
            }
        })
            .filter((type) => Boolean(type)))).join(",");
    }
    /**
     * Render a package manager specific command to upgrade all requested dependencies.
     */
    renderUpgradePackagesCommand(include) {
        function upgradePackages(command) {
            return () => {
                return `${command} ${include.join(" ")}`;
            };
        }
        const packageManager = this._project.package.packageManager;
        let lazy = undefined;
        switch (packageManager) {
            case javascript_1.NodePackageManager.YARN:
            case javascript_1.NodePackageManager.YARN_CLASSIC:
                lazy = upgradePackages("yarn upgrade");
                break;
            case javascript_1.NodePackageManager.YARN2:
            case javascript_1.NodePackageManager.YARN_BERRY:
                lazy = upgradePackages("yarn up");
                break;
            case javascript_1.NodePackageManager.NPM:
                lazy = upgradePackages("npm update");
                break;
            case javascript_1.NodePackageManager.PNPM:
                lazy = upgradePackages("pnpm update");
                break;
            case javascript_1.NodePackageManager.BUN:
                lazy = upgradePackages("bun update");
                break;
            default:
                throw new Error(`unexpected package manager ${packageManager}`);
        }
        // return a lazy function so that dependencies include ones that were
        // added post project instantiation (i.e using project.addDeps)
        return lazy;
    }
    buildDependencyList(includeDependenciesWithConstraint) {
        return Array.from(new Set(this.options.include ??
            this.filterDependencies(includeDependenciesWithConstraint)));
    }
    filterDependencies(includeConstraint) {
        const dependencies = [];
        const deps = this.project.deps.all
            // remove those that have a constraint version (unless includeConstraint is true)
            .filter((d) => includeConstraint || this.packageCanBeUpgradedInPackageJson(d.version))
            // remove override dependencies
            .filter((d) => d.type !== dependencies_1.DependencyType.OVERRIDE);
        for (const type of this.depTypes) {
            dependencies.push(...deps
                .filter((d) => d.type === type)
                .filter((d) => !(this.options.exclude ?? []).includes(d.name)));
        }
        return dependencies.map((d) => d.name);
    }
    /**
     * Projen can alter a package's version in package.json when either the version is omitted, or set to "*".
     * Otherwise, the exact version selected is placed in the package.json file and upgrading is handled through the package manager
     * rather than npm-check-updates.
     *
     * @param version semver from DependencyCoordinates.version, may be undefined
     * @returns whether the version is the default versioning behavior
     */
    packageCanBeUpgradedInPackageJson(version) {
        // No version means "latest"
        return !version || version === "*";
    }
    createWorkflow(task, github, branch) {
        const schedule = this.options.workflowOptions?.schedule ??
            UpgradeDependenciesSchedule.DAILY;
        const workflowName = `${task.name}${branch ? `-${branch.replace(/\//g, "-")}` : ""}`;
        const workflow = github.addWorkflow(workflowName);
        const triggers = {
            workflowDispatch: {},
            schedule: schedule.cron.length > 0
                ? schedule.cron.map((e) => ({ cron: e }))
                : undefined,
        };
        workflow.on(triggers);
        const upgrade = this.createUpgrade(task, github, branch);
        const pr = this.createPr(workflow, upgrade);
        const jobs = {};
        jobs[upgrade.jobId] = upgrade.job;
        jobs[pr.jobId] = pr.job;
        workflow.addJobs(jobs);
        return workflow;
    }
    createUpgrade(task, github, branch) {
        const with_ = {
            ...(branch ? { ref: branch } : {}),
            ...(github.downloadLfs ? { lfs: true } : {}),
        };
        const steps = [
            github_1.WorkflowSteps.checkout({ with: with_ }),
            ...this._project.renderWorkflowSetup({ mutable: false }),
            {
                name: "Upgrade dependencies",
                run: this._project.runTaskCommand(task),
            },
        ];
        steps.push(...this.postBuildSteps);
        steps.push(...workflow_actions_1.WorkflowActions.uploadGitPatch({
            stepId: CREATE_PATCH_STEP_ID,
            outputName: PATCH_CREATED_OUTPUT,
        }));
        return {
            job: {
                name: "Upgrade",
                container: this.containerOptions,
                permissions: this.permissions,
                ...(0, runner_options_1.filteredRunsOnOptions)(this.options.workflowOptions?.runsOn, this.options.workflowOptions?.runsOnGroup),
                steps: steps,
                outputs: {
                    [PATCH_CREATED_OUTPUT]: {
                        stepId: CREATE_PATCH_STEP_ID,
                        outputName: PATCH_CREATED_OUTPUT,
                    },
                },
            },
            jobId: "upgrade",
            ref: branch,
        };
    }
    createPr(workflow, upgrade) {
        const credentials = this.options.workflowOptions?.projenCredentials ??
            workflow.projenCredentials;
        const semanticCommit = this.options.semanticCommit ?? "chore";
        return {
            job: github_1.WorkflowJobs.pullRequestFromPatch({
                patch: {
                    jobId: upgrade.jobId,
                    outputName: PATCH_CREATED_OUTPUT,
                    ref: upgrade.ref,
                },
                workflowName: workflow.name,
                credentials,
                ...(0, runner_options_1.filteredRunsOnOptions)(this.options.workflowOptions?.runsOn, this.options.workflowOptions?.runsOnGroup),
                pullRequestTitle: `${semanticCommit}(deps): ${this.pullRequestTitle}`,
                pullRequestDescription: "Upgrades project dependencies.",
                gitIdentity: this.gitIdentity,
                assignees: this.options.workflowOptions?.assignees,
                labels: this.options.workflowOptions?.labels,
                signoff: this.options.signoff,
            }),
            jobId: "pr",
        };
    }
}
exports.UpgradeDependencies = UpgradeDependencies;
_a = JSII_RTTI_SYMBOL_1;
UpgradeDependencies[_a] = { fqn: "projen.javascript.UpgradeDependencies", version: "0.91.13" };
/**
 * How often to check for new versions and raise pull requests for version upgrades.
 */
class UpgradeDependenciesSchedule {
    /**
     * Create a schedule from a raw cron expression.
     */
    static expressions(cron) {
        return new UpgradeDependenciesSchedule(cron);
    }
    constructor(cron) {
        this.cron = cron;
    }
}
exports.UpgradeDependenciesSchedule = UpgradeDependenciesSchedule;
_b = JSII_RTTI_SYMBOL_1;
UpgradeDependenciesSchedule[_b] = { fqn: "projen.javascript.UpgradeDependenciesSchedule", version: "0.91.13" };
/**
 * Disables automatic upgrades.
 */
UpgradeDependenciesSchedule.NEVER = new UpgradeDependenciesSchedule([]);
/**
 * At 00:00.
 */
UpgradeDependenciesSchedule.DAILY = new UpgradeDependenciesSchedule(["0 0 * * *"]);
/**
 * At 00:00 on every day-of-week from Monday through Friday.
 */
UpgradeDependenciesSchedule.WEEKDAY = new UpgradeDependenciesSchedule([
    "0 0 * * 1-5",
]);
/**
 * At 00:00 on Monday.
 */
UpgradeDependenciesSchedule.WEEKLY = new UpgradeDependenciesSchedule([
    "0 0 * * 1",
]);
/**
 * At 00:00 on day-of-month 1.
 */
UpgradeDependenciesSchedule.MONTHLY = new UpgradeDependenciesSchedule([
    "0 0 1 * *",
]);
//# sourceMappingURL=data:application/json;base64,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