"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProjectType = exports.Project = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const os_1 = require("os");
const path = require("path");
const constructs_1 = require("constructs");
const glob = require("glob");
const cleanup_1 = require("./cleanup");
const common_1 = require("./common");
const dependencies_1 = require("./dependencies");
const file_1 = require("./file");
const gitattributes_1 = require("./gitattributes");
const ignore_file_1 = require("./ignore-file");
const render_options_1 = require("./javascript/render-options");
const json_1 = require("./json");
const logger_1 = require("./logger");
const object_file_1 = require("./object-file");
const project_build_1 = require("./project-build");
const projenrc_json_1 = require("./projenrc-json");
const renovatebot_1 = require("./renovatebot");
const tasks_1 = require("./tasks");
const util_1 = require("./util");
const constructs_2 = require("./util/constructs");
/**
 * The default output directory for a project if none is specified.
 */
const DEFAULT_OUTDIR = ".";
/**
 * Base project
 */
class Project extends constructs_1.Construct {
    /**
     * Test whether the given construct is a project.
     */
    static isProject(x) {
        return (0, constructs_2.isProject)(x);
    }
    /**
     * Find the closest ancestor project for given construct.
     * When given a project, this it the project itself.
     *
     * @throws when no project is found in the path to the root
     */
    static of(construct) {
        return (0, constructs_2.findClosestProject)(construct);
    }
    /**
     * The command to use in order to run the projen CLI.
     */
    get projenCommand() {
        return this._projenCommand ?? "npx projen";
    }
    constructor(options) {
        const outdir = determineOutdir(options.parent, options.outdir);
        const autoId = `${new.target.name}#${options.name}@${path.normalize(options.outdir ?? "<root>")}`;
        if (options.parent?.subprojects.find((p) => p.outdir === outdir)) {
            throw new Error(`There is already a subproject with "outdir": ${outdir}`);
        }
        super(options.parent, autoId);
        this.tips = new Array();
        (0, constructs_2.tagAsProject)(this);
        this.node.addMetadata("type", "project");
        this.node.addMetadata("construct", new.target.name);
        this.initProject = (0, render_options_1.resolveInitProject)(options);
        this.name = options.name;
        this.parent = options.parent;
        this.excludeFromCleanup = [];
        this._ejected = (0, util_1.isTruthy)(process.env.PROJEN_EJECTING);
        this._projenCommand = options.projenCommand;
        if (this.ejected) {
            this._projenCommand = "scripts/run-task.cjs";
        }
        this.outdir = outdir;
        // ------------------------------------------------------------------------
        this.gitattributes = new gitattributes_1.GitAttributesFile(this, {
            endOfLine: options.gitOptions?.endOfLine,
        });
        this.annotateGenerated("/.projen/**"); // contents  of the .projen/ directory are generated by projen
        this.annotateGenerated(`/${this.gitattributes.path}`); // the .gitattributes file itself is generated
        if (options.gitOptions?.lfsPatterns) {
            for (const pattern of options.gitOptions.lfsPatterns) {
                this.gitattributes.addAttributes(pattern, "filter=lfs", "diff=lfs", "merge=lfs", "-text");
            }
        }
        this.gitignore = new ignore_file_1.IgnoreFile(this, ".gitignore", options.gitIgnoreOptions);
        this.gitignore.exclude("node_modules/"); // created by running `npx projen`
        this.gitignore.include(`/${this.gitattributes.path}`);
        // oh no: tasks depends on gitignore so it has to be initialized after
        // smells like dep injection but god forbid.
        this.tasks = new tasks_1.Tasks(this);
        if (!this.ejected) {
            this.defaultTask = this.tasks.addTask(Project.DEFAULT_TASK, {
                description: "Synthesize project files",
            });
            // Subtasks should call the root task for synth
            if (this.parent) {
                const cwd = path.relative(this.outdir, this.root.outdir);
                const normalizedCwd = (0, util_1.normalizePersistedPath)(cwd);
                this.defaultTask.exec(`${this.projenCommand} ${Project.DEFAULT_TASK}`, {
                    cwd: normalizedCwd,
                });
            }
            if (!this.parent) {
                this.ejectTask = this.tasks.addTask("eject", {
                    description: "Remove projen from the project",
                    env: {
                        PROJEN_EJECTING: "true",
                    },
                });
                this.ejectTask.spawn(this.defaultTask);
            }
        }
        this.projectBuild = new project_build_1.ProjectBuild(this);
        this.deps = new dependencies_1.Dependencies(this);
        this.logger = new logger_1.Logger(this, options.logging);
        const projenrcJson = options.projenrcJson ?? false;
        if (!this.parent && projenrcJson) {
            new projenrc_json_1.ProjenrcJson(this, options.projenrcJsonOptions);
        }
        if (options.renovatebot) {
            new renovatebot_1.Renovatebot(this, options.renovatebotOptions);
        }
        this.commitGenerated = options.commitGenerated ?? true;
        if (!this.ejected) {
            new json_1.JsonFile(this, cleanup_1.FILE_MANIFEST, {
                omitEmpty: true,
                obj: () => ({
                    // replace `\` with `/` to ensure paths match across platforms
                    files: this.files
                        .filter((f) => f.readonly)
                        .map((f) => (0, util_1.normalizePersistedPath)(f.path)),
                }),
                // This file is used by projen to track the generated files, so must be committed.
                committed: true,
            });
        }
    }
    /**
     * The root project.
     */
    get root() {
        return (0, constructs_2.isProject)(this.node.root) ? this.node.root : this;
    }
    /**
     * Returns all the components within this project.
     */
    get components() {
        return this.node
            .findAll()
            .filter((c) => (0, constructs_2.isComponent)(c) && c.project.node.path === this.node.path);
    }
    /**
     * Returns all the subprojects within this project.
     */
    get subprojects() {
        return this.node.children.filter(constructs_2.isProject);
    }
    /**
     * All files in this project.
     */
    get files() {
        return this.components
            .filter(isFile)
            .sort((f1, f2) => f1.path.localeCompare(f2.path));
    }
    /**
     * Adds a new task to this project. This will fail if the project already has
     * a task with this name.
     *
     * @param name The task name to add
     * @param props Task properties
     */
    addTask(name, props = {}) {
        return this.tasks.addTask(name, props);
    }
    /**
     * Removes a task from a project.
     *
     * @param name The name of the task to remove.
     *
     * @returns The `Task` that was removed, otherwise `undefined`.
     */
    removeTask(name) {
        return this.tasks.removeTask(name);
    }
    get buildTask() {
        return this.projectBuild.buildTask;
    }
    get compileTask() {
        return this.projectBuild.compileTask;
    }
    get testTask() {
        return this.projectBuild.testTask;
    }
    get preCompileTask() {
        return this.projectBuild.preCompileTask;
    }
    get postCompileTask() {
        return this.projectBuild.postCompileTask;
    }
    get packageTask() {
        return this.projectBuild.packageTask;
    }
    /**
     * Finds a file at the specified relative path within this project and all
     * its subprojects.
     *
     * @param filePath The file path. If this path is relative, it will be resolved
     * from the root of _this_ project.
     * @returns a `FileBase` or undefined if there is no file in that path
     */
    tryFindFile(filePath) {
        const absolute = path.isAbsolute(filePath)
            ? filePath
            : path.resolve(this.outdir, filePath);
        const candidate = this.node
            .findAll()
            .find((c) => (0, constructs_2.isComponent)(c) && isFile(c) && c.absolutePath === absolute);
        return candidate;
    }
    /**
     * Finds a json file by name.
     * @param filePath The file path.
     * @deprecated use `tryFindObjectFile`
     */
    tryFindJsonFile(filePath) {
        const file = this.tryFindObjectFile(filePath);
        if (!file) {
            return undefined;
        }
        if (!(file instanceof json_1.JsonFile)) {
            throw new Error(`found file ${filePath} but it is not a JsonFile. got: ${file.constructor.name}`);
        }
        return file;
    }
    /**
     * Finds an object file (like JsonFile, YamlFile, etc.) by name.
     * @param filePath The file path.
     */
    tryFindObjectFile(filePath) {
        const file = this.tryFindFile(filePath);
        if (!file) {
            return undefined;
        }
        if (!(file instanceof object_file_1.ObjectFile)) {
            throw new Error(`found file ${filePath} but it is not a ObjectFile. got: ${file.constructor.name}`);
        }
        return file;
    }
    /**
     * Finds a file at the specified relative path within this project and removes
     * it.
     *
     * @param filePath The file path. If this path is relative, it will be
     * resolved from the root of _this_ project.
     * @returns a `FileBase` if the file was found and removed, or undefined if
     * the file was not found.
     */
    tryRemoveFile(filePath) {
        const candidate = this.tryFindFile(filePath);
        if (candidate) {
            candidate.node.scope?.node.tryRemoveChild(candidate.node.id);
            return candidate;
        }
        return undefined;
    }
    /**
     * Prints a "tip" message during synthesis.
     * @param message The message
     * @deprecated - use `project.logger.info(message)` to show messages during synthesis
     */
    addTip(message) {
        this.tips.push(message);
    }
    /**
     * Exclude the matching files from pre-synth cleanup. Can be used when, for example, some
     * source files include the projen marker and we don't want them to be erased during synth.
     *
     * @param globs The glob patterns to match
     */
    addExcludeFromCleanup(...globs) {
        this.excludeFromCleanup.push(...globs);
    }
    /**
     * Returns the shell command to execute in order to run a task.
     *
     * By default, this is `npx projen@<version> <task>`
     *
     * @param task The task for which the command is required
     */
    runTaskCommand(task) {
        const pj = this._projenCommand ?? `npx projen@${common_1.PROJEN_VERSION}`;
        return `${pj} ${task.name}`;
    }
    /**
     * Exclude these files from the bundled package. Implemented by project types based on the
     * packaging mechanism. For example, `NodeProject` delegates this to `.npmignore`.
     *
     * @param _pattern The glob pattern to exclude
     */
    addPackageIgnore(_pattern) {
        // nothing to do at the abstract level
    }
    /**
     * Adds a .gitignore pattern.
     * @param pattern The glob pattern to ignore.
     */
    addGitIgnore(pattern) {
        this.gitignore.addPatterns(pattern);
    }
    /**
     * Consider a set of files as "generated". This method is implemented by
     * derived classes and used for example, to add git attributes to tell GitHub
     * that certain files are generated.
     *
     * @param _glob the glob pattern to match (could be a file path).
     */
    annotateGenerated(_glob) {
        // nothing to do at the abstract level
    }
    /**
     * Synthesize all project files into `outdir`.
     *
     * 1. Call "this.preSynthesize()"
     * 2. Delete all generated files
     * 3. Synthesize all subprojects
     * 4. Synthesize all components of this project
     * 5. Call "postSynthesize()" for all components of this project
     * 6. Call "this.postSynthesize()"
     */
    synth() {
        const outdir = this.outdir;
        this.logger.debug("Synthesizing project...");
        this.preSynthesize();
        for (const comp of this.components) {
            comp.preSynthesize();
        }
        // we exclude all subproject directories to ensure that when subproject.synth()
        // gets called below after cleanup(), subproject generated files are left intact
        for (const subproject of this.subprojects) {
            this.addExcludeFromCleanup(subproject.outdir + "/**");
        }
        // delete orphaned files before we start synthesizing new ones
        (0, cleanup_1.cleanup)(outdir, this.files.map((f) => (0, util_1.normalizePersistedPath)(f.path)), this.excludeFromCleanup);
        for (const subproject of this.subprojects) {
            subproject.synth();
        }
        for (const comp of this.components) {
            comp.synthesize();
        }
        if (!(0, util_1.isTruthy)(process.env.PROJEN_DISABLE_POST)) {
            for (const comp of this.components) {
                comp.postSynthesize();
            }
            // project-level hook
            this.postSynthesize();
        }
        if (this.ejected) {
            this.logger.debug("Ejecting project...");
            // Backup projenrc files
            const files = glob.sync(".projenrc.*", {
                cwd: this.outdir,
                dot: true,
                nodir: true,
                absolute: true,
            });
            for (const file of files) {
                (0, fs_1.renameSync)(file, `${file}.bak`);
            }
        }
        this.logger.debug("Synthesis complete");
    }
    /**
     * Whether or not the project is being ejected.
     */
    get ejected() {
        return this._ejected;
    }
    /**
     * Called before all components are synthesized.
     */
    preSynthesize() { }
    /**
     * Called after all components are synthesized. Order is *not* guaranteed.
     */
    postSynthesize() { }
}
exports.Project = Project;
_a = JSII_RTTI_SYMBOL_1;
Project[_a] = { fqn: "projen.Project", version: "0.91.13" };
/**
 * The name of the default task (the task executed when `projen` is run without arguments). Normally
 * this task should synthesize the project files.
 */
Project.DEFAULT_TASK = "default";
/**
 * Which type of project this is.
 *
 * @deprecated no longer supported at the base project level
 */
var ProjectType;
(function (ProjectType) {
    /**
     * This module may be a either a library or an app.
     */
    ProjectType["UNKNOWN"] = "unknown";
    /**
     * This is a library, intended to be published to a package manager and
     * consumed by other projects.
     */
    ProjectType["LIB"] = "lib";
    /**
     * This is an app (service, tool, website, etc). Its artifacts are intended to
     * be deployed or published for end-user consumption.
     */
    ProjectType["APP"] = "app";
})(ProjectType || (exports.ProjectType = ProjectType = {}));
/**
 * Resolves the project's output directory.
 */
function determineOutdir(parent, outdirOption) {
    if (parent && outdirOption && path.isAbsolute(outdirOption)) {
        throw new Error('"outdir" must be a relative path');
    }
    // if this is a subproject, it is relative to the parent
    if (parent) {
        if (!outdirOption) {
            throw new Error('"outdir" must be specified for subprojects');
        }
        return path.resolve(parent.outdir, outdirOption);
    }
    // if this is running inside a test and outdir is not explicitly set
    // use a temp directory (unless cwd is already under tmp)
    if (common_1.IS_TEST_RUN && !outdirOption) {
        const realCwd = (0, fs_1.realpathSync)(process.cwd());
        const realTmp = (0, fs_1.realpathSync)((0, os_1.tmpdir)());
        if (realCwd.startsWith(realTmp)) {
            return path.resolve(realCwd, outdirOption ?? DEFAULT_OUTDIR);
        }
        return (0, fs_1.mkdtempSync)(path.join((0, os_1.tmpdir)(), "projen."));
    }
    return path.resolve(outdirOption ?? DEFAULT_OUTDIR);
}
function isFile(c) {
    return c instanceof file_1.FileBase;
}
//# sourceMappingURL=data:application/json;base64,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