"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PythonProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const pip_1 = require("./pip");
const poetry_1 = require("./poetry");
const projenrc_1 = require("./projenrc");
const pytest_1 = require("./pytest");
const pytest_sample_1 = require("./pytest-sample");
const python_sample_1 = require("./python-sample");
const setuptools_1 = require("./setuptools");
const venv_1 = require("./venv");
const github_1 = require("../github");
const projenrc_2 = require("../javascript/projenrc");
const project_1 = require("../project");
const typescript_1 = require("../typescript");
const util_1 = require("../util");
/** Allowed characters in python project names */
const PYTHON_PROJECT_NAME_REGEX = /^[A-Za-z0-9-_\.]+$/;
/**
 * Python project.
 *
 * @pjid python
 */
class PythonProject extends github_1.GitHubProject {
    constructor(options) {
        super(options);
        if (!PYTHON_PROJECT_NAME_REGEX.test(options.name)) {
            throw new Error("Python projects must only consist of alphanumeric characters, hyphens, and underscores.");
        }
        this.moduleName = options.moduleName;
        this.version = options.version;
        this.sampleTestdir = options.sampleTestdir ?? "tests";
        const rcFileTypeOptions = [
            options.projenrcPython,
            options.projenrcJs,
            options.projenrcJson,
            options.projenrcTs,
        ];
        if ((0, util_1.multipleSelected)(rcFileTypeOptions)) {
            throw new Error("Only one of projenrcPython, projenrcJs, projenrcTs, and projenrcJson can be selected.");
        }
        const poetry = options.poetry ?? false;
        const pip = options.pip ?? !poetry;
        const venv = options.venv ?? !poetry;
        const setuptools = options.setuptools ?? (!poetry && this.projectType === project_1.ProjectType.LIB);
        if (poetry && (pip || venv || setuptools)) {
            throw new Error("poetry is true - pip, venv, and setuptools must be undefined or false");
        }
        if (!this.parent) {
            // default to projenrc.py if no other projenrc type was elected
            if (options.projenrcPython ?? !(0, util_1.anySelected)(rcFileTypeOptions)) {
                new projenrc_1.Projenrc(this, {
                    pythonExec: options.pythonExec,
                    ...options.projenrcPythonOptions,
                });
            }
            if (options.projenrcJs ?? false) {
                new projenrc_2.Projenrc(this, options.projenrcJsOptions);
            }
            if (options.projenrcTs ?? false) {
                new typescript_1.ProjenrcTs(this, options.projenrcTsOptions);
            }
        }
        if (venv) {
            this.envManager = new venv_1.Venv(this, {
                pythonExec: options.pythonExec,
                ...options.venvOptions,
            });
        }
        if (pip) {
            this.depsManager = new pip_1.Pip(this);
        }
        if (setuptools) {
            this.packagingManager = new setuptools_1.Setuptools(this, {
                version: options.version,
                description: options.description,
                authorName: options.authorName,
                authorEmail: options.authorEmail,
                license: options.license,
                homepage: options.homepage,
                classifiers: options.classifiers,
                setupConfig: options.setupConfig,
                pythonExec: options.pythonExec,
            });
        }
        if (poetry) {
            const poetryProject = new poetry_1.Poetry(this, {
                version: options.version,
                description: options.description,
                authorName: options.authorName,
                authorEmail: options.authorEmail,
                license: options.license,
                homepage: options.homepage,
                classifiers: options.classifiers,
                pythonExec: options.pythonExec,
                poetryOptions: {
                    readme: options.readme?.filename ?? "README.md",
                    ...options.poetryOptions,
                },
            });
            this.depsManager = poetryProject;
            this.envManager = poetryProject;
            this.packagingManager = poetryProject;
        }
        if (!this.envManager) {
            throw new Error("At least one tool must be chosen for managing the environment (venv, conda, pipenv, or poetry).");
        }
        if (!this.depsManager) {
            throw new Error("At least one tool must be chosen for managing dependencies (pip, conda, pipenv, or poetry).");
        }
        if (!this.packagingManager && this.projectType === project_1.ProjectType.LIB) {
            throw new Error("At least one tool must be chosen for managing packaging (setuptools or poetry).");
        }
        if (Number(venv) + Number(poetry) > 1) {
            throw new Error("More than one component has been chosen for managing the environment (venv, conda, pipenv, or poetry)");
        }
        if (Number(pip) + Number(poetry) > 1) {
            throw new Error("More than one component has been chosen for managing dependencies (pip, conda, pipenv, or poetry)");
        }
        if (Number(setuptools) + Number(poetry) > 1) {
            throw new Error("More than one component has been chosen for managing packaging (setuptools or poetry)");
        }
        if (options.pytest ?? true) {
            this.pytest = new pytest_1.Pytest(this, options.pytestOptions);
            if (options.sample ?? true) {
                new pytest_sample_1.PytestSample(this, {
                    moduleName: this.moduleName,
                    testdir: this.sampleTestdir,
                });
            }
        }
        if (options.sample ?? true) {
            new python_sample_1.PythonSample(this, {
                dir: this.moduleName,
            });
        }
        for (const dep of options.deps ?? []) {
            this.addDependency(dep);
        }
        for (const dep of options.devDeps ?? []) {
            this.addDevDependency(dep);
        }
        this.addDefaultGitIgnore();
    }
    /**
     * Adds default gitignore options for a Python project based on
     * https://github.com/github/gitignore/blob/master/Python.gitignore
     */
    addDefaultGitIgnore() {
        this.gitignore.exclude("# Byte-compiled / optimized / DLL files", "__pycache__/", "*.py[cod]", "*$py.class", "", "# C extensions", "*.so", "", "# Distribution / packaging", ".Python", "build/", "develop-eggs/", "dist/", "downloads/", "eggs/", ".eggs/", "lib/", "lib64/", "parts/", "sdist/", "var/", "wheels/", "share/python-wheels/", "*.egg-info/", ".installed.cfg", "*.egg", "MANIFEST", "", "# PyInstaller", "#  Usually these files are written by a python script from a template", "#  before PyInstaller builds the exe, so as to inject date/other infos into it.", "*.manifest", "*.spec", "", "# Installer logs", "pip-log.txt", "pip-delete-this-directory.txt", "", "# Unit test / coverage reports", "htmlcov/", ".tox/", ".nox/", ".coverage", ".coverage.*", ".cache", "nosetests.xml", "coverage.xml", "*.cover", "*.py,cover", ".hypothesis/", ".pytest_cache/", "cover/", "", "# Translations", "*.mo", "*.pot", "", "# Django stuff:", "*.log", "local_settings.py", "db.sqlite3", "db.sqlite3-journal", "", "# Flask stuff:", "instance/", ".webassets-cache", "", "# Scrapy stuff:", ".scrapy", "", "# Sphinx documentation", "docs/_build/", "", "# PyBuilder", ".pybuilder/", "target/", "", "# Jupyter Notebook", ".ipynb_checkpoints", "", "# IPython", "profile_default/", "ipython_config.py", "", "# PEP 582; used by e.g. github.com/David-OConnor/pyflow", "__pypackages__/", "", "# Celery stuff", "celerybeat-schedule", "celerybeat.pid", "", "# SageMath parsed files", "*.sage.py", "", "# Environments", ".env", ".venv", "env/", "venv/", "ENV/", "env.bak/", "venv.bak/", "", "# Spyder project settings", ".spyderproject", ".spyproject", "", "# Rope project settings", ".ropeproject", "", "# mkdocs documentation", "/site", "", "# mypy", ".mypy_cache/", ".dmypy.json", "dmypy.json", "", "# Pyre type checker", ".pyre/", "", "# pytype static type analyzer", ".pytype/", "", "# Cython debug symbols", "cython_debug/");
    }
    /**
     * Adds a runtime dependency.
     *
     * @param spec Format `<module>@<semver>`
     */
    addDependency(spec) {
        return this.depsManager.addDependency(spec);
    }
    /**
     * Adds a dev dependency.
     *
     * @param spec Format `<module>@<semver>`
     */
    addDevDependency(spec) {
        return this.depsManager.addDevDependency(spec);
    }
    postSynthesize() {
        super.postSynthesize();
        this.envManager.setupEnvironment();
        this.depsManager.installDependencies();
    }
}
exports.PythonProject = PythonProject;
_a = JSII_RTTI_SYMBOL_1;
PythonProject[_a] = { fqn: "projen.python.PythonProject", version: "0.91.13" };
//# sourceMappingURL=data:application/json;base64,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