import { Config } from "conventional-changelog-config-spec";
export interface BumpOptions {
    /**
     * The name of a .json file to set `version`.
     */
    readonly versionFile: string;
    /**
     * The name of the changelog file to generate.
     */
    readonly changelog: string;
    /**
     * Use a pre-release suffix.
     * @default - normal versioning
     */
    readonly prerelease?: string;
    /**
     * Defines the major version line. This is used to select the latest version
     * and also enforce that new major versions are not released accidentally.
     *
     * Can not be set together with `minMajorVersion`.
     *
     * @default - any version is supported
     */
    readonly majorVersion?: number;
    /**
     * Defines the minimal major version. This is used if you want to start with
     * a specific major version, and increment from there on.
     * This can be useful to set to 1, as breaking changes before the 1.x major
     * release are not incrementing the major version number.
     *
     * Can not be set together with `majorVersion`.
     *
     * @default - No minimum version is being enforced
     */
    readonly minMajorVersion?: number;
    /**
     * Defines the minor version line. This is used to select the latest version
     * and also enforce that new minor versions are not released accidentally.
     *
     * @default - any version is supported
     */
    readonly minorVersion?: number;
    /**
     * The name of a file which will include the output version number (a text file).
     *
     * Relative to cwd.
     *
     * @example ".version.txt"
     */
    readonly bumpFile: string;
    /**
     * The name of the file which will include the release tag (a text file).
     *
     * Relative to cwd.
     *
     * @example ".releasetag.txt"
     */
    readonly releaseTagFile: string;
    /**
     * The prefix applied to release tags. Bumps will be made based on the latest
     * version found with this prefix.
     */
    readonly tagPrefix?: string;
    /**
     * Configuration values that would append to versionrc file or overwrite values
     * coming to that from default one.
     */
    readonly versionrcOptions?: Config;
    /**
     * A shell command to list all release commits since the latest tag.
     *
     * A new release will be initiated, if the number of returned commits is greater than zero.
     *
     * `$LATEST_TAG` will be replaced with the actual latest tag for the given prefix.
     *
     * @default "git log --oneline $LATEST_TAG..HEAD"
     */
    readonly releasableCommits?: string;
    /**
     * The `commit-and-tag-version` compatible package used to bump the package version, as a dependency string.
     *
     * This can be any compatible package version, including the deprecated `standard-version@9`.
     *
     * @default "commit-and-tag-version@12"
     */
    readonly bumpPackage?: string;
    /**
     * A shell command to control the next version to release.
     *
     * If present, this shell command will be run before the bump is executed, and
     * it determines what version to release. It will be executed in the following
     * environment:
     *
     * - Working directory: the project directory.
     * - `$VERSION`: the current version. Looks like `1.2.3`.
     * - `$LATEST_TAG`: the most recent tag. Looks like `prefix-v1.2.3`, or may be unset.
     *
     * The command should print one of the following to `stdout`:
     *
     * - Nothing: the next version number will be determined based on commit history.
     * - `x.y.z`: the next version number will be `x.y.z`.
     * - `major|minor|patch`: the next version number will be the current version number
     *   with the indicated component bumped.
     *
     * This setting cannot be specified together with `minMajorVersion`; the invoked
     * script can be used to achieve the effects of `minMajorVersion`.
     *
     * @default - The next version will be determined based on the commit history and project settings.
     */
    readonly nextVersionCommand?: string;
}
/**
 * Resolves the latest version from git tags and uses `commit-and-tag-version` to bump
 * to the next version based on commits.
 *
 * This expects `commit-and-tag-version` to be installed in the path.
 *
 * @param cwd working directory (git repository)
 * @param options options
 */
export declare function bump(cwd: string, options: BumpOptions): Promise<void>;
