"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeArtifactAuthProvider = exports.Publisher = void 0;
exports.isAwsCodeArtifactRegistry = isAwsCodeArtifactRegistry;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const component_1 = require("../component");
const constants_1 = require("../github/constants");
const workflows_model_1 = require("../github/workflows-model");
const node_package_1 = require("../javascript/node-package");
const runner_options_1 = require("../runner-options");
const version_1 = require("../version");
const PUBLIB_VERSION = "latest";
const GITHUB_PACKAGES_REGISTRY = "npm.pkg.github.com";
const ARTIFACTS_DOWNLOAD_DIR = "dist";
const GITHUB_PACKAGES_MAVEN_REPOSITORY = "https://maven.pkg.github.com";
const GITHUB_PACKAGES_NUGET_REPOSITORY = "https://nuget.pkg.github.com";
const AWS_CODEARTIFACT_REGISTRY_REGEX = /.codeartifact.*.amazonaws.com/;
const PUBLIB_TOOLCHAIN = {
    js: {},
    java: { java: { version: "11" } },
    python: { python: { version: "3.x" } },
    go: { go: { version: "^1.18.0" } },
    dotnet: { dotnet: { version: "6.x" } },
};
const PUBLISH_JOB_PREFIX = "release_";
/**
 * Implements GitHub jobs for publishing modules to package managers.
 *
 * Under the hood, it uses https://github.com/aws/publib
 */
class Publisher extends component_1.Component {
    constructor(project, options) {
        super(project);
        // functions that create jobs associated with a specific branch
        this._jobFactories = [];
        this._gitHubPrePublishing = [];
        this._gitHubPostPublishing = [];
        // List of publish jobs added to the publisher
        // Maps between the basename and the jobname
        this.publishJobs = {};
        this.buildJobId = options.buildJobId;
        this.artifactName = options.artifactName;
        this.publibVersion =
            options.publibVersion ?? options.jsiiReleaseVersion ?? PUBLIB_VERSION;
        this.jsiiReleaseVersion = this.publibVersion;
        this.condition = options.condition;
        this.dryRun = options.dryRun ?? false;
        this.workflowNodeVersion = options.workflowNodeVersion ?? "lts/*";
        this.workflowContainerImage = options.workflowContainerImage;
        this.failureIssue = options.failureIssue ?? false;
        this.failureIssueLabel = options.failureIssueLabel ?? "failed-release";
        this.publishTasks = options.publishTasks ?? false;
        this.runsOn = options.workflowRunsOn;
        this.runsOnGroup = options.workflowRunsOnGroup;
    }
    /**
     * Called by `Release` to add the publishing jobs to a release workflow
     * associated with a specific branch.
     * @param branch The branch name
     * @param options Branch options
     *
     * @internal
     */
    _renderJobsForBranch(branch, options) {
        let jobs = {};
        for (const factory of this._jobFactories) {
            jobs = {
                ...jobs,
                ...factory(branch, options),
            };
        }
        return jobs;
    }
    /**
     * Adds pre publishing steps for the GitHub release job.
     *
     * @param steps The steps.
     */
    addGitHubPrePublishingSteps(...steps) {
        this._gitHubPrePublishing.push(...steps);
    }
    /**
     * Adds post publishing steps for the GitHub release job.
     *
     * @param steps The steps.
     */
    addGitHubPostPublishingSteps(...steps) {
        this._gitHubPostPublishing.push(...steps);
    }
    /**
     * Publish to git.
     *
     * This includes generating a project-level changelog and release tags.
     *
     * @param options Options
     */
    publishToGit(options) {
        const releaseTagFile = options.releaseTagFile;
        const versionFile = options.versionFile;
        const changelog = options.changelogFile;
        const projectChangelogFile = options.projectChangelogFile;
        const gitBranch = options.gitBranch ?? "main";
        const taskName = gitBranch === "main" || gitBranch === "master"
            ? Publisher.PUBLISH_GIT_TASK_NAME
            : `${Publisher.PUBLISH_GIT_TASK_NAME}:${gitBranch}`;
        const publishTask = this.project.addTask(taskName, {
            description: "Prepends the release changelog onto the project changelog, creates a release commit, and tags the release",
            env: {
                CHANGELOG: changelog,
                RELEASE_TAG_FILE: releaseTagFile,
                PROJECT_CHANGELOG_FILE: projectChangelogFile ?? "",
                VERSION_FILE: versionFile,
            },
            condition: version_1.CHANGES_SINCE_LAST_RELEASE,
        });
        if (projectChangelogFile) {
            publishTask.builtin("release/update-changelog");
        }
        publishTask.builtin("release/tag-version");
        if (options.gitPushCommand !== "") {
            const gitPushCommand = options.gitPushCommand || `git push --follow-tags origin ${gitBranch}`;
            publishTask.exec(gitPushCommand);
        }
        return publishTask;
    }
    /**
     * Creates a GitHub Release.
     * @param options Options
     */
    publishToGitHubReleases(options) {
        const jobName = "github";
        this.addPublishJob(jobName, (_branch, branchOptions) => {
            return {
                registryName: "GitHub Releases",
                prePublishSteps: options.prePublishSteps ?? this._gitHubPrePublishing,
                postPublishSteps: options.postPublishSteps ?? this._gitHubPostPublishing,
                publishTools: options.publishTools,
                permissions: {
                    contents: workflows_model_1.JobPermission.WRITE,
                },
                needs: Object.entries(this.publishJobs)
                    .filter(([name, _]) => name != jobName)
                    .map(([_, job]) => job),
                workflowEnv: {
                    GITHUB_TOKEN: "${{ secrets.GITHUB_TOKEN }}",
                    GITHUB_REPOSITORY: "${{ github.repository }}",
                    GITHUB_REF: "${{ github.sha }}",
                },
                run: this.githubReleaseCommand(options, branchOptions),
            };
        });
    }
    /**
     * Publishes artifacts from `js/**` to npm.
     * @param options Options
     */
    publishToNpm(options = {}) {
        const isGitHubPackages = options.registry?.startsWith(GITHUB_PACKAGES_REGISTRY);
        const isAwsCodeArtifact = isAwsCodeArtifactRegistry(options.registry);
        const isAwsCodeArtifactWithOidc = isAwsCodeArtifact &&
            options.codeArtifactOptions?.authProvider ===
                CodeArtifactAuthProvider.GITHUB_OIDC;
        const npmToken = (0, node_package_1.defaultNpmToken)(options.npmTokenSecret, options.registry);
        if (options.distTag) {
            this.project.logger.warn("The `distTag` option is deprecated. Use the npmDistTag option instead.");
        }
        const prePublishSteps = options.prePublishSteps ?? [];
        if (isAwsCodeArtifactWithOidc) {
            if (options.codeArtifactOptions?.accessKeyIdSecret ||
                options.codeArtifactOptions?.secretAccessKeySecret) {
                throw new Error("access and secret key pair should not be provided when using GITHUB_OIDC auth provider for AWS CodeArtifact");
            }
            else if (!options.codeArtifactOptions?.roleToAssume) {
                throw new Error('"roleToAssume" property is required when using GITHUB_OIDC for AWS CodeArtifact options');
            }
            const regionCaptureRegex = /codeartifact\.(.+)\.amazonaws\.com/;
            const region = options.registry?.match(regionCaptureRegex)?.[1];
            prePublishSteps.push({
                name: "Configure AWS Credentials via GitHub OIDC Provider",
                uses: "aws-actions/configure-aws-credentials@v4",
                with: {
                    "role-to-assume": options.codeArtifactOptions.roleToAssume,
                    "aws-region": region,
                },
            });
        }
        this.addPublishJob("npm", (_branch, branchOptions) => {
            if (branchOptions.npmDistTag && options.distTag) {
                throw new Error("cannot set branch-level npmDistTag and npmDistTag in publishToNpm()");
            }
            const npmProvenance = options.npmProvenance ? "true" : undefined;
            const needsIdTokenWrite = isAwsCodeArtifactWithOidc || npmProvenance;
            return {
                publishTools: PUBLIB_TOOLCHAIN.js,
                prePublishSteps,
                postPublishSteps: options.postPublishSteps ?? [],
                run: this.publibCommand("publib-npm"),
                registryName: "npm",
                env: {
                    NPM_DIST_TAG: branchOptions.npmDistTag ?? options.distTag ?? "latest",
                    NPM_REGISTRY: options.registry,
                    NPM_CONFIG_PROVENANCE: npmProvenance,
                },
                permissions: {
                    idToken: needsIdTokenWrite ? workflows_model_1.JobPermission.WRITE : undefined,
                    contents: workflows_model_1.JobPermission.READ,
                    packages: isGitHubPackages ? workflows_model_1.JobPermission.WRITE : undefined,
                },
                workflowEnv: {
                    NPM_TOKEN: npmToken ? secret(npmToken) : undefined,
                    // if we are publishing to AWS CodeArtifact, pass AWS access keys that will be used to generate NPM_TOKEN using AWS CLI.
                    AWS_ACCESS_KEY_ID: isAwsCodeArtifact && !isAwsCodeArtifactWithOidc
                        ? secret(options.codeArtifactOptions?.accessKeyIdSecret ??
                            "AWS_ACCESS_KEY_ID")
                        : undefined,
                    AWS_SECRET_ACCESS_KEY: isAwsCodeArtifact && !isAwsCodeArtifactWithOidc
                        ? secret(options.codeArtifactOptions?.secretAccessKeySecret ??
                            "AWS_SECRET_ACCESS_KEY")
                        : undefined,
                    AWS_ROLE_TO_ASSUME: isAwsCodeArtifact && !isAwsCodeArtifactWithOidc
                        ? options.codeArtifactOptions?.roleToAssume
                        : undefined,
                },
            };
        });
    }
    /**
     * Publishes artifacts from `dotnet/**` to NuGet Gallery.
     * @param options Options
     */
    publishToNuget(options = {}) {
        const isGitHubPackages = options.nugetServer?.startsWith(GITHUB_PACKAGES_NUGET_REPOSITORY);
        this.addPublishJob("nuget", (_branch, _branchOptions) => ({
            publishTools: PUBLIB_TOOLCHAIN.dotnet,
            prePublishSteps: options.prePublishSteps ?? [],
            postPublishSteps: options.postPublishSteps ?? [],
            run: this.publibCommand("publib-nuget"),
            registryName: "NuGet Gallery",
            permissions: {
                contents: workflows_model_1.JobPermission.READ,
                packages: isGitHubPackages ? workflows_model_1.JobPermission.WRITE : undefined,
            },
            workflowEnv: {
                NUGET_API_KEY: secret(isGitHubPackages
                    ? "GITHUB_TOKEN"
                    : options.nugetApiKeySecret ?? "NUGET_API_KEY"),
                NUGET_SERVER: options.nugetServer ?? undefined,
            },
        }));
    }
    /**
     * Publishes artifacts from `java/**` to Maven.
     * @param options Options
     */
    publishToMaven(options = {}) {
        const isGitHubPackages = options.mavenRepositoryUrl?.startsWith(GITHUB_PACKAGES_MAVEN_REPOSITORY);
        const isGitHubActor = isGitHubPackages && options.mavenUsername == undefined;
        const mavenServerId = options.mavenServerId ?? (isGitHubPackages ? "github" : undefined);
        if (isGitHubPackages && mavenServerId != "github") {
            throw new Error('publishing to GitHub Packages requires the "mavenServerId" to be "github"');
        }
        this.addPublishJob("maven", (_branch, _branchOptions) => ({
            registryName: "Maven Central",
            publishTools: PUBLIB_TOOLCHAIN.java,
            prePublishSteps: options.prePublishSteps ?? [],
            postPublishSteps: options.postPublishSteps ?? [],
            run: this.publibCommand("publib-maven"),
            env: {
                MAVEN_ENDPOINT: options.mavenEndpoint,
                MAVEN_SERVER_ID: mavenServerId,
                MAVEN_REPOSITORY_URL: options.mavenRepositoryUrl,
            },
            workflowEnv: {
                MAVEN_GPG_PRIVATE_KEY: isGitHubPackages
                    ? undefined
                    : secret(options.mavenGpgPrivateKeySecret ?? "MAVEN_GPG_PRIVATE_KEY"),
                MAVEN_GPG_PRIVATE_KEY_PASSPHRASE: isGitHubPackages
                    ? undefined
                    : secret(options.mavenGpgPrivateKeyPassphrase ??
                        "MAVEN_GPG_PRIVATE_KEY_PASSPHRASE"),
                MAVEN_PASSWORD: secret(options.mavenPassword ??
                    (isGitHubPackages ? "GITHUB_TOKEN" : "MAVEN_PASSWORD")),
                MAVEN_USERNAME: isGitHubActor
                    ? "${{ github.actor }}"
                    : secret(options.mavenUsername ?? "MAVEN_USERNAME"),
                MAVEN_STAGING_PROFILE_ID: isGitHubPackages
                    ? undefined
                    : secret(options.mavenStagingProfileId ?? "MAVEN_STAGING_PROFILE_ID"),
            },
            permissions: {
                contents: workflows_model_1.JobPermission.READ,
                packages: isGitHubPackages ? workflows_model_1.JobPermission.WRITE : undefined,
            },
        }));
    }
    /**
     * Publishes wheel artifacts from `python` to PyPI.
     * @param options Options
     */
    publishToPyPi(options = {}) {
        let permissions = { contents: workflows_model_1.JobPermission.READ };
        const prePublishSteps = options.prePublishSteps ?? [];
        let workflowEnv = {};
        const isAwsCodeArtifact = isAwsCodeArtifactRegistry(options.twineRegistryUrl);
        if (isAwsCodeArtifact) {
            const { domain, account, region } = awsCodeArtifactInfoFromUrl(options.twineRegistryUrl);
            const { authProvider, roleToAssume, accessKeyIdSecret, secretAccessKeySecret, } = options.codeArtifactOptions ?? {};
            const useOidcAuth = authProvider === CodeArtifactAuthProvider.GITHUB_OIDC;
            if (useOidcAuth) {
                if (!roleToAssume) {
                    throw new Error('"roleToAssume" property is required when using GITHUB_OIDC for AWS CodeArtifact options');
                }
                permissions = { ...permissions, idToken: workflows_model_1.JobPermission.WRITE };
                prePublishSteps.push({
                    name: "Configure AWS Credentials via GitHub OIDC Provider",
                    uses: "aws-actions/configure-aws-credentials@v4",
                    with: {
                        "role-to-assume": roleToAssume,
                        "aws-region": region,
                    },
                });
            }
            prePublishSteps.push({
                name: "Generate CodeArtifact Token",
                run: `echo "TWINE_PASSWORD=$(aws codeartifact get-authorization-token --domain ${domain} --domain-owner ${account} --region ${region} --query authorizationToken --output text)" >> $GITHUB_ENV`,
                env: useOidcAuth
                    ? undefined
                    : {
                        AWS_ACCESS_KEY_ID: secret(accessKeyIdSecret ?? "AWS_ACCESS_KEY_ID"),
                        AWS_SECRET_ACCESS_KEY: secret(secretAccessKeySecret ?? "AWS_SECRET_ACCESS_KEY"),
                    },
            });
            workflowEnv = { TWINE_USERNAME: "aws" };
        }
        else {
            workflowEnv = {
                TWINE_USERNAME: secret(options.twineUsernameSecret ?? "TWINE_USERNAME"),
                TWINE_PASSWORD: secret(options.twinePasswordSecret ?? "TWINE_PASSWORD"),
            };
        }
        this.addPublishJob("pypi", (_branch, _branchOptions) => ({
            registryName: "PyPI",
            publishTools: PUBLIB_TOOLCHAIN.python,
            permissions,
            prePublishSteps,
            postPublishSteps: options.postPublishSteps ?? [],
            run: this.publibCommand("publib-pypi"),
            env: {
                TWINE_REPOSITORY_URL: options.twineRegistryUrl,
            },
            workflowEnv,
        }));
    }
    /**
     * Adds a go publishing job.
     * @param options Options
     */
    publishToGo(options = {}) {
        const prePublishSteps = options.prePublishSteps ?? [];
        const workflowEnv = {};
        if (options.githubUseSsh) {
            workflowEnv.GITHUB_USE_SSH = "true";
            workflowEnv.SSH_AUTH_SOCK = "/tmp/ssh_agent.sock";
            prePublishSteps.push({
                name: "Setup GitHub deploy key",
                run: 'ssh-agent -a ${SSH_AUTH_SOCK} && ssh-add - <<< "${GITHUB_DEPLOY_KEY}"',
                env: {
                    GITHUB_DEPLOY_KEY: secret(options.githubDeployKeySecret ?? "GO_GITHUB_DEPLOY_KEY"),
                    SSH_AUTH_SOCK: workflowEnv.SSH_AUTH_SOCK,
                },
            });
        }
        else {
            workflowEnv.GITHUB_TOKEN = secret(options.githubTokenSecret ?? "GO_GITHUB_TOKEN");
        }
        this.addPublishJob("golang", (_branch, _branchOptions) => ({
            publishTools: PUBLIB_TOOLCHAIN.go,
            prePublishSteps: prePublishSteps,
            postPublishSteps: options.postPublishSteps ?? [],
            run: this.publibCommand("publib-golang"),
            registryName: "GitHub Go Module Repository",
            env: {
                GIT_BRANCH: options.gitBranch,
                GIT_USER_NAME: options.gitUserName ?? constants_1.DEFAULT_GITHUB_ACTIONS_USER.name,
                GIT_USER_EMAIL: options.gitUserEmail ?? constants_1.DEFAULT_GITHUB_ACTIONS_USER.email,
                GIT_COMMIT_MESSAGE: options.gitCommitMessage,
            },
            workflowEnv: workflowEnv,
        }));
    }
    addPublishJob(
    /**
     * The basename of the publish job (should be lowercase).
     * Will be extended with a prefix.
     */
    basename, factory) {
        const jobname = `${PUBLISH_JOB_PREFIX}${basename}`;
        this.publishJobs[basename] = jobname;
        this._jobFactories.push((branch, branchOptions) => {
            const opts = factory(branch, branchOptions);
            if (jobname in this._jobFactories) {
                throw new Error(`Duplicate job with name "${jobname}"`);
            }
            const commandToRun = this.dryRun
                ? `echo "DRY RUN: ${opts.run}"`
                : opts.run;
            const requiredEnv = new Array();
            // jobEnv is the env we pass to the github job (task environment + secrets/expressions).
            const jobEnv = { ...opts.env };
            const workflowEnvEntries = Object.entries(opts.workflowEnv ?? {}).filter(([_, value]) => value != undefined);
            for (const [env, expression] of workflowEnvEntries) {
                requiredEnv.push(env);
                jobEnv[env] = expression;
            }
            if (this.publishTasks) {
                const branchSuffix = branch === "main" || branch === "master" ? "" : `:${branch}`;
                // define a task which can be used through `projen publish:xxx`.
                const task = this.project.addTask(`publish:${basename.toLocaleLowerCase()}${branchSuffix}`, {
                    description: `Publish this package to ${opts.registryName}`,
                    env: opts.env,
                    requiredEnv: requiredEnv,
                });
                // first verify that we are on the correct branch
                task.exec(`test "$(git branch --show-current)" = "${branch}"`);
                // run commands
                task.exec(commandToRun);
            }
            const steps = [
                {
                    name: "Download build artifacts",
                    uses: "actions/download-artifact@v4",
                    with: {
                        name: constants_1.BUILD_ARTIFACT_NAME,
                        path: ARTIFACTS_DOWNLOAD_DIR, // this must be "dist" for publib
                    },
                },
                {
                    name: "Restore build artifact permissions",
                    continueOnError: true,
                    run: [
                        `cd ${ARTIFACTS_DOWNLOAD_DIR} && setfacl --restore=${constants_1.PERMISSION_BACKUP_FILE}`,
                    ].join("\n"),
                },
                ...opts.prePublishSteps,
                {
                    name: "Release",
                    // it would have been nice if we could just run "projen publish:xxx" here but that is not possible because this job does not checkout sources
                    run: commandToRun,
                    env: jobEnv,
                },
                ...opts.postPublishSteps,
            ];
            const perms = opts.permissions ?? { contents: workflows_model_1.JobPermission.READ };
            const container = this.workflowContainerImage
                ? {
                    image: this.workflowContainerImage,
                }
                : undefined;
            if (this.failureIssue) {
                steps.push(...[
                    {
                        name: "Extract Version",
                        if: "${{ failure() }}",
                        id: "extract-version",
                        run: 'echo "VERSION=$(cat dist/version.txt)" >> $GITHUB_OUTPUT',
                    },
                    {
                        name: "Create Issue",
                        if: "${{ failure() }}",
                        uses: "imjohnbo/issue-bot@v3",
                        with: {
                            labels: this.failureIssueLabel,
                            title: `Publishing v\${{ steps.extract-version.outputs.VERSION }} to ${opts.registryName} failed`,
                            body: "See https://github.com/${{ github.repository }}/actions/runs/${{ github.run_id }}",
                        },
                        env: {
                            GITHUB_TOKEN: "${{ secrets.GITHUB_TOKEN }}",
                        },
                    },
                ]);
                Object.assign(perms, { issues: workflows_model_1.JobPermission.WRITE });
            }
            return {
                [jobname]: {
                    tools: {
                        node: { version: this.workflowNodeVersion },
                        ...opts.publishTools,
                    },
                    name: `Publish to ${opts.registryName}`,
                    permissions: perms,
                    if: this.condition,
                    needs: [this.buildJobId, ...(opts.needs ?? [])],
                    ...(0, runner_options_1.filteredRunsOnOptions)(this.runsOn, this.runsOnGroup),
                    container,
                    steps,
                },
            };
        });
    }
    publibCommand(command) {
        return `npx -p publib@${this.publibVersion} ${command}`;
    }
    githubReleaseCommand(options, branchOptions) {
        const changelogFile = options.changelogFile;
        const releaseTagFile = options.releaseTagFile;
        // create a github release
        const releaseTag = `$(cat ${releaseTagFile})`;
        const ghReleaseCommand = [
            `gh release create ${releaseTag}`,
            "-R $GITHUB_REPOSITORY",
            `-F ${changelogFile}`,
            `-t ${releaseTag}`,
            "--target $GITHUB_REF",
        ];
        if (branchOptions.prerelease) {
            ghReleaseCommand.push("-p");
        }
        const ghRelease = ghReleaseCommand.join(" ");
        // release script that does not error when re-releasing a given version
        const idempotentRelease = [
            "errout=$(mktemp);",
            `${ghRelease} 2> $errout && true;`,
            "exitcode=$?;",
            'if [ $exitcode -ne 0 ] && ! grep -q "Release.tag_name already exists" $errout; then',
            "cat $errout;",
            "exit $exitcode;",
            "fi",
        ].join(" ");
        return idempotentRelease;
    }
}
exports.Publisher = Publisher;
_a = JSII_RTTI_SYMBOL_1;
Publisher[_a] = { fqn: "projen.release.Publisher", version: "0.91.13" };
Publisher.PUBLISH_GIT_TASK_NAME = "publish:git";
function secret(secretName) {
    return `\${{ secrets.${secretName} }}`;
}
/**
 * Options for authorizing requests to a AWS CodeArtifact npm repository.
 */
var CodeArtifactAuthProvider;
(function (CodeArtifactAuthProvider) {
    /**
     * Fixed credentials provided via Github secrets.
     */
    CodeArtifactAuthProvider["ACCESS_AND_SECRET_KEY_PAIR"] = "ACCESS_AND_SECRET_KEY_PAIR";
    /**
     * Ephemeral credentials provided via Github's OIDC integration with an IAM role.
     * See:
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_create_oidc.html
     * https://docs.github.com/en/actions/deployment/security-hardening-your-deployments/configuring-openid-connect-in-amazon-web-services
     */
    CodeArtifactAuthProvider["GITHUB_OIDC"] = "GITHUB_OIDC";
})(CodeArtifactAuthProvider || (exports.CodeArtifactAuthProvider = CodeArtifactAuthProvider = {}));
/**
 * Evaluates if the `registryUrl` is a AWS CodeArtifact registry.
 * @param registryUrl url of registry
 * @returns true for AWS CodeArtifact
 */
function isAwsCodeArtifactRegistry(registryUrl) {
    return registryUrl && AWS_CODEARTIFACT_REGISTRY_REGEX.test(registryUrl);
}
/**
 * Parses info about code artifact domain from given AWS code artifact url
 * @param url Of code artifact domain
 * @returns domain, account, and region of code artifact domain
 */
function awsCodeArtifactInfoFromUrl(url) {
    const captureRegex = /([a-z0-9-]+)-(.+)\.d\.codeartifact\.(.+)\.amazonaws\.com/;
    const matches = url?.match(captureRegex) ?? [];
    const [_, domain, account, region] = matches;
    return { domain, account, region };
}
//# sourceMappingURL=data:application/json;base64,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