"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Release = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const publisher_1 = require("./publisher");
const release_trigger_1 = require("./release-trigger");
const consts_1 = require("../build/private/consts");
const component_1 = require("../component");
const github_1 = require("../github");
const constants_1 = require("../github/constants");
const util_1 = require("../github/private/util");
const workflows_model_1 = require("../github/workflows-model");
const runner_options_1 = require("../runner-options");
const util_2 = require("../util");
const name_1 = require("../util/name");
const path_1 = require("../util/path");
const version_1 = require("../version");
const BUILD_JOBID = "release";
const GIT_REMOTE_STEPID = "git_remote";
const TAG_EXISTS_STEPID = "check_tag_exists";
const LATEST_COMMIT_OUTPUT = "latest_commit";
const TAG_EXISTS_OUTPUT = "tag_exists";
/**
 * Conditional (Github Workflow Job `if`) to check if a release job should be run.
 */
const DEPENDENT_JOB_CONDITIONAL = `needs.${BUILD_JOBID}.outputs.${TAG_EXISTS_OUTPUT} != 'true' && needs.${BUILD_JOBID}.outputs.${LATEST_COMMIT_OUTPUT} == github.sha`;
/**
 * Manages releases (currently through GitHub workflows).
 *
 * By default, no branches are released. To add branches, call `addBranch()`.
 */
class Release extends component_1.Component {
    /**
     * Returns the `Release` component of a project or `undefined` if the project
     * does not have a Release component.
     */
    static of(project) {
        const isRelease = (c) => c instanceof Release;
        return project.components.find(isRelease);
    }
    /**
     * @param scope should be part of the project the Release belongs to.
     * @param options options to configure the Release Component.
     */
    constructor(scope, options) {
        super(scope);
        this._branches = new Array();
        this.jobs = {};
        if (Array.isArray(options.releaseBranches)) {
            throw new Error('"releaseBranches" is no longer an array. See type annotations');
        }
        this.github = github_1.GitHub.of(this.project.root);
        this.buildTask = options.task;
        this.preBuildSteps = options.releaseWorkflowSetupSteps ?? [];
        this.postBuildSteps = options.postBuildSteps ?? [];
        this.artifactsDirectory =
            options.artifactsDirectory ?? consts_1.DEFAULT_ARTIFACTS_DIRECTORY;
        (0, util_1.ensureNotHiddenPath)(this.artifactsDirectory, "artifactsDirectory");
        this.versionFile = options.versionFile;
        this.releaseTrigger = options.releaseTrigger ?? release_trigger_1.ReleaseTrigger.continuous();
        this.containerImage = options.workflowContainerImage;
        this.workflowRunsOn = options.workflowRunsOn;
        this.workflowRunsOnGroup = options.workflowRunsOnGroup;
        this.workflowPermissions = {
            contents: workflows_model_1.JobPermission.WRITE,
            ...options.workflowPermissions,
        };
        this._branchHooks = [];
        /**
         * Use manual releases with no changelog if releaseEveryCommit is explicitly
         * disabled and no other trigger is set.
         *
         * TODO: Remove this when releaseEveryCommit and releaseSchedule are removed
         */
        if (!((options.releaseEveryCommit ?? true) ||
            options.releaseSchedule ||
            options.releaseTrigger)) {
            this.releaseTrigger = release_trigger_1.ReleaseTrigger.manual({ changelog: false });
        }
        if (options.releaseSchedule) {
            this.releaseTrigger = release_trigger_1.ReleaseTrigger.scheduled({
                schedule: options.releaseSchedule,
            });
        }
        this.version = new version_1.Version(this.project, {
            versionInputFile: this.versionFile,
            artifactsDirectory: this.artifactsDirectory,
            versionrcOptions: options.versionrcOptions,
            tagPrefix: options.releaseTagPrefix,
            releasableCommits: options.releasableCommits,
            bumpPackage: options.bumpPackage,
            nextVersionCommand: options.nextVersionCommand,
        });
        this.releaseTagFilePath = path.posix.normalize(path.posix.join(this.artifactsDirectory, this.version.releaseTagFileName));
        this.publisher = new publisher_1.Publisher(this.project, {
            artifactName: this.artifactsDirectory,
            condition: DEPENDENT_JOB_CONDITIONAL,
            buildJobId: BUILD_JOBID,
            jsiiReleaseVersion: options.jsiiReleaseVersion,
            failureIssue: options.releaseFailureIssue,
            failureIssueLabel: options.releaseFailureIssueLabel,
            ...(0, runner_options_1.filteredWorkflowRunsOnOptions)(options.workflowRunsOn, options.workflowRunsOnGroup),
            publishTasks: options.publishTasks,
            dryRun: options.publishDryRun,
            workflowNodeVersion: options.workflowNodeVersion,
            workflowContainerImage: options.workflowContainerImage,
        });
        const githubRelease = options.githubRelease ?? true;
        if (githubRelease) {
            this.publisher.publishToGitHubReleases({
                changelogFile: path.posix.join(this.artifactsDirectory, this.version.changelogFileName),
                versionFile: path.posix.join(this.artifactsDirectory, this.version.versionFileName),
                releaseTagFile: path.posix.join(this.artifactsDirectory, this.version.releaseTagFileName),
            });
        }
        // add the default branch (we need the internal method which does not require majorVersion)
        this.defaultBranch = this._addBranch(options.branch, {
            prerelease: options.prerelease,
            majorVersion: options.majorVersion,
            minMajorVersion: options.minMajorVersion,
            workflowName: options.releaseWorkflowName ??
                (0, name_1.workflowNameForProject)("release", this.project),
            tagPrefix: options.releaseTagPrefix,
            npmDistTag: options.npmDistTag,
        });
        for (const [name, opts] of Object.entries(options.releaseBranches ?? {})) {
            this.addBranch(name, opts);
        }
    }
    /**
     * Add a hook that should be run for every branch (including those that will
     * be added by future `addBranch` calls).
     * @internal
     */
    _forEachBranch(hook) {
        for (const branch of this._branches) {
            hook(branch.name);
        }
        this._branchHooks.push(hook);
    }
    /**
     * Adds a release branch.
     *
     * It is a git branch from which releases are published. If a project has more than one release
     * branch, we require that `majorVersion` is also specified for the primary branch in order to
     * ensure branches always release the correct version.
     *
     * @param branch The branch to monitor (e.g. `main`, `v2.x`)
     * @param options Branch definition
     */
    addBranch(branch, options) {
        this._addBranch(branch, options);
        // run all branch hooks
        for (const hook of this._branchHooks) {
            hook(branch);
        }
    }
    /**
     * Adds a release branch.
     *
     * It is a git branch from which releases are published. If a project has more than one release
     * branch, we require that `majorVersion` is also specified for the primary branch in order to
     * ensure branches always release the correct version.
     *
     * @param branch The branch to monitor (e.g. `main`, `v2.x`)
     * @param options Branch definition
     */
    _addBranch(branch, options) {
        if (this._branches.find((b) => b.name === branch)) {
            throw new Error(`The release branch ${branch} is already defined`);
        }
        // if we add a branch, we require that the default branch will also define a
        // major version.
        if (this.defaultBranch &&
            options.majorVersion &&
            this.defaultBranch.majorVersion === undefined) {
            throw new Error('you must specify "majorVersion" for the default branch when adding multiple release branches');
        }
        const releaseBranch = {
            name: branch,
            ...options,
            workflow: this.createWorkflow(branch, options),
        };
        this._branches.push(releaseBranch);
        return releaseBranch;
    }
    preSynthesize() {
        for (const branch of this._branches) {
            if (!branch.workflow) {
                continue;
            }
            branch.workflow.addJobs(this.publisher._renderJobsForBranch(branch.name, branch));
            branch.workflow.addJobs(this.jobs);
        }
    }
    /**
     * Adds jobs to all release workflows.
     * @param jobs The jobs to add (name => job)
     */
    addJobs(jobs) {
        for (const [name, job] of Object.entries(jobs)) {
            this.jobs[name] = job;
        }
    }
    /**
     * Retrieve all release branch names
     */
    get branches() {
        return this._branches.map((b) => b.name);
    }
    /**
     * @returns a workflow or `undefined` if github integration is disabled.
     */
    createWorkflow(branchName, branch) {
        const workflowName = branch.workflowName ??
            (0, name_1.workflowNameForProject)(`release-${branchName}`, this.project);
        // to avoid race conditions between two commits trying to release the same
        // version, we check if the head sha is identical to the remote sha. if
        // not, we will skip the release and just finish the build.
        const noNewCommits = `\${{ steps.${GIT_REMOTE_STEPID}.outputs.${LATEST_COMMIT_OUTPUT} == github.sha }}`;
        // The arrays are being cloned to avoid accumulating values from previous branches
        const preBuildSteps = [...this.preBuildSteps];
        const env = {
            RELEASE: "true",
            ...this.version.envForBranch({
                majorVersion: branch.majorVersion,
                minorVersion: branch.minorVersion,
                minMajorVersion: branch.minMajorVersion,
                prerelease: branch.prerelease,
                tagPrefix: branch.tagPrefix,
            }),
        };
        // the "release" task prepares a release but does not publish anything. the
        // output of the release task is: `dist`, `.version.txt`, and
        // `.changelog.md`. this is what publish tasks expect.
        // if this is the release for "main" or "master", just call it "release".
        // otherwise, "release:BRANCH"
        const releaseTaskName = branchName === "main" || branchName === "master"
            ? "release"
            : `release:${branchName}`;
        const releaseTask = this.project.addTask(releaseTaskName, {
            description: `Prepare a release from "${branchName}" branch`,
            env,
        });
        releaseTask.exec(`rm -fr ${this.artifactsDirectory}`);
        releaseTask.spawn(this.version.bumpTask);
        releaseTask.spawn(this.buildTask);
        releaseTask.spawn(this.version.unbumpTask);
        // anti-tamper check (fails if there were changes to committed files)
        // this will identify any non-committed files generated during build (e.g. test snapshots)
        releaseTask.exec(Release.ANTI_TAMPER_CMD);
        if (this.releaseTrigger.isManual) {
            const publishTask = this.publisher.publishToGit({
                changelogFile: path.posix.join(this.artifactsDirectory, this.version.changelogFileName),
                versionFile: path.posix.join(this.artifactsDirectory, this.version.versionFileName),
                releaseTagFile: path.posix.join(this.artifactsDirectory, this.version.releaseTagFileName),
                projectChangelogFile: this.releaseTrigger.changelogPath,
                gitBranch: branchName,
                gitPushCommand: this.releaseTrigger.gitPushCommand,
            });
            releaseTask.spawn(publishTask);
        }
        const postBuildSteps = [...this.postBuildSteps];
        // Read the releasetag, then check if it already exists.
        // If it does, we will cancel this release
        postBuildSteps.push(github_1.WorkflowSteps.tagExists(`$(cat ${this.releaseTagFilePath})`, {
            name: "Check if version has already been tagged",
            id: TAG_EXISTS_STEPID,
        }));
        // check if new commits were pushed to the repo while we were building.
        // if new commits have been pushed, we will cancel this release
        postBuildSteps.push({
            name: "Check for new commits",
            id: GIT_REMOTE_STEPID,
            run: [
                `echo "${LATEST_COMMIT_OUTPUT}=$(git ls-remote origin -h \${{ github.ref }} | cut -f1)" >> $GITHUB_OUTPUT`,
                "cat $GITHUB_OUTPUT",
            ].join("\n"),
        });
        const projectPathRelativeToRoot = path.relative(this.project.root.outdir, this.project.outdir);
        const normalizedProjectPathRelativeToRoot = (0, util_2.normalizePersistedPath)(projectPathRelativeToRoot);
        postBuildSteps.push({
            name: "Backup artifact permissions",
            if: noNewCommits,
            continueOnError: true,
            run: `cd ${this.artifactsDirectory} && getfacl -R . > ${constants_1.PERMISSION_BACKUP_FILE}`,
        }, github_1.WorkflowSteps.uploadArtifact({
            if: noNewCommits,
            with: {
                name: constants_1.BUILD_ARTIFACT_NAME,
                path: normalizedProjectPathRelativeToRoot.length > 0
                    ? `${normalizedProjectPathRelativeToRoot}/${this.artifactsDirectory}`
                    : this.artifactsDirectory,
            },
        }));
        if (this.github && !this.releaseTrigger.isManual) {
            // Use target (possible parent) GitHub to create the workflow
            const workflow = new github_1.GithubWorkflow(this.github, workflowName, {
                // see https://github.com/projen/projen/issues/3761
                limitConcurrency: true,
            });
            workflow.on({
                schedule: this.releaseTrigger.schedule
                    ? [{ cron: this.releaseTrigger.schedule }]
                    : undefined,
                push: this.releaseTrigger.isContinuous
                    ? { branches: [branchName], paths: this.releaseTrigger.paths }
                    : undefined,
                workflowDispatch: {}, // allow manual triggering
            });
            // Create job based on child (only?) project GitHub
            const taskjob = new github_1.TaskWorkflowJob(this, releaseTask, {
                outputs: {
                    [LATEST_COMMIT_OUTPUT]: {
                        stepId: GIT_REMOTE_STEPID,
                        outputName: LATEST_COMMIT_OUTPUT,
                    },
                    [TAG_EXISTS_OUTPUT]: {
                        stepId: TAG_EXISTS_STEPID,
                        outputName: "exists",
                    },
                },
                container: this.containerImage
                    ? { image: this.containerImage }
                    : undefined,
                env: {
                    CI: "true",
                },
                permissions: this.workflowPermissions,
                checkoutWith: {
                    // fetch-depth= indicates all history for all branches and tags
                    // we must use this in order to fetch all tags
                    // and to inspect the history to decide if we should release
                    fetchDepth: 0,
                },
                preBuildSteps,
                postBuildSteps,
                jobDefaults: normalizedProjectPathRelativeToRoot.length > 0 // is subproject
                    ? {
                        run: {
                            workingDirectory: (0, path_1.ensureRelativePathStartsWithDot)(normalizedProjectPathRelativeToRoot),
                        },
                    }
                    : undefined,
                ...(0, runner_options_1.filteredRunsOnOptions)(this.workflowRunsOn, this.workflowRunsOnGroup),
            });
            workflow.addJob(BUILD_JOBID, taskjob);
            return workflow;
        }
        else {
            return undefined;
        }
    }
}
exports.Release = Release;
_a = JSII_RTTI_SYMBOL_1;
Release[_a] = { fqn: "projen.release.Release", version: "0.91.13" };
Release.ANTI_TAMPER_CMD = "git diff --ignore-space-at-eol --exit-code";
//# sourceMappingURL=data:application/json;base64,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